#pragma once

#include <util/generic/hash.h>
#include <util/system/mutex.h>
#include <util/system/defaults.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/threading/atomic_shared_ptr/atomic_shared_ptr.h>


#include <util/generic/list.h>
#include "ktime.h"

//*************************************************************************************************************************
//                                              TCntComplShingle
//*************************************************************************************************************************

class TCntComplShingle {
private:
    ui64 m_shingle;
    ui8 m_type;
    TString m_shingle_str;

public:
    TCntComplShingle();
    TCntComplShingle(ui64 shingleA, ui8 typeA);

    const TString& Shingle_str() const {
        return m_shingle_str;
    }
    ui64 Shingle() const  {
        return m_shingle;
    }
    ui8 Type() const  {
        return m_type;
    }
};

//*************************************************************************************************************************
//                                                  TCntComplShingleRequest
//*************************************************************************************************************************

struct TUserWeights{
    TUserWeights(TInstant creationTime,
                 TInstant upTime,
                 TDuration duration,
                 bool shouldBeMalic,
                 double weight,
                 TString user) noexcept
    : CreationTime(creationTime)
    , UpTime(upTime)
    , Weight(weight)
    , Duration(duration)
    , ShouldBeMalic(shouldBeMalic)
    , User(std::move(user)) {}

    TUserWeights() = default;

    friend IOutputStream& operator<<(IOutputStream& s, const TUserWeights& userWeights) {
        const auto& CreationTime = userWeights.CreationTime;
        const auto& UpTime = userWeights.UpTime;
        const auto& Weight = userWeights.Weight;
        const auto& Duration = userWeights.Duration;
        const auto& ShouldBeMalic = userWeights.ShouldBeMalic;
        const auto& User = userWeights.User;
        return s << LabeledOutput(CreationTime, UpTime, Weight, Duration, ShouldBeMalic, User);
    }

    [[nodiscard]] TInstant LastSet() const {
        return UpTime;
    }

    [[nodiscard]] bool GetEffectiveShouldBeMalic() const {
        return (Now() < (UpTime + Duration)) && ShouldBeMalic;
    }
    [[nodiscard]] double GetEffectiveWeight() const {
        return (Now() < (UpTime + Duration)) ? Weight : 0.;
    }

    [[nodiscard]] const TString & GetUser() const {
        return User;
    }

    [[nodiscard]] TDuration GetDuration() const {
        return Duration;
    }

    [[nodiscard]] TInstant FreshUntil() const {
        return UpTime + Duration;
    }

private:
    TInstant CreationTime{}, UpTime{};
    double Weight{};
    TDuration Duration{};
    bool ShouldBeMalic{};
    TString User;
};

struct TUserWeightsPair{
    TUserWeights In, Out;
};

using TUserWeightsPairMap = THashMap<std::pair<ui64, int>, TUserWeightsPair>;
using TUserWeightsPairPtr = TTrueAtomicSharedPtr<TUserWeightsPairMap>;

class TCntComplShingleRequest {
private:
    TCntComplShingle m_shingle;
    TString m_res_shingles_source;
    bool m_is_abuse;
    bool m_is_count;
    ui32 m_uniq_sd_spam;
    ui32 m_uniq_sd_ham;
    ui32 m_ham;
    ui32 m_spam;
    ui32 m_malic;
    ui32 m_persham;
    ui32 m_persspam;

    ui32 m_today_firsttime;      //����� �������� ������
    ui32 m_today_lasttime;       //����� ���������� ���������� ������
    ui32 m_today_ham;            //���-�� ���� (� ��������)
    ui32 m_today_spam;           //���-�� ����� (� ��������)
    ui32 m_today_malic;          //���-�� ������ (� ��������)
    ui32 m_today_pers_ham;       //���-�� ������������� ���� (� ��������)
    ui32 m_today_pers_spam;      //���-�� ������������� ����� (� ��������)
    ui32 m_today_complaint_ham;  //���-�� ����� �� ������
    ui32 m_today_complaint_spam; //���-�� ����� �� ���� (unsubscribe ���� � ����)
    ui32 m_today_virus_count;    //���-�� ����� � ��������
    ui32 m_today_uniq_sd_spam;   //���������� ����� �� ���� (�� suid)
    ui32 m_today_uniq_sd_ham;    //���������� ����� �� ��� (�� suid)
    ui32 m_today_expert_spam;    //���-�� ����� ��������� �� ����
    ui32 m_today_expert_ham;     //���-�� ����� ��������� �� ���

    kday_t m_history_firsttime;              //����� �������� ������
    kday_t m_history_lasttime;               //����� ���������� ���������� ������
    ui32 m_history_ham;                      //���-�� ���� (� ��������)
    ui32 m_history_spam;                     //���-�� ����� (� ��������)
    ui32 m_history_malic;                    //���-�� ������ (� ��������)
    ui32 m_history_pers_ham;                 //���-�� ������������� ���� (� ��������)
    ui32 m_history_pers_spam;                //���-�� ������������� ����� (� ��������)
    ui32 m_history_complaint_ham;            //���-�� ����� �� ������
    ui32 m_history_complaint_spam;           //���-�� ����� �� ���� (unsubscribe ���� � ����)
    ui32 m_history_virus_count;              //���-�� ����� � ��������
    ui32 m_history_uniq_sd_spam;             //���������� ����� �� ���� (�� suid)
    ui32 m_history_uniq_sd_ham;              //���������� ����� �� ��� (�� suid)
    ui32 m_history_expert_spam;              //���-�� ����� ��������� �� ����
    ui32 m_history_expert_ham;               //���-�� ����� ��������� �� ���
    ui32 m_history_day_count_with_complaint; //���-�� ����, ����� ��������� ������
    ui32 m_history_day_count_with_virus;     //���-�� ����, ����� ��������� ������
public:
    bool IsUserWeightOnly() const;

private:

    void ResClear();


public:
    void SetShingleSource(const TString& source);
    void SetTodayCounters(ui32 tv1, ui32 tv2, ui32 tv3, ui32 tv4, ui32 tv5, ui32 tv6, ui32 tv7, ui32 tv8, ui32 tv9, ui32 tv10, ui32 tv11, ui32 tv12, ui32 tv13, ui32 tv14);
    void SetHistoryCounters(ui32 hv1, ui32 hv2, ui32 hv3, ui32 hv4, ui32 hv5, ui32 hv6, ui32 hv7, ui32 hv8, ui32 hv9, ui32 hv10, ui32 hv11, ui32 hv12, ui32 hv13, ui32 hv14, ui32 hv15, ui32 hv16);

    bool UseAbuseConstructor() {
        return m_is_abuse;
    }
    bool UseCountConstructor() {
        return m_is_count;
    }
    TString GetRequestGetCluster();
    TString GetRequestAddComplaint();
    TString GetRequestAddVirus();
    TString GetRequestAddCount();

    ui32 GetUniqSdSpam() const { return m_uniq_sd_spam; }
    ui32 GetUniqSdHam() const { return m_uniq_sd_ham; }

public:
    TCntComplShingleRequest();

    //����������� ��� ��������� ������ �� ������ (cs_get?...)
    TCntComplShingleRequest(ui64 shingleA, ui8 typeA);

    //����������� ��� ������������ ������ � ������ (cs_abuse?...)
    TCntComplShingleRequest(ui64 shingleA, ui8 typeA, const TString& shingle_source, ui32 uniq_sd_spam, ui32 uniq_sd_ham);

    //����������� ��� ������������ ������ � �������� ��������� (cs_count?...)
    TCntComplShingleRequest(ui64 shingleA, ui8 typeA, ui32 ham, ui32 spam, ui32 malic, ui32 persham, ui32 persspam);

    ~TCntComplShingleRequest();

    const char* sShingle() const;
    ui64 Shingle() const {
        return m_shingle.Shingle();
    } //shingle
    ui8 Type() const {
        return m_shingle.Type();
    } //���
    TString GetShingleSource() {
        return m_res_shingles_source;
    } //shingle source

    bool IsValid() const;
    bool ResTodayEmpty() const;
    bool ResHistoryEmpty() const;

    ui32 GetTodayFirsttime() {
        return m_today_firsttime;
    }
    ui32 GetTodayLasttime() {
        return m_today_lasttime;
    }
    ui32 GetTodayFirsttimeElapsed(); //������� ������ ������ ������������ �������� �������, ����� ���������� ��������� � ����
    ui32 GetTodayLasttimeElapsed();  //������� ������ ������ ������������ �������� �������, ����� ������ � ���� � ��������� ��� ����������
    ui32 GetTodayHam() {
        return m_today_ham;
    }
    ui32 GetTodaySpam() {
        return m_today_spam;
    }
    ui32 GetTodayMalic() {
        return m_today_malic;
    }
    ui32 GetTodayPersHam() {
        return m_today_pers_ham;
    }
    ui32 GetTodayPersSpam() {
        return m_today_pers_spam;
    }
    ui32 GetTodayComplaintHam() const {
        return m_today_complaint_ham;
    }
    ui32 GetTodayComplaintSpam() const {
        return m_today_complaint_spam;
    }
    ui32 GetTodayVirusCount() const {
        return m_today_virus_count;
    }
    ui32 GetTodayUniqSDSpam() const {
        return m_today_uniq_sd_spam;
    }
    ui32 GetTodayUniqSDHam() const {
        return m_today_uniq_sd_ham;
    }
    ui32 GetTodayExpertSpam() const {
        return m_today_expert_spam;
    }
    ui32 GetTodayExpertHam() const {
        return m_today_expert_ham;
    }

    kday_t GetHistoryFirsttime() const {
        return m_history_firsttime;
    }
    kday_t GetHistoryLasttime() const {
        return m_history_lasttime;
    }
    ui32 GetHistoryFirsttimeElapsed() const; //������� ���� ������ ������������ �������� �������, ����� ���������� ��������� � ����
    ui32 GetHistoryLasttimeElapsed() const;  //������� ���� ������ ������������ �������� �������, ����� ������ � ���� � ��������� ��� ����������
    ui32 GetHistoryHam() const {
        return m_history_ham;
    }
    ui32 GetHistorySpam() const {
        return m_history_spam;
    }
    ui32 GetHistoryMalic() const {
        return m_history_malic;
    }
    ui32 GetHistoryPersHam() const {
        return m_history_pers_ham;
    }
    ui32 GetHistoryPersSpam() const {
        return m_history_pers_spam;
    }
    ui32 GetHistoryComplaintHam() const {
        return m_history_complaint_ham;
    }
    ui32 GetHistoryComplaintSpam() const {
        return m_history_complaint_spam;
    }
    ui32 GetHistoryVirusCount() const {
        return m_history_virus_count;
    }
    ui32 GetHistoryUniqSDSpam() const {
        return m_history_uniq_sd_spam;
    }
    ui32 GetHistoryUniqSDHam() const {
        return m_history_uniq_sd_ham;
    }
    ui32 GetHistoryExpertSpam() const {
        return m_history_expert_spam;
    }
    ui32 GetHistoryExpertHam() const {
        return m_history_expert_ham;
    }
    ui32 GetHistoryDayCountWithComplaint() const {
        return m_history_day_count_with_complaint;
    }
    ui32 GetHistoryDayCountWithVirus() const {
        return m_history_day_count_with_virus;
    }
};

typedef TList<TCntComplShingleRequest> TCntComplShingleRequestList;
