#include "tshinglerenv.h"

#include <util/string/printf.h>

#include <math.h>

TString TShingleSpamTypeToStroka(TShingleSpamType spamtype) {
    TString res = "";

    switch (spamtype) {
        case HAM:
            res = "HAM";
            break;
        case MALIC:
            res = "DLV";
            break;
        case SPAM:
            res = "SPAM";
            break;
        case UNKNOWN:
            res = "UNKNOWN";
            break;
    };

    return res;
}

TString TActionTypeToStroka(TActionType act) {
    TString res = "";

    switch (act) {
        case AUNDEF:
            res = "AUNDEF";
            break;
        case AEXISTS:
            res = "AEXISTS";
            break;
        case ABUSY:
            res = "ABUSY";
            break;
        case AEXCEEDTYPE:
            res = "AEXCEEDTYPE";
            break;
        case ABADTYPEHASH:
            res = "ABADTYPEHASH";
            break;
        case AEXCEEDPART:
            res = "AEXCEEDPART";
            break;
        case ABADPARTHASH:
            res = "ABADPARTHASH";
            break;
        case ANOTEXISTS:
            res = "ANOTEXISTS";
            break;
    };

    return res;
}

TString TPersTypeToStroka(TPersType pers) {
    TString res = "";

    switch (pers) {
        case PERSUNDEF:
            res = "PERSUNDEF";
            break;
        case PERSHAM:
            res = "PERSHAM";
            break;
        case PERSSPAM:
            res = "PERSSPAM";
            break;
    };

    return res;
}

//********************************************************************************************************************************
//                                           TShingleDataSingle - хранится в памяти heavy
//********************************************************************************************************************************

TShingleDataSingle::TShingleDataSingle() {
    Clear();
}

TShingleDataSingle::TShingleDataSingle(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam, float weight) {
    Clear();
    Set(ham, malic, spam, persham, persspam, weight);
}

TShingleDataSingle::TShingleDataSingle(TShingleSpamType spamtype, ui16 count, TPersType pers) {
    Clear();
    Add(spamtype, count, pers);
}

TShingleDataSingle::TShingleDataSingle(float weight) {
    Clear();
    SetWeight(weight);
}

TShingleDataSingle::~TShingleDataSingle() {
}

void TShingleDataSingle::Clear() {
    m_ham = 0;
    m_spam = 0;
    m_malic = 0;
    m_pers_ham = 0;
    m_pers_spam = 0;
    m_weight = 0;
    m_ham_diff = 0;
    m_spam_diff = 0;
    m_malic_diff = 0;
    m_persham_diff = 0;
    m_persspam_diff = 0;
    m_pr = 0;
}

void TShingleDataSingle::ClearDiffs() {
    m_ham_diff = 0;
    m_spam_diff = 0;
    m_malic_diff = 0;
    m_persham_diff = 0;
    m_persspam_diff = 0;
}

void TShingleDataSingle::Add(TShingleSpamType spamtype, ui16 count, TPersType pers) {
    switch (spamtype) {
        default:
            break;
        case HAM:
            m_ham = IncMax16(m_ham, count);
            m_ham_diff = IncMax8(m_ham_diff, count);
            break;
        case MALIC:
            m_malic = IncMax16(m_malic, count);
            m_malic_diff = IncMax8(m_malic_diff, count);
            break;
        case SPAM:
            m_spam = IncMax16(m_spam, count);
            m_spam_diff = IncMax8(m_spam_diff, count);
            break;
    };

    switch (pers) {
        default:
            break;
        case PERSHAM:
            m_pers_ham = IncMax16(m_pers_ham, count);
            m_persham_diff = IncMax8(m_persham_diff, count);
            break;
        case PERSSPAM:
            m_pers_spam = IncMax16(m_pers_spam, count);
            m_persspam_diff = IncMax8(m_persspam_diff, count);
            break;
    };
}

void TShingleDataSingle::Add(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam) {
    if (ham > 0) {
        m_ham = IncMax16(m_ham, ham);
        m_ham_diff = IncMax8(m_ham_diff, ham);
    }
    if (malic > 0) {
        m_malic = IncMax16(m_malic, malic);
        m_malic_diff = IncMax8(m_malic_diff, malic);
    }
    if (spam > 0) {
        m_spam = IncMax16(m_spam, spam);
        m_spam_diff = IncMax8(m_spam_diff, spam);
    }
    if (persham > 0) {
        m_pers_ham = IncMax16(m_pers_ham, persham);
        m_persham_diff = IncMax8(m_persham_diff, persham);
    }
    if (persspam > 0) {
        m_pers_spam = IncMax16(m_pers_spam, persspam);
        m_persspam_diff = IncMax8(m_persspam_diff, persspam);
    }
}

void TShingleDataSingle::AddWODiff(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam) {
    if (ham > 0) {
        m_ham = IncMax16(m_ham, ham);
    }
    if (malic > 0) {
        m_malic = IncMax16(m_malic, malic);
    }
    if (spam > 0) {
        m_spam = IncMax16(m_spam, spam);
    }
    if (persham > 0) {
        m_pers_ham = IncMax16(m_pers_ham, persham);
    }
    if (persspam > 0) {
        m_pers_spam = IncMax16(m_pers_spam, persspam);
    }
}

void TShingleDataSingle::Set(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam, float weight) {
    m_ham = ham;
    m_malic = malic;
    m_spam = spam;
    m_pers_ham = persham;
    m_pers_spam = persspam;
    m_weight = weight;
    m_ham_diff = 0;
    m_spam_diff = 0;
    m_malic_diff = 0;
    m_persham_diff = 0;
    m_persspam_diff = 0;
}

void TShingleDataSingle::SetWeight(float weight) {
    m_weight = weight;
}

TString TShingleDataSingle::GetFollowText() {
    TString ret;
    ret = Sprintf("(h,m,s,ph,ps,w,hdf,mdf,sdf,phdf,psdf)='%d,%d,%d,%d,%d,%f,%d,%d,%d,%d,%d'", m_ham, m_malic, m_spam, m_pers_ham, m_pers_spam, m_weight, m_ham_diff, m_malic_diff, m_spam_diff, m_persham_diff, m_persspam_diff);
    return ret;
}

//********************************************************************************************************************************
//                                      TShingleData - хранится в памяти (совмещенная today/yesterday)
//********************************************************************************************************************************

TShingleDataTE::TShingleDataTE() {
    Clear();
}

TShingleDataTE::TShingleDataTE(ui16 t_ham, ui16 t_malic, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_malic, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight) {
    Clear();
    Set(t_ham, t_malic, t_spam, t_persham, t_persspam, t_weight, y_ham, y_malic, y_spam, y_persham, y_persspam, y_weight);
}

TShingleDataTE::TShingleDataTE(TShingleDataSingle* t_data, TShingleDataSingle* y_data) {
    Clear();
    if (t_data != nullptr) {
        m_ham[TODAY] = t_data->Ham();
        m_spam[TODAY] = t_data->Spam();
        m_malic[TODAY] = t_data->Malic();
        m_pers_ham[TODAY] = t_data->PersHam();
        m_pers_spam[TODAY] = t_data->PersSpam();
        m_weight[TODAY] = t_data->Weight();
        m_ham_diff = t_data->DiffHam();
        m_spam_diff = t_data->DiffSpam();
        m_malic_diff = t_data->DiffMalic();
        m_persham_diff = t_data->DiffPersHam();
        m_persspam_diff = t_data->DiffPersSpam();
    }
    if (y_data != nullptr) {
        m_ham[YESTERDAY] = y_data->Ham();
        m_spam[YESTERDAY] = y_data->Spam();
        m_malic[YESTERDAY] = y_data->Malic();
        m_pers_ham[YESTERDAY] = y_data->PersHam();
        m_pers_spam[YESTERDAY] = y_data->PersSpam();
        m_weight[YESTERDAY] = y_data->Weight();
    }
}

TShingleDataTE::~TShingleDataTE() {
}

void TShingleDataTE::Clear() {
    m_ham[TODAY] = 0;
    m_spam[TODAY] = 0;
    m_malic[TODAY] = 0;
    m_pers_ham[TODAY] = 0;
    m_pers_spam[TODAY] = 0;
    m_weight[TODAY] = 0;

    m_ham[YESTERDAY] = 0;
    m_spam[YESTERDAY] = 0;
    m_malic[YESTERDAY] = 0;
    m_weight[YESTERDAY] = 0;
    m_pers_ham[YESTERDAY] = 0;
    m_pers_spam[YESTERDAY] = 0;

    m_ham_diff = 0;
    m_spam_diff = 0;
    m_malic_diff = 0;
    m_persham_diff = 0;
    m_persspam_diff = 0;
}

void TShingleDataTE::Add(TShingleSpamType spamtype, ui16 count, TPersType pers) {
    switch (spamtype) {
        default:
            break;
        case HAM:
            m_ham[TODAY] = IncMax16(m_ham[TODAY], count);
            m_ham_diff = IncMax16(m_ham_diff, count);
            break;
        case MALIC:
            m_malic[TODAY] = IncMax16(m_malic[TODAY], count);
            m_malic_diff = IncMax16(m_malic_diff, count);
            break;
        case SPAM:
            m_spam[TODAY] = IncMax16(m_spam[TODAY], count);
            m_spam_diff = IncMax16(m_spam_diff, count);
            break;
    };

    switch (pers) {
        default:
            break;
        case PERSHAM:
            m_pers_ham[TODAY] = IncMax16(m_pers_ham[TODAY], count);
            m_persham_diff = IncMax16(m_persham_diff, count);
            break;
        case PERSSPAM:
            m_pers_spam[TODAY] = IncMax16(m_pers_spam[TODAY], count);
            m_persspam_diff = IncMax16(m_persspam_diff, count);
            break;
    };
}

void TShingleDataTE::Set(ui16 t_ham, ui16 t_malic, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_malic, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight) {
    m_ham[TODAY] = t_ham;
    m_malic[TODAY] = t_malic;
    m_spam[TODAY] = t_spam;
    m_pers_ham[TODAY] = t_persham;
    m_pers_spam[TODAY] = t_persspam;
    m_weight[TODAY] = t_weight;
    m_ham[YESTERDAY] = y_ham;
    m_malic[YESTERDAY] = y_malic;
    m_spam[YESTERDAY] = y_spam;
    m_pers_ham[YESTERDAY] = y_persham;
    m_pers_spam[YESTERDAY] = y_persspam;
    m_weight[YESTERDAY] = y_weight;
    m_ham_diff = 0;
    m_spam_diff = 0;
    m_malic_diff = 0;
    m_persham_diff = 0;
    m_persspam_diff = 0;
}

//********************************************************************************************************************************
//                                        TObrabShingleDebugInfo - отладочные данные по обработке шингла
//********************************************************************************************************************************

TString TSOSERVDBG_LOCKTYPEToStroka(TSOSERVDBG_LOCKTYPE type) {
    TString res = "";

    switch (type) {
        case TD_NONE:
            res = "NONE";
            break;
        case TD_GET:
            res = "GET";
            break;
        case TD_PUTA:
            res = "PUTA";
            break;
        case TD_PUTMA:
            res = "PUTMA";
            break;
        case TD_SETMA:
            res = "SETMA";
            break;
        case TD_SETWEIGHTA:
            res = "SETWEIGHTA";
            break;
        case TD_CLEARSHINGLEA:
            res = "CLEARSHINGLEA";
            break;
        case TD_WRITEDUMPA:
            res = "WRITEDUMPA";
            break;
        case TD_CLEANUP:
            res = "CLEANUP";
            break;
        case TD_MIDNIGHT:
            res = "MIDNIGHT";
            break;
        case TD_SETLASTMIDNIGHTTIME:
            res = "SETLASTMIDNIGHTTIME";
            break;
        case TD_SETLASTCLEANUPTIME:
            res = "SETLASTCLEANUPTIME";
            break;
    };

    return res;
};

TObrabShingleDebugInfo::TObrabShingleDebugInfo() {
    m_shingle = 0;
    m_type = 0;
    m_att_shingle = 0;
    m_att_type = 0;

    Clear();
}

TObrabShingleDebugInfo::~TObrabShingleDebugInfo() {
}

void TObrabShingleDebugInfo::Init(ui64 shingle, int type) {
    Init(shingle, type, 0, 0);
}

void TObrabShingleDebugInfo::Init(ui64 shingle, int type, ui64 att_shingle, int att_type) {
    m_shingle = shingle;
    m_type = type;
    m_att_shingle = att_shingle;
    m_att_type = att_type;
}

void TObrabShingleDebugInfo::Clear() {
    data.Clear();
    att_data.Clear();
}

void TObrabShingleDebugInfo::AddToAttdata(TObrabShingleDebugInfo& value) {
    att_data = value.data;
}

TString TObrabShingleDebugInfo::GetShToLog() {
    TString res = "";

    if (Shingle() != 0)
        res = res + "SH:" + ShingleToStroka(Shingle()) + "-" + IntToStroka3(Type()) + "[" + data.toLog(false) + "]=";
    if (Shingle_uniq() != 0)
        res = res + ShingleToStroka(Shingle_uniq()) + "-" + IntToStroka3(Type_uniq()) + "[" + att_data.toLog(false) + "]";
    res = res + ";";

    return res;
}

TString TObrabShingleDebugInfo::GetStToLog() {
    TString res = "";

    if (Shingle() != 0)
        res = res + "ST:" + ShingleToStroka(Shingle()) + "[" + data.toLog(true) + "]";
    res = res + ";";

    return res;
}

//********************************************************************************************************************************
//                                             TShingleStat
//********************************************************************************************************************************

TShingleStat::TShingleStat() {
    m_shform = TSHUNKNOWN;
    m_shingle = 0;
    m_type = 0;
    m_uniq_shingle = 0;
    m_uniq_shingle_type = 0;
    m_count = 0;
    Clear();
}

TShingleStat::TShingleStat(ui64 shingle, ui16 type, int count) {
    SetShingleType(TSHSHINGLE, shingle, type, 0, 0);
    m_count = count;
    Clear();
}

TShingleStat::TShingleStat(ui64 shingle, ui16 type, ui64 uniq_shingle, int uniq_shingle_type, int count) {
    SetShingleType(TSHSHINGLE, shingle, type, uniq_shingle, uniq_shingle_type);
    m_count = count;
    Clear();
}

TShingleStat::TShingleStat(ui64 shingle) {
    SetShingleType(TSHSTORAGE, shingle, 0xFFFF, 0, 0);
    m_count = 0;
    Clear();
}

TShingleStat::~TShingleStat() {
}

void TShingleStat::SetShingleType(TShingleForm shform, ui64 shingle, ui16 type, ui64 uniq_shingle, int uniq_shingle_type) {
    m_shform = shform;
    m_shingle = shingle;
    m_type = type;
    m_uniq_shingle = uniq_shingle;
    m_uniq_shingle_type = uniq_shingle_type;

#ifdef SOSERV_DEBUG
    debuginfo.Init(shingle, type, uniq_shingle, uniq_shingle_type);
#endif
}

void TShingleStat::SetShingleData(TShingleDataTE shingle_data, bool heavy, ui32 today_pr) {
    m_empty = false;
    m_shingle_data = shingle_data;
    m_heavy = heavy;
    m_today_pr = today_pr;
}

void TShingleStat::SetAction(TActionType existsA) {
    m_exists = existsA;
}

void TShingleStat::SetAgeKoef(float age_koef) {
    m_age_koef = age_koef;
}

void TShingleStat::SetLastMidnight(ui32 lastmidnight) {
    m_last_midnight = lastmidnight;
}

void TShingleStat::SetEmptyStatus(bool empty) {
    m_empty = empty;
}

void TShingleStat::SetStorageData(const TString& stdata) {
    m_empty = false;
    m_st_data = stdata;
}

float TShingleStat::GetAgeCoeff(time_t last_midnight, ui32 koef_div) {
float res = 0;
ui32 t_diff = 0;
ui32 current_time = time(nullptr);

if (current_time >= last_midnight) {
t_diff = current_time - last_midnight;
if (t_diff <= koef_div)
res = ((float)1 - ((float)t_diff / (float)koef_div));
}

return res;
}

void TShingleStat::ClearShingledata() {
    m_empty = true;
    m_shingle_data.Clear();
    m_heavy = false;
    m_exists = AUNDEF;
    m_exist_today = false;
    m_exist_yesterday = false;
    m_today_pr = 0;
}

void TShingleStat::ClearUniqShingledata() {
    m_uniq_empty = true;
    m_uniq_shingle_data.Clear();
    m_uniq_heavy = false;
    m_uniq_exists = AUNDEF;
    m_uniq_exist_today = false;
    m_uniq_exist_yesterday = false;
}

void TShingleStat::Clear() {
    ClearShingledata();
    ClearUniqShingledata();
    m_age_koef = 1;
    m_last_midnight = 0;
    m_koef_div = 0;
    m_st_data = "";
}

bool TShingleStat::CalcHeavyStatus(ui32 heavytreshold) {
    bool res = false;
    ui32 tvalue = 0;

    if (TotalWeight() != 0)
        res = true;
    else {
        tvalue = TotalHam() + TotalSpam() + TotalMalic();
        if (tvalue > heavytreshold)
            res = true;
    }

    return res;
}

bool TShingleStat::IsFirstRecord() {
    bool res = true;

    if (m_exist_today /* || m_exist_yesterday*/)
        res = false;

    return res;
}

TString TShingleStat::UniqShinglePeriod() {
    TString exist_today_s = "exist_today";
    TString exist_yesterday_s = " + exist_yesterday";

    return exist_today_s /* + exist_yesterday_s*/;
}

TString TShingleStat::GetActionStateS() const {
    TString res = "";

    switch (m_exists) {
        case AEXISTS:
            res = "exists";
            break;
        case ABUSY:
            res = "busy";
            break;
        case AEXCEEDTYPE:
            res = "exceedtype";
            break;
        case ABADTYPEHASH:
            res = "badtypehash";
            break;
        case AEXCEEDPART:
            res = "exceedpart";
            break;
        case ABADPARTHASH:
            res = "badparthash";
            break;
        case ANOTEXISTS:
            res = "noexists";
            break;
        default:
            res = "unknown";
    };

    return res;
}

TString TShingleStat::GetActionStateShortS() const {
    TString res = "";

    switch (m_exists) {
        case AEXISTS:
            res = "";
            break;
        case ABUSY:
            res = "busy";
            break;
        case AEXCEEDTYPE:
            res = "exceedtype";
            break;
        case ABADTYPEHASH:
            res = "badtypehash";
            break;
        case AEXCEEDPART:
            res = "exceedpart";
            break;
        case ABADPARTHASH:
            res = "badparthash";
            break;
        case ANOTEXISTS:
            res = "noexists";
            break;
        default:
            res = "unknown";
    };

    return res;
}

TString TShingleStat::GetActionStateS_uniq() const {
    TString res = "";

    switch (m_uniq_exists) {
        case AEXISTS:
            res = "exists";
            break;
        case ABUSY:
            res = "busy";
            break;
        case AEXCEEDTYPE:
            res = "exceedtype";
            break;
        case ABADTYPEHASH:
            res = "badtypehash";
            break;
        case AEXCEEDPART:
            res = "exceedpart";
            break;
        case ABADPARTHASH:
            res = "badparthash";
            break;
        case ANOTEXISTS:
            res = "noexists";
            break;
        default:
            res = "unknown";
    };

    return res;
}

TString TShingleStat::GetActionStateShortS_uniq() const {
    TString res = "";

    switch (m_uniq_exists) {
        case AEXISTS:
            res = "";
            break;
        case ABUSY:
            res = "busy";
            break;
        case AEXCEEDTYPE:
            res = "exceedtype";
            break;
        case ABADTYPEHASH:
            res = "badtypehash";
            break;
        case AEXCEEDPART:
            res = "exceedpart";
            break;
        case ABADPARTHASH:
            res = "badparthash";
            break;
        case ANOTEXISTS:
            res = "noexists";
            break;
        default:
            res = "unknown";
    };

    return res;
}

void TShingleStat::CalcAgeKoef(time_t last_midnight, ui32 koef_div) {
m_age_koef = GetAgeCoeff(last_midnight, koef_div);
m_koef_div = koef_div;
m_last_midnight = last_midnight;
}

float TShingleStat::Round(float Argument, int Precision) const {
    float div = 1.0;
    if (Precision >= 0)
        while (Precision--)
            div *= 10.0;
    else
        while (Precision++)
            div /= 10.0;
    return floor(Argument * div + 0.5) / div;
}

ui16 TShingleStat::TotalHam() const {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Ham(TODAY) + (m_age_koef * (float)m_shingle_data.Ham(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalSpam() const {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Spam(TODAY) + (m_age_koef * (float)m_shingle_data.Spam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalMalic() const {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Malic(TODAY) + (m_age_koef * (float)m_shingle_data.Malic(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersHam() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.PersHam(TODAY) + (m_age_koef * (float)m_shingle_data.PersHam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersSpam() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.PersSpam(TODAY) + (m_age_koef * (float)m_shingle_data.PersSpam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalHam_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Ham(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_shingle_data.Ham(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalSpam_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Spam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_shingle_data.Spam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalMalic_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.Malic(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_shingle_data.Malic(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersHam_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.PersHam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_shingle_data.PersHam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersSpam_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_shingle_data.PersSpam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_shingle_data.PersSpam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

float TShingleStat::TotalWeight() {
    float res = 0;

    res = (m_shingle_data.Weight(TODAY) != 0) ? m_shingle_data.Weight(TODAY) : m_shingle_data.Weight(YESTERDAY);

    return res;
}

ui16 TShingleStat::TotalHam_uniq() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Ham(TODAY) + (m_age_koef * (float)m_uniq_shingle_data.Ham(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalSpam_uniq() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Spam(TODAY) + (m_age_koef * (float)m_uniq_shingle_data.Spam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalMalic_uniq() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Malic(TODAY) + (m_age_koef * (float)m_uniq_shingle_data.Malic(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersHam_uniq() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.PersHam(TODAY) + (m_age_koef * (float)m_uniq_shingle_data.PersHam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersSpam_uniq() {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.PersSpam(TODAY) + (m_age_koef * (float)m_uniq_shingle_data.PersSpam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalHam_uniq_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Ham(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_uniq_shingle_data.Ham(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalSpam_uniq_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Spam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_uniq_shingle_data.Spam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalMalic_uniq_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.Malic(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_uniq_shingle_data.Malic(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersHam_uniq_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.PersHam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_uniq_shingle_data.PersHam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

ui16 TShingleStat::TotalPersSpam_uniq_clt(ui32 koef_div) {
    ui16 res = 0;
    float t_value = 0;

    t_value = (float)m_uniq_shingle_data.PersSpam(TODAY) + (GetAgeCoeff(m_last_midnight, koef_div) * (float)m_uniq_shingle_data.PersSpam(YESTERDAY));
    res = (ui16)(t_value < 0xFFFF ? Round(t_value, 1) : 0xFFFF);

    return res;
}

float TShingleStat::TotalWeight_uniq() {
    float res = 0;

    res = (m_uniq_shingle_data.Weight(TODAY) != 0) ? m_uniq_shingle_data.Weight(TODAY) : m_uniq_shingle_data.Weight(YESTERDAY);

    return res;
}

/*TString TShingleStat::GetSummaryAnswer()
{
   TString res = "";

   if (m_shform == TSHSHINGLE)
   {
      char   buff[128];
      snprintf(buff, sizeof(buff) - 1, "<SH: %016" PRIx64 "-%d empty='%s' si='%u,%u,%u,%u,%u,%f'>", m_shingle, m_type, BoolToStroka(m_empty).c_str(), TotalHam(), TotalDlv(), TotalSpam(), TotalWeight());
      res = TString(buff);

   } else if (m_shform == TSHSTORAGE)
   {
      res = "<ST: " + ShingleToStroka(m_shingle) + " empty='" + BoolToStroka(m_empty) + "' std='" + m_st_data + "'>";

   }

   return res;
}*/

TString TShingleStat::GetFollowInfo() {
    TString res = "";

    char buff[2048];
    TString pr_str = "";
    TString tpr_s = "";

    //tpr_s = ui8ToBinary(m_today_pr);
    tpr_s = IntToStroka(m_today_pr);

    if (m_heavy) {
        if (!pr_str.empty())
            pr_str = pr_str + ", ";
        pr_str = pr_str + "heavy";

    } else {
        if (!pr_str.empty())
            pr_str = pr_str + ", ";
        pr_str = pr_str + "o";
    }

    switch (m_exists) {
        case AEXISTS:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "exists";
            break;
        case ABUSY:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "busy";
            break;
        case AEXCEEDTYPE:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "exceedtype";
            break;
        case ABADTYPEHASH:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "badtypehash";
            break;
        case AEXCEEDPART:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "exceedpart";
            break;
        case ABADPARTHASH:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "badparthash";
            break;
        case ANOTEXISTS:
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "noexists";
            break;
        default: //AUNDEF
            if (!pr_str.empty())
                pr_str = pr_str + ", ";
            pr_str = pr_str + "undef";
    };

    if (m_empty) {
        snprintf(buff, sizeof(buff) - 1, "empty='%s' pr='%s' tpr='%s'", BoolToStroka(m_empty).c_str(), pr_str.c_str(), tpr_s.c_str());

    } else {
        snprintf(buff, sizeof(buff) - 1, "empty='%s' pr='%s' tpr='%s' t(h,m,s,ph,ps,w)='%u,%u,%u,%u,%u,%f' y(h,m,s,ph,ps,w)='%u,%u,%u,%u,%u,%f' ak='%f'", BoolToStroka(m_empty).c_str(), pr_str.c_str(), tpr_s.c_str(),
                 m_shingle_data.Ham(TODAY), m_shingle_data.Malic(TODAY), m_shingle_data.Spam(TODAY), m_shingle_data.PersHam(TODAY), m_shingle_data.PersSpam(TODAY), m_shingle_data.Weight(TODAY),
                 m_shingle_data.Ham(YESTERDAY), m_shingle_data.Malic(YESTERDAY), m_shingle_data.Spam(YESTERDAY), m_shingle_data.PersHam(YESTERDAY), m_shingle_data.PersSpam(YESTERDAY), m_shingle_data.Weight(YESTERDAY),
                 m_age_koef);
    }

    res = TString(buff);

    return res;
}

TString TShingleStat::GetFullAnswer() {
    TString res = "";

    if (m_shform == TSHSHINGLE) {
        char buff[1024];
        TString pr_str = "";
        TString pr_str_uniq = "";

        if (m_heavy)
            pr_str = pr_str + "h";
        else
            pr_str = pr_str + "o";

        switch (m_exists) {
            case AEXISTS:
                pr_str = pr_str + "e";
                break;
            case ABUSY:
                pr_str = pr_str + "b";
                break;
            case AEXCEEDTYPE:
                pr_str = pr_str + "t";
                break;
            case ABADTYPEHASH:
                pr_str = pr_str + "s";
                break;
            case AEXCEEDPART:
                pr_str = pr_str + "p";
                break;
            case ABADPARTHASH:
                pr_str = pr_str + "q";
                break;
            case ANOTEXISTS:
                pr_str = pr_str + "n";
                break;
            default: //AUNDEF
                pr_str = pr_str + "u";
        };

        if (m_uniq_shingle == 0) {
            snprintf(buff, sizeof(buff) - 1, "<SH: %016" PRIx64 "-%d empty='%s' ti='%u,%u,%u,%u,%u,%f' yi='%u,%u,%u,%u,%u,%f' ak='%f' pr='%s' lmt='%u' >",
                    m_shingle, m_type, BoolToStroka(m_empty).c_str(),
                    m_shingle_data.Ham(TODAY), m_shingle_data.Malic(TODAY), m_shingle_data.Spam(TODAY), m_shingle_data.PersHam(TODAY), m_shingle_data.PersSpam(TODAY), m_shingle_data.Weight(TODAY),
                    m_shingle_data.Ham(YESTERDAY), m_shingle_data.Malic(YESTERDAY), m_shingle_data.Spam(YESTERDAY), m_shingle_data.PersHam(YESTERDAY), m_shingle_data.PersSpam(YESTERDAY), m_shingle_data.Weight(YESTERDAY),
                    m_age_koef, pr_str.c_str(), m_last_midnight);

        } else {
            if (m_uniq_heavy)
                pr_str_uniq = pr_str_uniq + "h";
            else
                pr_str_uniq = pr_str_uniq + "o";

            switch (m_uniq_exists) {
                case AEXISTS:
                    pr_str_uniq = pr_str_uniq + "e";
                    break;
                case ABUSY:
                    pr_str_uniq = pr_str_uniq + "b";
                    break;
                case AEXCEEDTYPE:
                    pr_str_uniq = pr_str_uniq + "t";
                    break;
                case ABADTYPEHASH:
                    pr_str_uniq = pr_str_uniq + "s";
                    break;
                case AEXCEEDPART:
                    pr_str_uniq = pr_str_uniq + "p";
                    break;
                case ABADPARTHASH:
                    pr_str_uniq = pr_str_uniq + "q";
                    break;
                case ANOTEXISTS:
                    pr_str_uniq = pr_str_uniq + "n";
                    break;
                default: //AUNDEF
                    pr_str_uniq = pr_str_uniq + "u";
            };

            snprintf(buff, sizeof(buff) - 1, "<SH: %016" PRIx64 "-%d empty='%s' ti='%u,%u,%u,%u,%u,%f' yi='%u,%u,%u,%u,%u,%f' ak='%f' pr='%s' unsh='%016" PRIx64 "-%d' unemp='%s' unti='%u,%u,%u,%u,%u,%f' unyi='%u,%u,%u,%u,%u,%f' unpr='%s' lmt='%u' >",
                    m_shingle, m_type, BoolToStroka(m_empty).c_str(),
                    m_shingle_data.Ham(TODAY), m_shingle_data.Malic(TODAY), m_shingle_data.Spam(TODAY), m_shingle_data.PersHam(TODAY), m_shingle_data.PersSpam(TODAY), m_shingle_data.Weight(TODAY),
                    m_shingle_data.Ham(YESTERDAY), m_shingle_data.Malic(YESTERDAY), m_shingle_data.Spam(YESTERDAY), m_shingle_data.PersHam(YESTERDAY), m_shingle_data.PersSpam(YESTERDAY), m_shingle_data.Weight(YESTERDAY),
                    m_age_koef, pr_str.c_str(),
                    m_uniq_shingle, m_uniq_shingle_type, BoolToStroka(m_uniq_empty).c_str(),
                    m_uniq_shingle_data.Ham(TODAY), m_uniq_shingle_data.Malic(TODAY), m_uniq_shingle_data.Spam(TODAY), m_uniq_shingle_data.PersHam(TODAY), m_uniq_shingle_data.PersSpam(TODAY), m_uniq_shingle_data.Weight(TODAY),
                    m_uniq_shingle_data.Ham(YESTERDAY), m_uniq_shingle_data.Malic(YESTERDAY), m_uniq_shingle_data.Spam(YESTERDAY), m_uniq_shingle_data.PersHam(YESTERDAY), m_uniq_shingle_data.PersSpam(YESTERDAY), m_uniq_shingle_data.Weight(YESTERDAY),
                    pr_str_uniq.c_str(), m_last_midnight);
        }

        res = TString(buff);

    } else if (m_shform == TSHSTORAGE) {
        res = "<ST: " + ShingleToStroka(m_shingle) + " empty='" + BoolToStroka(m_empty) + "' std='" + m_st_data + "'>";
    }

    return res;
}

TString TShingleStat::GetDebugInfo() {
    TString res = "";

#ifdef SOSERV_DEBUG
    if (m_shform == TSHSHINGLE) {
        res = res + debuginfo.GetShToLog();

    } else if (m_shform == TSHSTORAGE) {
        res = res + debuginfo.GetStToLog();
    }
#endif

    return res;
}

TString TShingleStat::GetOldAnswer() {
    TString res = "";

    if (m_shform == TSHSHINGLE) {
        char buff[1024];
        ui32 v1 = 0, v2 = 0, v3 = 0, v4 = 0, v5 = 0;

        buff[sizeof(buff) - 1] = 0x00;
        v1 = Type();
        v2 = TotalHam();
        v3 = TotalMalic();
        v4 = TotalSpam();
        v5 = 0;
        snprintf(buff, sizeof(buff) - 1, "%016" PRIx64 " %u %u %u %u %u %.3f\n", Shingle(), v1, v2, v3, v4, v5, TotalWeight());
        res = TString(buff);

    } else if (m_shform == TSHSTORAGE) {
        if (!m_st_data.empty())
            res = m_st_data + "\n";
    }

    return res;
}

TString TShingleStat::GetRequest() {
    TString res = "";
    char buff[64];

    if (m_shform == TSHSHINGLE) {
        if (m_uniq_shingle > 0)
            snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d=%" PRIx64 "-%d", m_shingle, m_type, m_uniq_shingle, m_uniq_shingle_type); //shingle-type=uniq_shingle-uniq_shingle_type&
        else
        snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d=", m_shingle, m_type); //shingle-type=&

    } else if (m_shform == TSHSTORAGE) {
        //shingle&
        snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "=", m_shingle);
    }

    res = TString(buff);

    return res;
}

TString TShingleStat::PutRequest(int count, TShingleCount spamtype, TPersType pers) {
    TString res = "";

    if (m_shform == TSHSHINGLE) {
        //shingle-type=count-spamtype-pers,uniq_shingle-uniq_shingle_type&
        char buff[128];

        if (m_uniq_shingle > 0)
            snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d=%d-%d-%d,%" PRIx64 "-%d", m_shingle, m_type, count, spamtype, pers, m_uniq_shingle, m_uniq_shingle_type);
        else
            snprintf(buff, sizeof(buff) - 1, "%" PRIx64 "-%d=%d-%d-%d", m_shingle, m_type, count, spamtype, pers);
        res = TString(buff);
    }

    return res;
}

bool TShingleStat::IsValid() {
    bool res = false;

    if (Shingle() != 0)
        res = true;
    else
        res = false;

    return res;
}

TString TShingleStat::ShingleFormS() {
    TString res = "";

    switch (m_shform) {
        case TSHSHINGLE:
            res = "SHIN";
            break;
        case TSHSTORAGE:
            res = "STRG";
            break;
        default:
            res = "UNKN";
    };

    return res;
}

TString TShingleStat::ShingleAndTypeS() {
    TString res = "";

    switch (m_shform) {
        case TSHSHINGLE:
            if (m_uniq_shingle == 0)
                res = "SH:" + ShingleToStroka0x(m_shingle) + "-" + IntToStroka3(m_type);
            else
                res = "SH:" + ShingleToStroka0x(m_shingle) + "-" + IntToStroka3(m_type) + "+" + ShingleToStroka0x(m_uniq_shingle) + "-" + IntToStroka3(m_uniq_shingle_type);
            break;
        case TSHSTORAGE:
            res = "ST:" + ShingleToStroka0x(m_shingle);
            break;
        default:
            if (m_uniq_shingle == 0)
                res = "UN:" + ShingleToStroka0x(m_shingle) + "-" + IntToStroka3(m_type);
            else
                res = "UN:" + ShingleToStroka0x(m_shingle) + "-" + IntToStroka3(m_type) + "+" + ShingleToStroka0x(m_uniq_shingle) + "-" + IntToStroka3(m_uniq_shingle_type);
    };

    return res;
}

void TShingleStat::SetUniqShingleData(TShingleStat& uniq_shingle_data) {
    m_uniq_shingle = uniq_shingle_data.m_shingle;
    m_uniq_shingle_type = uniq_shingle_data.m_type;
    m_uniq_empty = uniq_shingle_data.m_empty;
    m_uniq_shingle_data = uniq_shingle_data.m_shingle_data;
    m_uniq_heavy = uniq_shingle_data.m_heavy;
    m_uniq_exists = uniq_shingle_data.m_exists;
    m_uniq_exist_today = uniq_shingle_data.m_exist_today;
    m_uniq_exist_yesterday = uniq_shingle_data.m_exist_yesterday;
#ifdef SOSERV_DEBUG
    debuginfo.AddToAttdata(uniq_shingle_data.debuginfo);
#endif
}

float TShingleStat::ReCalcKoefDiv(ui32 last_midnight_time, float age_koef) {
    float res = 0;

    if (age_koef < 1)
        res = ((float)time(nullptr) - (float)last_midnight_time) / ((float)1 - age_koef);

    return res;
}

bool TShingleStat::operator<(const TShingleStat& arg) const {
    if (m_type < arg.Type())
        return true;
    else if (m_type == arg.Type())
        return (arg.TotalMalic() + arg.TotalHam() + arg.TotalSpam() < TotalMalic() + TotalHam() + TotalSpam());

    return false;
}

NJson::TJsonValue TShingleStat::toJson() const {
    NJson::TJsonValue result;

    result.InsertValue("shingle", ShingleStr());

    NJson::TJsonValue& today = result.InsertValue("today", {});
    NJson::TJsonValue& yesterday = result.InsertValue("yesterday", {});

    today.InsertValue("spam", m_shingle_data.Spam(TDayType::TODAY));
    yesterday.InsertValue("spam", m_shingle_data.Spam(TDayType::YESTERDAY));

    today.InsertValue("ham", m_shingle_data.Ham(TDayType::TODAY));
    yesterday.InsertValue("ham", m_shingle_data.Ham(TDayType::YESTERDAY));

    today.InsertValue("malic", m_shingle_data.Malic(TDayType::TODAY));
    yesterday.InsertValue("malic", m_shingle_data.Malic(TDayType::YESTERDAY));

    today.InsertValue("pers_ham", m_shingle_data.PersHam(TDayType::TODAY));
    yesterday.InsertValue("pers_ham", m_shingle_data.PersHam(TDayType::YESTERDAY));

    today.InsertValue("pers_spam", m_shingle_data.PersSpam(TDayType::TODAY));
    yesterday.InsertValue("pers_spam", m_shingle_data.PersSpam(TDayType::YESTERDAY));

    return result;
}

//********************************************************************************************************************************
//                                           NShingleStatParser
//********************************************************************************************************************************

ui32 NShingleStatParser::GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size) {
    long pos = -1;
    ui32 res = 0;
    ui8 dob = 0;

    for (ui32 i = 0; i < source_size; i++) {
        if ((source[i] == 0x0D) && ((i + 1) < source_size) && (source[i + 1] == 0x0A)) {
            pos = i;
            dob = 1;
            break;
        }
        if (source[i] == 0x0A) {
            pos = i;
            dob = 0;
            break;
        }
    }
    res = pos + 1 + dob;
    if (res == 0)
        res = source_size;
    if (res > 0) {
        memset(destination, 0, destination_size);
        if (res <= (destination_size - 1)) {
            memcpy(destination, source, res);
        } else {
            memcpy(destination, source, destination_size - 1);
        }
    }
    return res;
}

TString NShingleStatParser::GetServerIDFromHeaders(const TString& responce) {
    TString res = "";
    ui32 m_buff_pos = 0;
    const ui32 tbuffsize = 1024;
    ui32 strlength = 0;
    const char* buff = nullptr;
    size_t buffsize = 0;
    TString serverid_ident = "ServerID: ";

    if (!responce.empty()) {
        buff = responce.c_str();
        buffsize = responce.length();

        TVector<char> tbuff(tbuffsize);

        while (m_buff_pos < buffsize) {
            strlength = GetStr(buff + m_buff_pos, buffsize - m_buff_pos, tbuff.data(), tbuffsize);
            if (strlength > 0) {
                m_buff_pos += strlength;
                if ((strlength > serverid_ident.length()) && (memcmp(tbuff.data(), serverid_ident.c_str(), serverid_ident.length()) == 0)) {
                    res = Trim(TString(tbuff.data() + serverid_ident.length()));
                    break;
                }

            } else
                break;
        }
    }

    return res;
}

TString NShingleStatParser::GetServerIDFromBody(const TString& responce) {
    //<SRVID:F0B0-2944>
    TString res = "";
    const char* buff = nullptr;
    TString serverid_ident = "<SRVID:";
    const char *p1 = nullptr, *p2 = nullptr;
    int count = 0;

    if (!responce.empty()) {
        buff = responce.c_str(); //strstr(responce.c_str(), "\r\n\r\n");
        if (buff != nullptr) {
            p1 = strstr(buff, serverid_ident.c_str());
            if (p1 != nullptr) {
                p2 = strchr(p1, '>');
                count = p2 - p1 - serverid_ident.length();
                if (count > 0)
                    res = TString(p1 + serverid_ident.length(), count);
            }
        }
    }

    return res;
}

TString NShingleStatParser::GetRequestIDFromHeaders(const TString& responce) {
    TString res = "";
    ui32 m_buff_pos = 0;
    const ui32 tbuffsize = 1024;
    ui32 strlength = 0;
    const char* buff = nullptr;
    size_t buffsize = 0;
    TString serverid_ident = "RequestID: ";

    if (!responce.empty()) {
        buff = responce.c_str();
        buffsize = responce.length();

        TTempBuf tbuff(tbuffsize);

        while (m_buff_pos < buffsize) {
            strlength = GetStr(buff + m_buff_pos, buffsize - m_buff_pos, tbuff.Data(), tbuffsize);
            if (strlength > 0) {
                m_buff_pos += strlength;
                if ((strlength > serverid_ident.length()) && (memcmp(tbuff.Data(), serverid_ident.c_str(), serverid_ident.length()) == 0)) {
                    res = Trim(TString(tbuff.Data() + serverid_ident.length()));
                    break;
                }

            } else
                break;
        }
    }

    return res;
}

bool NShingleStatParser::GetDataByIdent(const char* start, const TString& Ident, TString& resdata) {
    bool res = "";
    const char* p = nullptr;
    const char* p2 = nullptr;
    const char* p3 = nullptr;
    int count = 0;

    resdata = "";
    p3 = strchr(start, '>');
    p = strstr(start, Ident.c_str());
    if ((p != nullptr) && (p3 != nullptr) && (p < p3)) {
        p2 = strchr(p + Ident.length(), '\'');
        if (p2 != nullptr) {
            count = p2 - p - Ident.length();
            if (count > 0) {
                resdata = TString(p + Ident.length(), count);
                res = true;
            }
        }
    }

    return res;
}

bool NShingleStatParser::ParseShinglesData(const TString& responce, TShingleStatList& shingle_list) {
    bool res = false;
    const char* pstart = nullptr;
    const char* pb = nullptr;
    const char* pe = nullptr;
    const char* p = nullptr;
    //    const char* p2 = nullptr;
    //    const char* pform = nullptr;
    TString tident = "";
    ui64 shingle = 0;
    int type = 0;
    ui64 uniq_shingle = 0;
    int uniq_type = 0;
    bool empty = true;
    bool uniq_empty = true;
    float agekoef = 0;
    ui32 lastmidnight = 0;
    TString stnn = "";
    int count = 0;
    TString shident = "<SH: ";
    TString stident = "<ST: ";
    TString empty_ident = "empty='";
    TString today_ident = "ti='";
    TString std_ident = "std='";
    TString yesterday_ident = "yi='";
    TString agekoef_ident = "ak='";
    TString pr_ident = "pr='";
    TString uniq_empty_ident = "unemp='";
    TString uniq_today_ident = "unti='";
    TString uniq_yesterday_ident = "unyi='";
    TString uniq_pr_ident = "unpr='";
    TString uniq_shingle_ident = "unsh='";
    TString lmt_ident = "lmt='";
    ui16 today_ham = 0;
    ui16 today_dlv = 0;
    ui16 today_spam = 0;
    ui16 today_persham = 0;
    ui16 today_persspam = 0;
    float today_weight = 0;
    ui16 yesterday_ham = 0;
    ui16 yesterday_dlv = 0;
    ui16 yesterday_spam = 0;
    ui16 yesterday_persham = 0;
    ui16 yesterday_persspam = 0;
    float yesterday_weight = 0;
    TString priznak = "";
    ui16 uniq_today_ham = 0;
    ui16 uniq_today_dlv = 0;
    ui16 uniq_today_spam = 0;
    ui16 uniq_today_persham = 0;
    ui16 uniq_today_persspam = 0;
    float uniq_today_weight = 0;
    ui16 uniq_yesterday_ham = 0;
    ui16 uniq_yesterday_dlv = 0;
    ui16 uniq_yesterday_spam = 0;
    ui16 uniq_yesterday_persham = 0;
    ui16 uniq_yesterday_persspam = 0;
    float uniq_yesterday_weight = 0;
    TString uniq_priznak = "";
    bool empty_pr = false;
    bool today_pr = false;
    bool std_pr = false;
    bool yesterday_pr = false;
    bool agekoef_pr = false;
    bool priznak_pr = false;
    bool heavy = false;
    bool uniq_shingle_pr = false;
    bool uniq_empty_pr = false;
    bool uniq_today_pr = false;
    bool uniq_yesterday_pr = false;
    bool uniq_priznak_pr = false;
    bool uniq_heavy = false;
    bool lastmidnight_pr = false;
    TString std_data = "";
    int v1 = 0, v2 = 0, v3 = 0, v4 = 0, v5 = 0;
    TShingleStat shinglestat;
    TShingleStat shinglestat_uniq;
    TString resdata = "";

    shingle_list.clear();
    pstart = responce.c_str(); //strstr(responce.c_str(), "\r\n\r\n");

    if (pstart != nullptr) //парсим тело
    {
        pe = nullptr;
        pb = strchr(pstart, '<');
        if (pb != nullptr)
            pe = strchr(pb, '>');

        while ((pb != nullptr) && (pe != nullptr)) {
            empty_pr = false;
            today_pr = false;
            yesterday_pr = false;
            agekoef_pr = false;
            priznak_pr = false;
            std_pr = false;
            tident = "";

            shingle = 0;
            type = 0;
            agekoef = 0;
            empty = true;
            today_ham = 0;
            today_dlv = 0;
            today_spam = 0;
            today_persham = 0;
            today_persspam = 0;
            today_weight = 0;
            yesterday_ham = 0;
            yesterday_dlv = 0;
            yesterday_spam = 0;
            yesterday_persham = 0;
            yesterday_persspam = 0;
            yesterday_weight = 0;
            heavy = false;
            std_data = "";

            if ((pe - pb) > (long)shident.length())
                tident = TString(pb, shident.length());

            if (tident == shident) {
                //<SH: b54092d607904197-2 empty='0' ti='0,0,2,0,0,0.000000' yi='0,0,0,0,0,0.000000' ak='0.000000' pr='oe' unsh='b54092d607904191-3' unemp='0' unti='0,0,2,0,0,0.000000' unyi='0,0,0,0,0,0.000000' unpr='oe' lmt='1234567' >
                pb = pb + shident.length();

                p = strchr(pb, ' ');
                if (p != nullptr) {
                    count = p - pb;
                    if (count > 0) {
                        stnn = TString(pb, count);
                        if (!stnn.empty()) {
                            if (sscanf(stnn.c_str(), "%" PRIx64 "-%d", &shingle, &type) == 2) {
                                //find empty
                                if (GetDataByIdent(pb, empty_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        empty = (bool)atoi(resdata.c_str());
                                        empty_pr = true;
                                    }
                                }

                                //find today data
                                if (GetDataByIdent(pb, today_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        v1 = 0;
                                        v2 = 0;
                                        v3 = 0;
                                        v4 = 0;
                                        v5 = 0;
                                        if (sscanf(resdata.c_str(), "%d,%d,%d,%d,%d,%f", &v1, &v2, &v3, &v4, &v5, &today_weight) == 6) {
                                            today_ham = v1;
                                            today_dlv = v2;
                                            today_spam = v3;
                                            today_persham = v4;
                                            today_persspam = v5;
                                            today_pr = true;
                                        }
                                    }
                                }

                                //find yesterday data
                                if (GetDataByIdent(pb, yesterday_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        v1 = 0;
                                        v2 = 0;
                                        v3 = 0;
                                        v4 = 0;
                                        v5 = 0;
                                        if (sscanf(resdata.c_str(), "%d,%d,%d,%d,%d,%f", &v1, &v2, &v3, &v4, &v5, &yesterday_weight) == 6) {
                                            yesterday_ham = v1;
                                            yesterday_dlv = v2;
                                            yesterday_spam = v3;
                                            yesterday_persham = v4;
                                            yesterday_persspam = v5;
                                            yesterday_pr = true;
                                        }
                                    }
                                }

                                //find age koef
                                if (GetDataByIdent(pb, agekoef_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        agekoef = atof(resdata.c_str());
                                        agekoef_pr = true;
                                    }
                                }

                                //find last midnight
                                if (GetDataByIdent(pb, lmt_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        lastmidnight = atol(resdata.c_str());
                                        lastmidnight_pr = true;
                                    }
                                }

                                //find priznak
                                if (GetDataByIdent(pb, pr_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        priznak = resdata;
                                        priznak_pr = true;
                                    }
                                }

                                //************************ uniq data ************************

                                //find uniq shingle data
                                if (GetDataByIdent(pb, uniq_shingle_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        if (sscanf(resdata.c_str(), "%" PRIx64 "-%d", &uniq_shingle, &uniq_type) == 2)
                                        uniq_shingle_pr = true;
                                    }
                                }

                                //find uniq empty
                                if (GetDataByIdent(pb, uniq_empty_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        uniq_empty = (bool)atoi(resdata.c_str());
                                        uniq_empty_pr = true;
                                    }
                                }

                                //find uniq today data
                                if (GetDataByIdent(pb, uniq_today_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        v1 = 0;
                                        v2 = 0;
                                        v3 = 0;
                                        v4 = 0;
                                        v5 = 0;
                                        if (sscanf(resdata.c_str(), "%d,%d,%d,%d,%d,%f", &v1, &v2, &v3, &v4, &v5, &uniq_today_weight) == 6) {
                                            uniq_today_ham = v1;
                                            uniq_today_dlv = v2;
                                            uniq_today_spam = v3;
                                            uniq_today_persham = v4;
                                            uniq_today_persspam = v5;
                                            uniq_today_pr = true;
                                        }
                                    }
                                }

                                //find uniq yesterday data
                                if (GetDataByIdent(pb, uniq_yesterday_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        v1 = 0;
                                        v2 = 0;
                                        v3 = 0;
                                        v4 = 0;
                                        v5 = 0;
                                        if (sscanf(resdata.c_str(), "%d,%d,%d,%d,%d,%f", &v1, &v2, &v3, &v4, &v5, &uniq_yesterday_weight) == 6) {
                                            uniq_yesterday_ham = v1;
                                            uniq_yesterday_dlv = v2;
                                            uniq_yesterday_spam = v3;
                                            uniq_yesterday_persham = v4;
                                            uniq_yesterday_persspam = v5;
                                            uniq_yesterday_pr = true;
                                        }
                                    }
                                }

                                //find uniq priznak
                                if (GetDataByIdent(pb, uniq_pr_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        uniq_priznak = resdata;
                                        uniq_priznak_pr = true;
                                    }
                                }

                                //********************* компануем шингл **************************

                                //формируем статистику шингла
                                if (empty_pr && today_pr && yesterday_pr && agekoef_pr) {
                                    res = true;
                                    shinglestat = TShingleStat(shingle, type, 0);
                                    if (priznak_pr) {
                                        if (priznak.length() > 0) {
                                            if (priznak[0] == 'h')
                                                heavy = true;
                                            else
                                                heavy = false;
                                        }

                                        if (priznak.length() > 1) {
                                            if (priznak[1] == 'e')
                                                shinglestat.SetAction(AEXISTS);
                                            else if (priznak[1] == 'b')
                                                shinglestat.SetAction(ABUSY);
                                            else if (priznak[1] == 't')
                                                shinglestat.SetAction(AEXCEEDTYPE);
                                            else if (priznak[1] == 's')
                                                shinglestat.SetAction(ABADTYPEHASH);
                                            else if (priznak[1] == 'p')
                                                shinglestat.SetAction(AEXCEEDPART);
                                            else if (priznak[1] == 'q')
                                                shinglestat.SetAction(ABADPARTHASH);
                                            else if (priznak[1] == 'n')
                                                shinglestat.SetAction(ANOTEXISTS);
                                            else
                                                shinglestat.SetAction(AUNDEF);
                                        }
                                    }
                                    shinglestat.SetShingleData(TShingleDataTE(today_ham, today_dlv, today_spam, today_persham, today_persspam, today_weight, yesterday_ham, yesterday_dlv, yesterday_spam, yesterday_persham, yesterday_persspam, yesterday_weight), heavy, 0);
                                    shinglestat.SetAgeKoef(agekoef);
                                    shinglestat.SetEmptyStatus(empty);
                                    shinglestat.SetLastMidnight(lastmidnight);

                                    //данные по uniq shingle
                                    if (uniq_shingle_pr && uniq_empty_pr && uniq_today_pr && uniq_yesterday_pr) {
                                        shinglestat_uniq = TShingleStat(uniq_shingle, uniq_type, 0);
                                        if (uniq_priznak_pr) {
                                            if (uniq_priznak.length() > 0) {
                                                if (uniq_priznak[0] == 'h')
                                                    uniq_heavy = true;
                                                else
                                                    uniq_heavy = false;
                                            }
                                            if (uniq_priznak.length() > 1) {
                                                if (uniq_priznak[1] == 'e')
                                                    shinglestat_uniq.SetAction(AEXISTS);
                                                else if (uniq_priznak[1] == 'b')
                                                    shinglestat_uniq.SetAction(ABUSY);
                                                else if (uniq_priznak[1] == 't')
                                                    shinglestat_uniq.SetAction(AEXCEEDTYPE);
                                                else if (uniq_priznak[1] == 's')
                                                    shinglestat_uniq.SetAction(ABADTYPEHASH);
                                                else if (uniq_priznak[1] == 'p')
                                                    shinglestat_uniq.SetAction(AEXCEEDPART);
                                                else if (uniq_priznak[1] == 'q')
                                                    shinglestat_uniq.SetAction(ABADPARTHASH);
                                                else if (uniq_priznak[1] == 'n')
                                                    shinglestat_uniq.SetAction(ANOTEXISTS);
                                                else
                                                    shinglestat_uniq.SetAction(AUNDEF);
                                            }
                                        }
                                        shinglestat_uniq.SetShingleData(TShingleDataTE(uniq_today_ham, uniq_today_dlv, uniq_today_spam, uniq_today_persham, uniq_today_persspam, uniq_today_weight, uniq_yesterday_ham, uniq_yesterday_dlv, uniq_yesterday_spam, uniq_yesterday_persham, uniq_yesterday_persspam, uniq_yesterday_weight), uniq_heavy, 0);
                                        shinglestat_uniq.SetAgeKoef(agekoef);
                                        shinglestat_uniq.SetEmptyStatus(uniq_empty);

                                        shinglestat.SetUniqShingleData(shinglestat_uniq);
                                    }

                                    //добавляем в список
                                    shingle_list.push_back(shinglestat);
                                }
                            }
                        }
                    }
                }

            } else if (tident == stident) {
                //<ST: b54092d607904197 empty='0' std='absdefg'>
                pb = pb + shident.length();

                p = strchr(pb, ' ');
                if (p != nullptr) {
                    count = p - pb;
                    if (count > 0) {
                        stnn = TString(pb, count);
                        if (!stnn.empty()) {
                            if (sscanf(stnn.c_str(), "%" PRIx64, &shingle) == 1) {
                                //find empty
                                if (GetDataByIdent(pb, empty_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        empty = (bool)atoi(resdata.c_str());
                                        empty_pr = true;
                                    }
                                }

                                //find std data
                                if (GetDataByIdent(pb, std_ident, resdata)) {
                                    if (!resdata.empty()) {
                                        std_data = resdata;
                                        std_pr = true;
                                    }
                                }

                                //формируем статистику шингла
                                if (empty_pr && std_pr) {
                                    res = true;
                                    shinglestat = TShingleStat(shingle);

                                    shinglestat.SetStorageData(std_data);
                                    shinglestat.SetEmptyStatus(empty);

                                    shingle_list.push_back(shinglestat);
                                }
                            }
                        }
                    }
                }
            }

            pstart = pe + 1;
            if (pstart != nullptr) {
                pe = nullptr;
                pb = strchr(pstart, '<');
                if (pb != nullptr)
                    pe = strchr(pb, '>');
            } else
                break;
        }
    }

    return res;
}

//********************************************************************************************************************************
