#pragma once

#ifndef tshinglerenv_H
#define tshinglerenv_H

#include <util/generic/hash.h>
#include <util/system/mutex.h>
#include <util/system/defaults.h>
#include <util/stream/format.h>
#include <vector>
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <mail/so/spamstop/tools/so-common/shtime.h>

enum TShingleCount { HAM,
    MALIC,
    SPAM,
    UNKNOWN };
enum TDay { TODAY,
    YESTERDAY };
typedef TShingleCount TShingleSpamType;
typedef TDay TDayType;
enum TActionType { AUNDEF,
                   AEXISTS,
                   ABUSY,
                   AEXCEEDTYPE,
                   ABADTYPEHASH,
                   AEXCEEDPART,
                   ABADPARTHASH,
                   ANOTEXISTS };
enum TPersType { PERSUNDEF,
                 PERSHAM,
                 PERSSPAM };

TString TShingleSpamTypeToStroka(TShingleSpamType spamtype);
TString TActionTypeToStroka(TActionType act);
TString TPersTypeToStroka(TPersType pers);

//********************************************************************************************************************************
//                                           TShingleDataSingle - хранится в памяти heavy
//********************************************************************************************************************************

#define SDS_HAVE_YESTERDAY_DATA 0x00000001
#define SDS_DONTHAVE_YESTERDAY_DATA 0x00000002

#define SET_SDS_HAVE_YESTERDAY_DATA(flag) \
    { (flag) |= SDS_HAVE_YESTERDAY_DATA; }
#define SET_SDS_DONTHAVE_YESTERDAY_DATA(flag) \
    { (flag) |= SDS_DONTHAVE_YESTERDAY_DATA; }

#define IS_SDS_HAVE_YESTERDAY_DATA(flag) ((flag)&SDS_HAVE_YESTERDAY_DATA)
#define IS_SDS_DONTHAVE_YESTERDAY_DATA(flag) ((flag)&SDS_DONTHAVE_YESTERDAY_DATA)

class TShingleDataSingle {
private:
    ui16 m_ham;
    ui16 m_spam;
    ui16 m_malic;
    ui16 m_pers_ham;
    ui16 m_pers_spam;
    float m_weight;
    ui8 m_ham_diff;
    ui8 m_spam_diff;
    ui8 m_malic_diff;
    ui8 m_persham_diff;
    ui8 m_persspam_diff;
    ui8 m_pr;

public:
    TShingleDataSingle();
    TShingleDataSingle(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam, float weight);
    TShingleDataSingle(TShingleSpamType spamtype, ui16 count, TPersType pers);
    TShingleDataSingle(float weight);
    ~TShingleDataSingle();

    void Add(TShingleSpamType spamtype, ui16 count, TPersType pers);
    void Add(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam);
    void AddWODiff(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam);
    void Set(ui16 ham, ui16 malic, ui16 spam, ui16 persham, ui16 persspam, float weight);
    void SetWeight(float weight);

    TString GetFollowText();

    ui16 Ham() {
        return m_ham;
    }
    ui16 Malic() {
        return m_malic;
    }
    ui16 Spam() {
        return m_spam;
    }
    ui16 PersHam() {
        return m_pers_ham;
    }
    ui16 PersSpam() {
        return m_pers_spam;
    }
    float Weight() {
        return m_weight;
    }
    ui16 DiffHam() {
        return m_ham_diff;
    }
    ui16 DiffSpam() {
        return m_spam_diff;
    }
    ui16 DiffMalic() {
        return m_malic_diff;
    }
    ui16 DiffPersHam() {
        return m_persham_diff;
    }
    ui16 DiffPersSpam() {
        return m_persspam_diff;
    }

    bool HaveYesterdayData() {
        return IS_SDS_HAVE_YESTERDAY_DATA(m_pr);
    }
    void SetHaveYesterdayData() {
        SET_SDS_HAVE_YESTERDAY_DATA(m_pr);
    }
    bool DontHaveYesterdayData() {
        return IS_SDS_DONTHAVE_YESTERDAY_DATA(m_pr);
    }
    void SetDontHaveYesterdayData() {
        SET_SDS_DONTHAVE_YESTERDAY_DATA(m_pr);
    }
    ui8 GetPr() {
        return m_pr;
    }

    void Clear();
    void ClearDiffs();
};

typedef THashMap<ui64, TShingleDataSingle> TShingleDataSingleHash;
typedef TShingleDataSingleHash::iterator TShingleDataSingleHashIt;

//********************************************************************************************************************************
//                                           TShingleData - хранится в памяти (совмещенная today/yesterday)
//********************************************************************************************************************************

class TShingleDataTE {
private:
    ui16 m_ham[2];
    ui16 m_spam[2];
    ui16 m_malic[2]; //используем как malic
    ui16 m_pers_ham[2];
    ui16 m_pers_spam[2];
    float m_weight[2];
    ui16 m_ham_diff;
    ui16 m_spam_diff;
    ui16 m_malic_diff;
    ui16 m_persham_diff;
    ui16 m_persspam_diff;

public:
    TShingleDataTE();
    TShingleDataTE(ui16 t_ham, ui16 t_malic, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_malic, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight);
    TShingleDataTE(TShingleDataSingle* t_data, TShingleDataSingle* y_data);
    ~TShingleDataTE();

    void Add(TShingleSpamType spamtype, ui16 count, TPersType pers);
    void Set(ui16 t_ham, ui16 t_malic, ui16 t_spam, ui16 t_persham, ui16 t_persspam, float t_weight, ui16 y_ham, ui16 y_malic, ui16 y_spam, ui16 y_persham, ui16 y_persspam, float y_weight);

    ui16 Ham(TDayType daytype) const {
        return m_ham[daytype];
    }
    ui16 Malic(TDayType daytype) const {
        return m_malic[daytype];
    }
    ui16 Spam(TDayType daytype) const {
        return m_spam[daytype];
    }

    ui16 PersHam(TDayType daytype) const {
        return m_pers_ham[daytype];
    }
    ui16 PersSpam(TDayType daytype) const {
        return m_pers_spam[daytype];
    }
    float Weight(TDayType daytype) const {
        return m_weight[daytype];
    }

    void Clear();
};

typedef THashMap<ui64, TShingleDataTE> TShingleDataTEHash;
typedef TShingleDataTEHash::iterator TShingleDataTEHashIt;

//********************************************************************************************************************************
//                                        TObrabShingleDebugInfo - отладочные данные по обработке шингла
//********************************************************************************************************************************

enum TSOSERVDBG_LOCKTYPE { TD_NONE,
                           TD_GET,
                           TD_PUTA,
                           TD_PUTMA,
                           TD_SETMA,
                           TD_SETWEIGHTA,
                           TD_CLEARSHINGLEA,
                           TD_WRITEDUMPA,
                           TD_CLEANUP,
                           TD_MIDNIGHT,
                           TD_SETLASTMIDNIGHTTIME,
                           TD_SETLASTCLEANUPTIME };

TString TSOSERVDBG_LOCKTYPEToStroka(TSOSERVDBG_LOCKTYPE type);

struct TPrevRequestTiming {
    ui32 m_find_today{};     //поиск в хэше сегодня
    ui32 m_find_yesterday{}; //поиск в хэше вчера
    ui32 m_createshingle{};  //время создания (добавления) шингла
    ui32 m_lockheavy{};      //ожидание на локе хэша тяжелых шинглов
    ui32 m_findheavy{};      //поиск в хэше тяжелых шинглов
    ui32 m_addheavy{};       //добавляем в хэш тяжелых шинглов

    TPrevRequestTiming() {
        Clear();
    }

    void Clear() {
        m_find_today = 0;
        m_find_yesterday = 0;
        m_createshingle = 0;
        m_lockheavy = 0;
        m_findheavy = 0;
        m_addheavy = 0;
    }

    void SetFindToday(ui32 find_today) {
        m_find_today = find_today;
    }
    void SetFindYesterday(ui32 find_yesterday) {
        m_find_yesterday = find_yesterday;
    }
    void SetCreateShingle(ui32 createshingle) {
        m_createshingle = createshingle;
    }
    void SetLockHeavy(ui32 lockheavy) {
        m_lockheavy = lockheavy;
    }
    void SetFindHeavy(ui32 findheavy) {
        m_findheavy = findheavy;
    }
    void SetAddHeavy(ui32 addheavy) {
        m_addheavy = addheavy;
    }

    ui32 GetFindToday() {
        return m_find_today;
    }
    ui32 GetFindYesterday() {
        return m_find_yesterday;
    }
    ui32 GetCreateShingle() {
        return m_createshingle;
    }
    ui32 GetLockHeavy() {
        return m_lockheavy;
    }
    ui32 GetFindHeavy() {
        return m_findheavy;
    }
    ui32 GetAddHeavy() {
        return m_addheavy;
    }
};

struct TObrabShingleDebugInfoItem {
    TSOSERVDBG_LOCKTYPE m_lock_type;         //тип блокировки перед ожиданием лока
    ui32 m_summary{};                        //суммарное время выполнения запроса
    ui32 m_lockheavy{};                      //ожидание на локе хэша тяжелых шинглов
    ui32 m_findheavy{};                      //поиск в хэше тяжелых шинглов
    ui32 m_addheavy{};                       //добавляем в хэш тяжелых шинглов
    ui32 m_busyaddstat{};                    //время добавления статистики по busy
    ui32 m_busyaddstat_heavy{};              //время добавления статистики по busy heavy
    ui32 m_busyapply{};                      //применения накруток из очереди
    ui32 m_lock{};                           //ожидание на логе вчера/сегодня
    ui32 m_find_today{};                     //поиск в хэше сегодня
    ui32 m_find_yesterday{};                 //поиск в хэше вчера
    ui32 m_createshingle{};                  //время создания (добавления) шингла
    ui32 m_add_to_rplpool{};                 //добавляем в пул на репликацию
    ui32 m_find_follow_shingle{};            //поиск шингла в списке следящих шинглов
    ui32 m_find_follow_shingle2{};           //поиск шингла в списке следящих шинглов (2й отсчет)
    ui32 m_find_follow_shingle_rpl{};        //поиск шингла в списке следящих шинглов (replic)
    ui32 m_writelog_follow_shingle{};        //запись в лог информации по отслеживаемому шинглу
    ui32 m_writelog_follow_shingle2{};       //запись в лог информации по отслеживаемому шинглу (2й отсчет)
    ui32 m_writelog_follow_shingle_rpl{};    //запись в лог информации по отслеживаемому шинглу (2й отсчет)
    ui32 m_storage_delay{};                  //выдача строки storage
    TPrevRequestTiming m_prev_rqst_timing;   //информация об обработке внутри OrdinaryLock()

    void SetLockType(TSOSERVDBG_LOCKTYPE lock_type) {
        m_lock_type = lock_type;
    }
    void SetSummary(ui32 summary) {
        m_summary = summary;
    }
    void SetLockHeavy(ui32 lockheavy) {
        m_lockheavy = lockheavy;
    }
    void SetFindHeavy(ui32 findheavy) {
        m_findheavy = findheavy;
    }
    void SetAddHeavy(ui32 addheavy) {
        m_addheavy = addheavy;
    }
    void SetBusyAddStat(ui32 busyaddstat) {
        m_busyaddstat = busyaddstat;
    }
    void SetBusyAddStatHeavy(ui32 busyaddstat_heavy) {
        m_busyaddstat_heavy = busyaddstat_heavy;
    }
    void SetBusyApply(ui32 busyapply) {
        m_busyapply = busyapply;
    }
    void SetLock(ui32 lock) {
        m_lock = lock;
    }
    void SetFindToday(ui32 find_today) {
        m_find_today = find_today;
    }
    void SetFindYesterday(ui32 find_yesterday) {
        m_find_yesterday = find_yesterday;
    }
    void SetCreateShingle(ui32 createshingle) {
        m_createshingle = createshingle;
    }
    void SetAddToRplPool(ui32 add_to_rplpool) {
        m_add_to_rplpool = add_to_rplpool;
    }
    void SetFindFollowShingle(ui32 find_follow_shingle) {
        m_find_follow_shingle = find_follow_shingle;
    }
    void SetFindFollowShingle2(ui32 find_follow_shingle2) {
        m_find_follow_shingle2 = find_follow_shingle2;
    }
    void SetFindFollowShingleRpl(ui32 find_follow_shingle_rpl) {
        m_find_follow_shingle_rpl = find_follow_shingle_rpl;
    }
    void SetWriteLogFollowShingle(ui32 writelog_follow_shingle) {
        m_writelog_follow_shingle = writelog_follow_shingle;
    }
    void SetWriteLogFollowShingle2(ui32 writelog_follow_shingle2) {
        m_writelog_follow_shingle2 = writelog_follow_shingle2;
    }
    void SetWriteLogFollowShingleRpl(ui32 writelog_follow_shingle_rpl) {
        m_writelog_follow_shingle_rpl = writelog_follow_shingle_rpl;
    }
    void SetStorageDelay(ui32 storage_delay) {
        m_storage_delay = storage_delay;
    }

    TSOSERVDBG_LOCKTYPE GetLockType() {
        return m_lock_type;
    }
    ui32 GetSummary() {
        return m_summary;
    }
    ui32 GetLockHeavy() {
        return m_lockheavy;
    }
    ui32 GetFindHeavy() {
        return m_findheavy;
    }
    ui32 GetAddHeavy() {
        return m_addheavy;
    }
    ui32 GetBusyAddStat() {
        return m_busyaddstat;
    }
    ui32 GetBusyAddStatHeavy() {
        return m_busyaddstat_heavy;
    }
    ui32 GetBusyApply() {
        return m_busyapply;
    }
    ui32 GetLock() {
        return m_lock;
    }
    ui32 GetFindToday() {
        return m_find_today;
    }
    ui32 GetFindYesterday() {
        return m_find_yesterday;
    }
    ui32 GetCreateShingle() {
        return m_createshingle;
    }
    ui32 GetAddToRplPool() {
        return m_add_to_rplpool;
    }
    ui32 GetFindFollowShingle() {
        return m_find_follow_shingle;
    }
    ui32 GetFindFollowShingle2() {
        return m_find_follow_shingle2;
    }
    ui32 GetFindFollowShingleRpl() {
        return m_find_follow_shingle_rpl;
    }
    ui32 GetWriteLogFollowShingle() {
        return m_writelog_follow_shingle;
    }
    ui32 GetWriteLogFollowShingle2() {
        return m_writelog_follow_shingle2;
    }
    ui32 GetWriteLogFollowShingleRpl() {
        return m_writelog_follow_shingle_rpl;
    }
    ui32 GetStorageDelay() {
        return m_storage_delay;
    }

    TObrabShingleDebugInfoItem() {
        Clear();
    }

    void Clear() {
        m_lock_type = TD_NONE;
        m_summary = 0;
        m_lockheavy = 0;
        m_findheavy = 0;
        m_addheavy = 0;
        m_busyaddstat = 0;
        m_busyaddstat_heavy = 0;
        m_busyapply = 0;
        m_lock = 0;
        m_find_today = 0;
        m_find_yesterday = 0;
        m_createshingle = 0;
        m_add_to_rplpool = 0;
        m_find_follow_shingle = 0;
        m_find_follow_shingle2 = 0;
        m_find_follow_shingle_rpl = 0;
        m_writelog_follow_shingle = 0;
        m_writelog_follow_shingle2 = 0;
        m_writelog_follow_shingle_rpl = 0;
        m_storage_delay = 0;
        m_prev_rqst_timing.Clear();
    }

    void AddPrevRqstTiming(TPrevRequestTiming value) {
        m_prev_rqst_timing = value;
    }

    TString toLog(bool is_storage) {
        char buff[256];

        if (!is_storage) {
            buff[sizeof(buff) - 1] = 0x00;
            snprintf(buff, sizeof(buff) - 1, "%s,%u,%u-%u-%u,%u-%u-%u-%u-%u-%u,%u-%u-%u,%u-%u-%u-%u,%u-%u-%u-%u-%u-%u-%u",
                     TSOSERVDBG_LOCKTYPEToStroka(m_lock_type).c_str(), m_summary,
                     m_lockheavy, m_findheavy, m_addheavy,
                     m_prev_rqst_timing.m_find_today, m_prev_rqst_timing.m_find_yesterday, m_prev_rqst_timing.m_createshingle, m_prev_rqst_timing.m_lockheavy, m_prev_rqst_timing.m_findheavy, m_prev_rqst_timing.m_addheavy,
                     m_busyaddstat, m_busyaddstat_heavy, m_busyapply,
                     m_lock, m_find_today, m_find_yesterday, m_createshingle,
                     m_add_to_rplpool, m_find_follow_shingle, m_find_follow_shingle2, m_find_follow_shingle_rpl, m_writelog_follow_shingle, m_writelog_follow_shingle2, m_writelog_follow_shingle_rpl);

        } else {
            buff[sizeof(buff) - 1] = 0x00;
            snprintf(buff, sizeof(buff) - 1, "%u", m_storage_delay);
        }

        return TString(buff);
    }
};

class TObrabShingleDebugInfo {
private:
    ui64 m_shingle;
    int m_type;
    ui64 m_att_shingle;
    int m_att_type;

    TObrabShingleDebugInfoItem data;
    TObrabShingleDebugInfoItem att_data;

public:
    TObrabShingleDebugInfo();
    ~TObrabShingleDebugInfo();

    void Init(ui64 shingle, int type);
    void Init(ui64 shingle, int type, ui64 att_shingle, int att_type);

    ui64 Shingle() {
        return m_shingle;
    }
    int Type() {
        return m_type;
    }
    ui64 Shingle_uniq() {
        return m_att_shingle;
    }
    int Type_uniq() {
        return m_att_type;
    }

    void Clear();
    ui32 GetTick() {
        return CShingleTime::GetMs();
    }

    void SetLockType(TSOSERVDBG_LOCKTYPE lock_type) {
        data.SetLockType(lock_type);
    }
    void SetSummary(ui32 summary) {
        data.SetSummary(summary);
    }
    void SetLockHeavy(ui32 lockheavy) {
        data.SetLockHeavy(lockheavy);
    }
    void SetFindHeavy(ui32 findheavy) {
        data.SetFindHeavy(findheavy);
    }
    void SetAddHeavy(ui32 addheavy) {
        data.SetAddHeavy(addheavy);
    }
    void SetBusyAddStat(ui32 busyaddstat) {
        data.SetBusyAddStat(busyaddstat);
    }
    void SetBusyAddStatHeavy(ui32 busyaddstat_heavy) {
        data.SetBusyAddStatHeavy(busyaddstat_heavy);
    }
    void SetBusyApply(ui32 busyapply) {
        data.SetBusyApply(busyapply);
    }
    void SetLock(ui32 lock) {
        data.SetLock(lock);
    }
    void SetFindToday(ui32 find_today) {
        data.SetFindToday(find_today);
    }
    void SetFindYesterday(ui32 find_yesterday) {
        data.SetFindYesterday(find_yesterday);
    }
    void SetCreateShingle(ui32 createshingle) {
        data.SetCreateShingle(createshingle);
    }
    void SetAddToRplPool(ui32 add_to_rplpool) {
        data.SetAddToRplPool(add_to_rplpool);
    }
    void SetFindFollowShingle(ui32 find_follow_shingle) {
        data.SetFindFollowShingle(find_follow_shingle);
    }
    void SetFindFollowShingle2(ui32 find_follow_shingle2) {
        data.SetFindFollowShingle2(find_follow_shingle2);
    }
    void SetFindFollowShingleRpl(ui32 find_follow_shingle_rpl) {
        data.SetFindFollowShingleRpl(find_follow_shingle_rpl);
    }
    void SetWriteLogFollowShingle(ui32 writelog_follow_shingle) {
        data.SetWriteLogFollowShingle(writelog_follow_shingle);
    }
    void SetWriteLogFollowShingle2(ui32 writelog_follow_shingle2) {
        data.SetWriteLogFollowShingle2(writelog_follow_shingle2);
    }
    void SetWriteLogFollowShingleRpl(ui32 writelog_follow_shingle_rpl) {
        data.SetWriteLogFollowShingleRpl(writelog_follow_shingle_rpl);
    }
    void SetStorageDelay(ui32 storage_delay) {
        data.SetStorageDelay(storage_delay);
    }

    void SetSummary_start() {
        data.SetSummary(GetTick());
    }
    void SetSummary_stop() {
        data.SetSummary(GetTick() - data.GetSummary());
    }

    void SetLockHeavy_start() {
        data.SetLockHeavy(GetTick());
    }
    void SetLockHeavy_stop() {
        data.SetLockHeavy(GetTick() - data.GetLockHeavy());
    }

    void SetFindHeavy_start() {
        data.SetFindHeavy(GetTick());
    }
    void SetFindHeavy_stop() {
        data.SetFindHeavy(GetTick() - data.GetFindHeavy());
    }

    void SetAddHeavy_start() {
        data.SetAddHeavy(GetTick());
    }
    void SetAddHeavy_stop() {
        data.SetAddHeavy(GetTick() - data.GetAddHeavy());
    }

    void SetBusyAddStat_start() {
        data.SetBusyAddStat(GetTick());
    }
    void SetBusyAddStat_stop() {
        data.SetBusyAddStat(GetTick() - data.GetBusyAddStat());
    }

    void SetBusyAddStatHeavy_start() {
        data.SetBusyAddStatHeavy(GetTick());
    }
    void SetBusyAddStatHeavy_stop() {
        data.SetBusyAddStatHeavy(GetTick() - data.GetBusyAddStatHeavy());
    }

    void SetBusyApply_start() {
        data.SetBusyApply(GetTick());
    }
    void SetBusyApply_stop() {
        data.SetBusyApply(GetTick() - data.GetBusyApply());
    }

    void SetLock_start() {
        data.SetLock(GetTick());
    }
    void SetLock_stop() {
        data.SetLock(GetTick() - data.GetLock());
    }

    void SetFindToday_start() {
        data.SetFindToday(GetTick());
    }
    void SetFindToday_stop() {
        data.SetFindToday(GetTick() - data.GetFindToday());
    }

    void SetFindYesterday_start() {
        data.SetFindYesterday(GetTick());
    }
    void SetFindYesterday_stop() {
        data.SetFindYesterday(GetTick() - data.GetFindYesterday());
    }

    void SetCreateShingle_start() {
        data.SetCreateShingle(GetTick());
    }
    void SetCreateShingle_stop() {
        data.SetCreateShingle(GetTick() - data.GetCreateShingle());
    }

    void SetAddToRplPool_start() {
        data.SetAddToRplPool(GetTick());
    }
    void SetAddToRplPool_stop() {
        data.SetAddToRplPool(GetTick() - data.GetAddToRplPool());
    }

    void SetFindFollowShingle_start() {
        data.SetFindFollowShingle(GetTick());
    }
    void SetFindFollowShingle_stop() {
        data.SetFindFollowShingle(GetTick() - data.GetFindFollowShingle());
    }

    void SetFindFollowShingle2_start() {
        data.SetFindFollowShingle2(GetTick());
    }
    void SetFindFollowShingle2_stop() {
        data.SetFindFollowShingle2(GetTick() - data.GetFindFollowShingle2());
    }

    void SetFindFollowShingleRpl_start() {
        data.SetFindFollowShingleRpl(GetTick());
    }
    void SetFindFollowShingleRpl_stop() {
        data.SetFindFollowShingleRpl(GetTick() - data.GetFindFollowShingleRpl());
    }

    void SetWriteLogFollowShingle_start() {
        data.SetWriteLogFollowShingle(GetTick());
    }
    void SetWriteLogFollowShingle_stop() {
        data.SetWriteLogFollowShingle(GetTick() - data.GetWriteLogFollowShingle());
    }

    void SetWriteLogFollowShingle2_start() {
        data.SetWriteLogFollowShingle2(GetTick());
    }
    void SetWriteLogFollowShingle2_stop() {
        data.SetWriteLogFollowShingle2(GetTick() - data.GetWriteLogFollowShingle2());
    }

    void SetWriteLogFollowShingleRpl_start() {
        data.SetWriteLogFollowShingleRpl(GetTick());
    }
    void SetWriteLogFollowShingleRpl_stop() {
        data.SetWriteLogFollowShingleRpl(GetTick() - data.GetWriteLogFollowShingleRpl());
    }

    void SetStorageDelay_start() {
        data.SetStorageDelay(GetTick());
    }
    void SetStorageDelay_stop() {
        data.SetStorageDelay(GetTick() - data.GetStorageDelay());
    }

    void AddToAttdata(TObrabShingleDebugInfo& value);
    void AddPrevRqstTiming(TPrevRequestTiming value) {
        data.AddPrevRqstTiming(value);
    }

    TString GetShToLog();
    TString GetStToLog();

    ui32 GetFindToday() {
        return data.GetFindToday();
    }
    ui32 GetFindYesterday() {
        return data.GetFindYesterday();
    }
    ui32 GetCreateShingle() {
        return data.GetCreateShingle();
    }
    ui32 GetLockHeavy() {
        return data.GetLockHeavy();
    }
    ui32 GetFindHeavy() {
        return data.GetFindHeavy();
    }
    ui32 GetAddHeavy() {
        return data.GetAddHeavy();
    }
};

//********************************************************************************************************************************
//                                             TShingleStat - данные по шинглу и сопутствующие
//********************************************************************************************************************************

class TShingleStat {
public:
    enum TShingleForm { TSHUNKNOWN,
                        TSHSHINGLE,
                        TSHSTORAGE };
    static const ui32 YESTERDAY_DIV_KOEF = 86400; //new
                                                  //static const ui32 YESTERDAY_DIV_KOEF = 2073600; //old
public:
    friend IOutputStream& operator<<(IOutputStream& stream, const TShingleStat& stat) {
        return stream << stat.m_type << ':' << Hex(stat.m_shingle);
    }
    TShingleForm m_shform;
    float m_age_koef;
    ui32 m_koef_div;
    ui32 m_last_midnight;
    int m_count;

    //for shingle
    ui64 m_shingle;
    ui16 m_type;
    bool m_empty;
    TShingleDataTE m_shingle_data;
    bool m_heavy;
    TActionType m_exists;
    bool m_exist_today;
    bool m_exist_yesterday;
    ui8 m_today_pr; //признак оптимизации (нужно ли искать данные yesterday), приходит из одиночных данных по today для печати в следилку за шинглом

    //for uniq-shingle
    ui64 m_uniq_shingle;
    int m_uniq_shingle_type;
    bool m_uniq_empty;
    TShingleDataTE m_uniq_shingle_data;
    bool m_uniq_heavy;
    TActionType m_uniq_exists;
    bool m_uniq_exist_today;
    bool m_uniq_exist_yesterday;

    //for storage-shingle
    TString m_st_data;

    //function
    float GetAgeCoeff(time_t last_midnight, ui32 koef_div);
    float Round(float Argument, int Precision) const;
    void ClearShingledata();
    void ClearUniqShingledata();
    int GetCount() const {
        return m_count;
    }

//debug info
#ifdef SOSERV_DEBUG
    TObrabShingleDebugInfo debuginfo;
#endif
public:
    void SetShingleType(TShingleForm shform, ui64 shingle, ui16 type, ui64 uniq_shingle, int uniq_shingle_type);
    void SetShingleData(TShingleDataTE shingle_data, bool heavy, ui32 today_pr);
    void SetAction(TActionType existsA);
    void SetAgeKoef(float age_koef);
    void SetLastMidnight(ui32 lastmidnight);
    void SetEmptyStatus(bool empty);
    void SetStorageData(const TString& stdata);
    void CalcAgeKoef(time_t last_midnight, ui32 koef_div);
    bool CalcHeavyStatus(ui32 heavytreshold);
    TShingleDataTE GetShingleData() {
        return m_shingle_data;
    }
    void SetExistToday(bool exist) {
        m_exist_today = exist;
    }
    void SetExistYesterday(bool exist) {
        m_exist_yesterday = exist;
    }
    bool IsFirstRecord();
    TString UniqShinglePeriod(); //информация для отладки, на каком периоде ищем шингл, чтобы накрутить/ненаркутить уникальный шингл в паре
    TString GetFullAnswer();
    TString GetDebugInfo();
    TString GetFollowInfo();
    TString GetOldAnswer();
    TString GetRequest();
    TString PutRequest(int count, TShingleCount spamtype, TPersType pers);
    void SetUniqShingleData(TShingleStat& uniq_shingle_data);
    float ReCalcKoefDiv(ui32 last_midnight_time, float age_koef);
    TString GetKoef_clt() {
        return "currenttime=" + IntToStroka(time(nullptr)) + ", rcv_lastmidnighttime=" + IntToStroka(m_last_midnight) + ", rcv_koef=" + FloatToStr(AgeKoef()) + ", recalc_koef_div=" + FloatToStr(ReCalcKoefDiv(m_last_midnight, AgeKoef()));
    }

public:
    TShingleStat();
    ~TShingleStat();

    //общие функции
    bool IsValid();
    TShingleForm ShingleForm() const {
        return m_shform;
    }
    TString ShingleFormS();
    void Clear();
    TString ShingleAndTypeS(); //запись всего шингла (с учетом типов и uniq-шинглов) в виде строки
    ui64 ShingleAndTypeAsSummaryShingle() {
        return ShingleFromStroka(ShingleAndTypeS());
    } //шингл от строки, которую вернула ShingleAndTypeS()

    //обычные шинглы (ShingleForm() == TSHSHINGLE)
    TShingleStat(ui64 shingle, ui16 type, int count);                                           //конструктор обычного count-шингла
    TShingleStat(ui64 shingle, ui16 type, ui64 uniq_shingle, int uniq_shingle_type, int count); //конструктор обычного count-шингла с uniq-шинглом
    //алгоритм использования составного шингла (который count-шингл с uniq-шинглом):
    //SHIN1: shingle-type(ip_suid)
    //SHIN2: uniq_shingle-uniq_shingle_type (ip)
    // if (exists(SHIN1))
    // {
    //   SHIN1 += count;
    // }
    // else
    // {
    //   create SHIN1;
    //   SHIN1 += count;
    //   if (!exists SHIN2) create SHIN2;
    //   SHIN2 += 1;
    // }

    float AgeKoef() const {
        return m_age_koef;
    }
    ui32 DivKoef() const {
        return m_koef_div;
    }
    ui32 LastMidnight() const {
        return m_last_midnight;
    }

    ui64 Shingle() const {
        return m_shingle;
    }
    TString ShingleStr() const {
        return ShingleToStroka(m_shingle);
    }
    ui16 Type() const {
        return m_type;
    }
    bool IsEmpty() const {
        return m_empty;
    }

    ui16 TotalHam() const;   //считается с учетом коэфициента m_age_koef, пришедшего с сервера
    ui16 TotalSpam() const;  //считается с учетом коэфициента m_age_koef, пришедшего с сервера
    ui16 TotalMalic() const; //считается с учетом коэфициента m_age_koef, пришедшего с сервера
    ui16 TotalPersHam();     //считается с учетом коэфициента m_age_koef, пришедшего с сервера
    ui16 TotalPersSpam();    //считается с учетом коэфициента m_age_koef, пришедшего с сервера
    float TotalWeight();

    ui16 TotalHam_clt(ui32 koef_div);      //считается с учетом коэфициента, расчитанного здесь в классе; с сервера приходит время последнего миднайта m_last_midnight
    ui16 TotalSpam_clt(ui32 koef_div);     //считается с учетом коэфициента, расчитанного здесь в классе; с сервера приходит время последнего миднайта m_last_midnight
    ui16 TotalMalic_clt(ui32 koef_div);    //считается с учетом коэфициента, расчитанного здесь в классе; с сервера приходит время последнего миднайта m_last_midnight
    ui16 TotalPersHam_clt(ui32 koef_div);  //считается с учетом коэфициента, расчитанного здесь в классе; с сервера приходит время последнего миднайта m_last_midnight
    ui16 TotalPersSpam_clt(ui32 koef_div); //считается с учетом коэфициента, расчитанного здесь в классе; с сервера приходит время последнего миднайта m_last_midnight

    ui16 Ham(TDayType daytype) const {
        return m_shingle_data.Ham(daytype);
    }
    ui16 Malic(TDayType daytype) const {
        return m_shingle_data.Malic(daytype);
    }
    ui16 Spam(TDayType daytype) const {
        return m_shingle_data.Spam(daytype);
    }
    ui16 PersHam(TDayType daytype) const {
        return m_shingle_data.PersHam(daytype);
    }
    ui16 PersSpam(TDayType daytype) const {
        return m_shingle_data.PersSpam(daytype);
    }
    float Weight(TDayType daytype) const {
        return m_shingle_data.Weight(daytype);
    }

    bool IsHeavyShingle() const {
        return m_heavy;
    }
    TActionType GetActionState() const {
        return m_exists;
    }
    TString GetActionStateS() const;
    TString GetActionStateShortS() const;

    ui64 Shingle_uniq() const {
        return m_uniq_shingle;
    }
    TString ShingleStr_uniq() const {
        return ShingleToStroka(m_uniq_shingle);
    }
    ui16 Type_uniq() const {
        return m_uniq_shingle_type;
    }
    bool IsEmpty_uniq() const {
        return m_uniq_empty;
    }

    ui16 TotalHam_uniq();
    ui16 TotalSpam_uniq();
    ui16 TotalMalic_uniq();
    ui16 TotalPersHam_uniq();
    ui16 TotalPersSpam_uniq();
    float TotalWeight_uniq();

    ui16 TotalHam_uniq_clt(ui32 koef_div);
    ui16 TotalSpam_uniq_clt(ui32 koef_div);
    ui16 TotalMalic_uniq_clt(ui32 koef_div);
    ui16 TotalPersHam_uniq_clt(ui32 koef_div);
    ui16 TotalPersSpam_uniq_clt(ui32 koef_div);

    ui16 Ham_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.Ham(daytype);
    }
    ui16 Malic_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.Malic(daytype);
    }
    ui16 Spam_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.Spam(daytype);
    }
    ui16 PersHam_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.PersHam(daytype);
    }
    ui16 PersSpam_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.PersSpam(daytype);
    }
    float Weight_uniq(TDayType daytype) const {
        return m_uniq_shingle_data.Weight(daytype);
    }

    bool IsHeavyShingle_uniq() const {
        return m_uniq_heavy;
    }
    TActionType GetActionState_uniq() const {
        return m_uniq_exists;
    }
    TString GetActionStateS_uniq() const;
    TString GetActionStateShortS_uniq() const;

    //шинглы storage (ShingleForm() == TSHSTORAGE)
    TShingleStat(ui64 shingle); //конструктор для создания шингла типа storage

    ui64 ShingleST() {
        return Shingle();
    }
    TString ShingleSTStr() {
        return ShingleStr();
    }
    bool IsEmptyST() {
        return IsEmpty();
    }
    const TString& GetSTData() const {
        return m_st_data;
    }

    bool operator<(const TShingleStat& arg) const;

    NJson::TJsonValue toJson() const;
};

typedef TList<TShingleStat> TShingleStatList;
typedef TShingleStatList::iterator TShingleStatListIt;

//********************************************************************************************************************************
//                                           TShingleStatParser
//********************************************************************************************************************************

namespace NShingleStatParser {
    TString GetServerIDFromHeaders(const TString& responce);
    TString GetServerIDFromBody(const TString& responce);
    TString GetRequestIDFromHeaders(const TString& responce);
    bool ParseShinglesData(const TString& responce, TShingleStatList& shingle_list);
    bool GetDataByIdent(const char* start, const TString& Ident, TString& resdata);

    ui32 GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size);
}

//********************************************************************************************************************************

#endif
