#include "functional_clients/FreemailShClient.h"
#include <library/cpp/testing/unittest/registar.h>

using namespace NFuncClient;

Y_UNIT_TEST_SUITE(FreemailClient) {
    NCurl::TCurl curl;
    Y_UNIT_TEST(Empty) {
        CFreemailLogic client;

        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID("23"), "RU", NFreeMail::SPAM, { NFreeMail::TEmailInfo::UUID("16"), NFreeMail::TEmailInfo::UUID("17") }), RES_OK);
        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID("23"), NFreeMail::TEmailInfo::External("other"), NFreeMail::SPAM), RES_OK);

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);
        UNIT_ASSERT_EQUAL(info.Defined(), false);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        TMaybe<NFreeMail::TPDDInfo> pdd;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", pdd), RES_OK);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);
    }

    using TErrorClient = UT::TErrorClient<CFreemailLogic>;
    Y_UNIT_TEST(Error) {
        TErrorClient client;

        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID("23"), "RU", NFreeMail::SPAM, { NFreeMail::TEmailInfo::UUID("16"), NFreeMail::TEmailInfo::UUID("17") }), RES_HTTP_ERROR);
        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID("23"), NFreeMail::TEmailInfo::External("other"), NFreeMail::SPAM), RES_HTTP_ERROR);

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_BAD_RESPONSE);

        TMaybe<NFreeMail::TPDDInfo> pdd;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", pdd), RES_BAD_RESPONSE);
    }

    Y_UNIT_TEST(BadParams) {
        TErrorClient client;
        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID("23"), "RU", NFreeMail::UNKNOWN, { NFreeMail::TEmailInfo::UUID("16"), NFreeMail::TEmailInfo::UUID("17") }), RES_BAD_PARAMS);
        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID("23"), NFreeMail::TEmailInfo::External("other"), NFreeMail::UNKNOWN), RES_BAD_PARAMS);

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::External("external"), info, bounce), RES_BAD_PARAMS);
    }

    Y_UNIT_TEST(WrongKey) {
        TErrorClient client;

        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID(nullptr), "RU", NFreeMail::SPAM, { NFreeMail::TEmailInfo::UUID("16"), NFreeMail::TEmailInfo::UUID("17") }), RES_BAD_PARAMS);
        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID(nullptr), NFreeMail::TEmailInfo::External("other"), NFreeMail::SPAM), RES_BAD_PARAMS);

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID(nullptr), info, bounce), RES_BAD_PARAMS);
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::External("other"), info, bounce), RES_BAD_PARAMS);

        TMaybe<NFreeMail::TPDDInfo> pdd;
        UNIT_ASSERT_EQUAL(client.Get(curl, nullptr, pdd), RES_BAD_PARAMS);
    }

    using TRequestClient = UT::TRequestClient<CFreemailLogic>;
    Y_UNIT_TEST(PutRequest) {
        TRequestClient client;

        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID("23"), "RU", NFreeMail::SPAM, { NFreeMail::TEmailInfo::UUID("16"), NFreeMail::TEmailInfo::UUID("17") }), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["time"],"fields":[{"uuid":590699460983228292}]},{"type":"Update","scheme":["counters"],"fields":[{"send_spam":2,"recepients_count":2,"uuid":590699460983228292}]},{"type":"Update","scheme":["active"],"fields":[{"active_days":1,"uuid":590699460983228292}]},{"type":"Update","scheme":["geo"],"fields":[{"geo":"RU","uuid":590699460983228292}]},{"type":"Update","scheme":["recepients_max"],"fields":[{"recepients_count":0,"uuid":590699460983228292,"recepients_max_count":{"f1":"recepients_count","f2":"recepients_max_count"}}]},{"type":"Update","scheme":["counters"],"fields":[{"receive_spam":1,"uuid":590700560494856540},{"receive_spam":1,"uuid":590700560494856541}]}])request");

        UNIT_ASSERT_EQUAL(client.Put(curl, NFreeMail::TEmailInfo::UUID("42"), nullptr, NFreeMail::HAM, { NFreeMail::TEmailInfo::External("13"), NFreeMail::TEmailInfo::UUID(nullptr) }), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["time"],"fields":[{"uuid":590697261959971939}]},{"type":"Update","scheme":["counters"],"fields":[{"send_ham":2,"recepients_count":2,"uuid":590697261959971939}]},{"type":"Update","scheme":["active"],"fields":[{"active_days":1,"uuid":590697261959971939}]},{"type":"Update","scheme":["recepients_max"],"fields":[{"recepients_count":0,"uuid":590697261959971939,"recepients_max_count":{"f1":"recepients_count","f2":"recepients_max_count"}}]}])request");
    }

    Y_UNIT_TEST(ComplaintRequest) {
        TRequestClient client;

        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID("23"), NFreeMail::TEmailInfo::External("other"), NFreeMail::SPAM), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["time"],"fields":[{"uuid":590699460983228292}]},{"type":"Update","scheme":["counters"],"fields":[{"complaint_spam":1,"uuid":590699460983228292}]},{"type":"Update","scheme":["complaint"],"fields":[{"hash":8918463269163405440,"spam":true,"uuid":590699460983228292}]}])request");

        UNIT_ASSERT_EQUAL(client.Complaint(curl, NFreeMail::TEmailInfo::UUID("42"), NFreeMail::TEmailInfo::UUID("13"), NFreeMail::HAM), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["time"],"fields":[{"uuid":590697261959971939}]},{"type":"Update","scheme":["counters"],"fields":[{"uuid":590697261959971939,"complaint_ham":1}]},{"type":"Update","scheme":["complaint"],"fields":[{"hash":590700560494856537,"spam":false,"uuid":590697261959971939}]}])request");
    }

    Y_UNIT_TEST(GetRequest) {
        TRequestClient client;

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Get","scheme":["counters_get"],"fields":[{"date":6,"uuid":590699460983228292}]},{"type":"Get","scheme":["geo_get"],"fields":[{"date":30,"uuid":590699460983228292}]},{"type":"Get","scheme":["time_get"],"fields":[{"date":90,"uuid":590699460983228292}]},{"type":"Get","scheme":["complaint_get"],"fields":[{"date":90,"uuid":590699460983228292}]},{"type":"Get","scheme":["bounce_get"],"fields":[{"date":90,"uuid":23}]},{"type":"Get","scheme":["current"],"fields":[]}])request");

        TMaybe<NFreeMail::TPDDInfo> pdd;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", pdd), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Get","scheme":["pdd"],"fields":[{"request":"so/domaininfo/","param":{"domain":"yandex.ru"}}]},{"type":"Get","scheme":["domain"],"fields":[{"domain":"yandex.ru"}]},{"type":"Get","scheme":["domain_all"],"fields":[{"domain":"yandex.ru"},{"date":90}]}])request");
    }

    using TGetClient = UT::TGetClient<CFreemailLogic>;
    Y_UNIT_TEST(GetEmpty) {
        TGetClient client("[]");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        TMaybe<NFreeMail::TPDDInfo> pdd;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", pdd), RES_OK);

        UNIT_ASSERT_EQUAL(pdd.Defined(), false);
        UNIT_ASSERT_EQUAL(info.Defined(), false);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);
    }

    Y_UNIT_TEST(GetTime) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "create_date": 6599,
                        "date": 6604
                    }
                ],
                "scheme": "time_get"
            },
            {
                "find": [
                    {
                        "days": 6660
                    }
                ],
                "scheme": "current"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->get_time, 6660);
        UNIT_ASSERT_EQUAL(info->create_time, 6599);
        UNIT_ASSERT_EQUAL(info->update_time, 6604);
        UNIT_ASSERT_EQUAL(info->send.ham, 0);
        UNIT_ASSERT_EQUAL(info->send.spam, 0);
        UNIT_ASSERT_EQUAL(info->receive.ham, 0);
        UNIT_ASSERT_EQUAL(info->receive.spam, 0);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 0);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 0);
        UNIT_ASSERT_EQUAL(info->active_days, 0);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 0);
    }

    Y_UNIT_TEST(GetCounters) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "send_spam": 40,
                        "send_ham": 17,
                        "receive_ham": 18,
                        "receive_spam": 19,
                        "complaint_ham": 16,
                        "complaint_spam": 61,
                        "recepients_count": 134,
                        "recepients_max_count": 111
                    }
                ],
                "scheme": "counters_get"
            },
            {
                "find": [{}],
                "scheme": "time_get"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->create_time, 0);
        UNIT_ASSERT_EQUAL(info->update_time, 0);
        UNIT_ASSERT_EQUAL(info->send.ham, 17);
        UNIT_ASSERT_EQUAL(info->send.spam, 40);
        UNIT_ASSERT_EQUAL(info->receive.ham, 18);
        UNIT_ASSERT_EQUAL(info->receive.spam, 19);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 16);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 61);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 0);
        UNIT_ASSERT_EQUAL(info->active_days, 0);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 134);
    }

    Y_UNIT_TEST(GetCounters2) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "send_spam": 40,
                        "send_ham": 17,
                        "receive_ham": 18,
                        "receive_spam": 19,
                        "complaint_ham": 16,
                        "complaint_spam": 61,
                        "recepients_count": 134,
                        "recepients_max_count": 111
                    },
                    {
                        "send_spam": 40,
                        "send_ham": 17,
                        "receive_ham": 18,
                        "receive_spam": 19,
                        "complaint_ham": 16,
                        "complaint_spam": 61,
                        "recepients_count": 0,
                        "recepients_max_count": 234
                    }
                ],
                "scheme": "counters_get"
            },
            {
                "find": [{}],
                "scheme": "time_get"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->create_time, 0);
        UNIT_ASSERT_EQUAL(info->update_time, 0);
        UNIT_ASSERT_EQUAL(info->send.ham, 34);
        UNIT_ASSERT_EQUAL(info->send.spam, 80);
        UNIT_ASSERT_EQUAL(info->receive.ham, 36);
        UNIT_ASSERT_EQUAL(info->receive.spam, 38);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 32);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 122);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 0);
        UNIT_ASSERT_EQUAL(info->active_days, 0);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 234);
    }

    Y_UNIT_TEST(GetComplaints) {
        TGetClient client(R"in(
            [{
                "find": [
                    { "spam": 1 },
                    { "spam": 0 },
                    { "spam": 1 }
                ],
                "scheme": "complaint_get"
            },
            {
                "find": [{}],
                "scheme": "time_get"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->create_time, 0);
        UNIT_ASSERT_EQUAL(info->update_time, 0);
        UNIT_ASSERT_EQUAL(info->send.ham, 0);
        UNIT_ASSERT_EQUAL(info->send.spam, 0);
        UNIT_ASSERT_EQUAL(info->receive.ham, 0);
        UNIT_ASSERT_EQUAL(info->receive.spam, 0);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 0);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 1);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 2);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 0);
        UNIT_ASSERT_EQUAL(info->active_days, 0);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 0);
    }

    Y_UNIT_TEST(GetGeo) {
        TGetClient client(R"in(
            [{
                "find": [{}, {}],
                "scheme": "geo_get"
            },
            {
                "find": [{}],
                "scheme": "time_get"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("23"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->create_time, 0);
        UNIT_ASSERT_EQUAL(info->update_time, 0);
        UNIT_ASSERT_EQUAL(info->send.ham, 0);
        UNIT_ASSERT_EQUAL(info->send.spam, 0);
        UNIT_ASSERT_EQUAL(info->receive.ham, 0);
        UNIT_ASSERT_EQUAL(info->receive.spam, 0);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 0);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 2);
        UNIT_ASSERT_EQUAL(info->active_days, 0);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 0);
    }

    Y_UNIT_TEST(GetActiveDay) {
        TGetClient client(R"in(
            [{
                "find": [
                    { "active_days": 4  }
                ],
                "scheme": "time_get"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("32"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(bounce.Defined(), false);

        UNIT_ASSERT_EQUAL(info->create_time, 0);
        UNIT_ASSERT_EQUAL(info->update_time, 0);
        UNIT_ASSERT_EQUAL(info->send.ham, 0);
        UNIT_ASSERT_EQUAL(info->send.spam, 0);
        UNIT_ASSERT_EQUAL(info->receive.ham, 0);
        UNIT_ASSERT_EQUAL(info->receive.spam, 0);
        UNIT_ASSERT_EQUAL(info->complaint.ham, 0);
        UNIT_ASSERT_EQUAL(info->complaint.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.ham, 0);
        UNIT_ASSERT_EQUAL(info->unique_complaint_user.spam, 0);
        UNIT_ASSERT_EQUAL(info->unique_geo_zone, 0);
        UNIT_ASSERT_EQUAL(info->active_days, 4);
        UNIT_ASSERT_EQUAL(info->max_recipients_count, 0);
    }

    Y_UNIT_TEST(GetBounce) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "date": 6660,
                        "today_spam": 23,
                        "today_unknown": 42,
                        "total_spam": 123,
                        "total_unknown": 842,
                        "last_type": 7
                    }
                ],
                "scheme": "bounce_get"
            },
            {
                "find": [
                    {
                        "days": 6666
                    }
                ],
                "scheme": "current"
            }]
        )in");

        TMaybe<NFreeMail::TInfo> info;
        TMaybe<NFreeMail::TBounceInfo> bounce;
        UNIT_ASSERT_EQUAL(client.Get(curl, NFreeMail::TEmailInfo::UUID("32"), info, bounce), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), false);
        UNIT_ASSERT_EQUAL(bounce.Defined(), true);

        UNIT_ASSERT_EQUAL(bounce->spam.today, 23);
        UNIT_ASSERT_EQUAL(bounce->spam.total, 123);
        UNIT_ASSERT_EQUAL(bounce->unknown.today, 42);
        UNIT_ASSERT_EQUAL(bounce->unknown.total, 842);
        UNIT_ASSERT_EQUAL(bounce->last_date, 6660);
        UNIT_ASSERT_EQUAL(bounce->get_date, 6666);
        UNIT_ASSERT_EQUAL(bounce->last_type, 7);
    }

    Y_UNIT_TEST(GetPDDBounceTotal) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "domain": "yandex.ru",
                        "count": 14
                    },
                    { "count": 13 },
                    { "count": 97 }
                ],
                "scheme": "domain_all"
            },
            {
                "find": [
                    {
                        "data":"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n<pddinfo>\n</pddinfo>",
                        "code":200
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(info->karma, 0);
        UNIT_ASSERT_EQUAL(info->first_time, 0);
        UNIT_ASSERT_EQUAL(info->mailbox_count, 0);
        UNIT_ASSERT_EQUAL(info->bounce.today, 0);
        UNIT_ASSERT_EQUAL(info->bounce.total, 124);
        UNIT_ASSERT_EQUAL(info->admin_uid, "");
        UNIT_ASSERT_EQUAL(info->host, "yandex.ru");
        UNIT_ASSERT_EQUAL(info->ip.Undefined(), true);
    }

    Y_UNIT_TEST(GetPDDBounceToday) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "domain": "yandex.ru",
                        "count": 14
                    },
                    { "count": 13 },
                    { "count": 97 }
                ],
                "scheme": "domain_all"
            },
            {
                "find": [
                    { "count": 14 }
                ],
                "scheme": "domain"
            },
            {
                "find": [
                    {
                        "data":"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n<pddinfo>\n</pddinfo>",
                        "code":200
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(info->karma, 0);
        UNIT_ASSERT_EQUAL(info->first_time, 0);
        UNIT_ASSERT_EQUAL(info->mailbox_count, 0);
        UNIT_ASSERT_EQUAL(info->bounce.today, 14);
        UNIT_ASSERT_EQUAL(info->bounce.total, 124);
        UNIT_ASSERT_EQUAL(info->admin_uid, "");
        UNIT_ASSERT_EQUAL(info->host, "yandex.ru");
        UNIT_ASSERT_EQUAL(info->ip.Undefined(), true);
    }

    Y_UNIT_TEST(GetPDDError) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "data":"",
                        "code":400
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), false);
    }

    Y_UNIT_TEST(GetPDDNotFound) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "data":"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n<pddinfo>\n\n    <error>bad_domain</error>\n\n</pddinfo>",
                        "code":200
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), false);
    }

    Y_UNIT_TEST(GetPDDXml) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "data":"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n<pddinfo>\n\n    <geo>ru</geo>\n    <firsttime>1409735828.0</firsttime>\n    <mailboxcnt>17</mailboxcnt>\n    <workspace>0</workspace>\n    <freereg>1</freereg>\n    <admlogin>yandex.login</admlogin>\n    <quota>1000</quota>\n    <manualreq>1</manualreq>\n    <karma>50</karma>\n    <ip>81.23.188.51</ip>\n\n</pddinfo>",
                        "code":200
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(info->karma, 50);
        UNIT_ASSERT_EQUAL(info->first_time, 1409735828);
        UNIT_ASSERT_EQUAL(info->mailbox_count, 17);
        UNIT_ASSERT_EQUAL(info->bounce.today, 0);
        UNIT_ASSERT_EQUAL(info->bounce.total, 0);
        UNIT_ASSERT_EQUAL(info->admin_uid, "yandex.login");
        UNIT_ASSERT_EQUAL(info->host, "");
        UNIT_ASSERT_EQUAL(info->ip, TKIPv6("81.23.188.51"));
    }

    Y_UNIT_TEST(GetPDDXmlEmpty) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "data":"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<pddinfo>\n<geo>ru</geo>\n<firsttime></firsttime>\n<mailboxcnt>16677</mailboxcnt>\n<workspace>0</workspace>\n<freereg>0</freereg>\n<admlogin>alexyudoc</admlogin>\n<quota>17000</quota>\n<manualreq>1</manualreq>\n<karma>50</karma>\n<ip>217.14.197.200</ip>\n</pddinfo>",
                        "code":200
                    }
                ],
                "scheme":"pdd"
            }]
        )in");

        TMaybe<NFreeMail::TPDDInfo> info;
        UNIT_ASSERT_EQUAL(client.Get(curl, "yandex.ru", info), RES_OK);

        UNIT_ASSERT_EQUAL(info.Defined(), true);
        UNIT_ASSERT_EQUAL(info->karma, 50);
        UNIT_ASSERT_EQUAL(info->first_time, 0);
        UNIT_ASSERT_EQUAL_C(info->mailbox_count, 16677, info->mailbox_count);
        UNIT_ASSERT_EQUAL(info->bounce.today, 0);
        UNIT_ASSERT_EQUAL(info->bounce.total, 0);
        UNIT_ASSERT_EQUAL(info->admin_uid, "alexyudoc");
        UNIT_ASSERT_EQUAL(info->host, "");
        UNIT_ASSERT_EQUAL(info->ip, TKIPv6("217.14.197.200"));
    }
}
