#include "functional_clients/UrlReputationShClient.h"
#include <library/cpp/testing/unittest/registar.h>

using namespace NFuncClient;

Y_UNIT_TEST_SUITE(UrlReputation) {
    NCurl::TCurl curl;
    Y_UNIT_TEST(Empty) {
        TUrlStatisticVector dummy;
        CUrlReputationlLogic client;

        UNIT_ASSERT_EQUAL(client.Check(curl, dummy, true), RES_OK);
        UNIT_ASSERT_EQUAL(client.Complaint(curl, dummy, true), RES_OK);
    }

    using TRequestClient = UT::TRequestClient<CUrlReputationlLogic>;
    Y_UNIT_TEST(EmptyRequest) {
        TRequestClient client;
        TUrlStatisticVector dummy;

        UNIT_ASSERT_EQUAL(client.Get(curl, dummy), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), nullptr);

        UNIT_ASSERT_EQUAL(client.Put(curl, dummy), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), nullptr);

        UNIT_ASSERT_EQUAL(client.Resolve(curl, dummy), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), nullptr);

        UNIT_ASSERT_EQUAL(client.Complaint(curl, dummy, true), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), nullptr);
    }

    static TUrlStatisticVector GetUrlsReal() {
        TUrlStatisticVector urls;
        urls.emplace_back("httpS://wWw.yandex.РФ", CODES_UTF8, false, false);
        urls.emplace_back("goo.gl/dfgU", CODES_UTF8, true, false);
        urls.back().AddLongUrl("mail.ya.ru/main/index.php", CODES_UTF8);
        urls.back().aliases.emplace_back("surbl");

        urls.front().aliases.emplace_back("dbl");
        urls.front().aliases.emplace_back("malware");

        return urls;
    }

    Y_UNIT_TEST(GetRequest) {
        TRequestClient client;
        TUrlStatisticVector urls = GetUrlsReal();

        UNIT_ASSERT_EQUAL(client.Get(curl, urls), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Get","scheme":["today"],"fields":[{"shingle":11242453612388259935},{"shingle":7480385171926061069}]},{"type":"Get","scheme":["history"],"fields":[{"shingle":11242453612388259935},{"shingle":7480385171926061069}]},{"type":"Get","scheme":["current"],"fields":[]},{"type":"Get","scheme":["malware"],"fields":[{"value":15740162488345465691},{"value":3722895317771385609},{"value":2536124716374566878},{"value":12535445584751298168},{"value":10392907247899004611},{"value":13246343342814705890},{"value":9760517899542623176},{"value":3002720841138180907},{"value":17582265190206238974}]},{"type":"Get","scheme":["dns"],"fields":[{"alias":"surbl","host":"goo.gl"},{"alias":"dbl","host":"yandex.xn--p1ai"},{"alias":"malware","host":"yandex.xn--p1ai"},{"alias":"surbl","host":"mail.ya.ru"}]}])request");
    }

    static TUrlStatisticVector GetUrls(bool link = true) {
        TUrlStatisticVector urls;
        urls.emplace_back("short", CODES_UTF8, link, false); //3860673601219843277
        if (link)
            urls.back().AddLongUrl("long", CODES_UTF8); //15016842284610189427
        else
            urls.emplace_back("long", CODES_UTF8, false, false);
        return urls;
    }

    Y_UNIT_TEST(ComplaintRequest) {
        TRequestClient client;
        TUrlStatisticVector urls = GetUrls();

        UNIT_ASSERT_EQUAL(client.Complaint(curl, urls, true), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["today"],"fields":[{"complaint_spam":1,"shingle":15016842284610189427}]},{"type":"Update","scheme":["history"],"fields":[{"complaint_spam":1,"shingle":15016842284610189427}]}])request");
        UNIT_ASSERT_EQUAL(client.Complaint(curl, urls, false), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["today"],"fields":[{"shingle":15016842284610189427,"complaint_ham":1}]},{"type":"Update","scheme":["history"],"fields":[{"shingle":15016842284610189427,"complaint_ham":1}]}])request");
    }

    Y_UNIT_TEST(PutRequest) {
        TRequestClient client;
        TUrlStatisticVector urls;

        urls.emplace_back("long", CODES_UTF8, false, true);
        UNIT_ASSERT_EQUAL(client.Put(curl, urls), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["today"],"fields":[{"spam":1,"shingle":15016842284610189427}]},{"type":"Update","scheme":["history"],"fields":[{"spam":1,"shingle":15016842284610189427}]}])request");

        TVector<TString> dummy;
        urls.back() = TUrlStatistic("long", CODES_UTF8, true, true, false, dummy, 0);
        UNIT_ASSERT_EQUAL(client.Put(curl, urls), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Update","scheme":["today"],"fields":[{"spam":1,"shingle":15016842284610189427}]},{"type":"Update","scheme":["history"],"fields":[{"spam":1,"shingle":15016842284610189427}]}])request");
    }

    Y_UNIT_TEST(ResolveRequest) {
        TRequestClient client;
        TUrlStatisticVector urls = GetUrls();
        urls.emplace_back("url", CODES_UTF8, true, false);
        urls.emplace_back("other", CODES_UTF8, true, false);
        urls.emplace_back("other", CODES_UTF8, false, false);

        UNIT_ASSERT_EQUAL(client.Resolve(curl, urls), RES_OK);
        UNIT_ASSERT_EQUAL(client.GetRequestData(), R"request([{"type":"Get","scheme":["resolve"],"fields":[{"url":"short"},{"url":"url"},{"url":"other"}]}])request");
    }

    static void CheckZero(const TUrlCounters& info) {
        UNIT_ASSERT_EQUAL(info.ham, 0);
        UNIT_ASSERT_EQUAL(info.spam, 0);
        UNIT_ASSERT_EQUAL(info.complaint_ham, 0);
        UNIT_ASSERT_EQUAL(info.complaint_spam, 0);
    }

    using TGetClient = UT::TGetClient<CUrlReputationlLogic>;
    Y_UNIT_TEST(GetEmpty) {
        TGetClient client("[]");

        TUrlStatisticVector urls = GetUrls();

        UNIT_ASSERT_EQUAL(client.Get(curl, urls), RES_OK);
        CheckZero(urls.front().longUrl.today);
        CheckZero(urls.front().longUrl.history);
        CheckZero(urls.front().checkUrl.today);
        CheckZero(urls.front().checkUrl.history);
    }

    Y_UNIT_TEST(GetData) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "shingle": 15016842284610189427,
                        "ham": 1,
                        "spam": 2,
                        "complaint_ham": 8,
                        "complaint_spam": 9
                    },
                    {
                        "shingle": 3860673601219843277,
                        "ham": 10,
                        "spam": 9,
                        "complaint_ham": 3,
                        "complaint_spam": 2
                    }
                ],
                "scheme": "today"
            },
            {
                "find": [
                    {
                        "shingle": 3860673601219843277,
                        "create_date": 5500,
                        "date": 5550,
                        "ham": 10,
                        "spam": 20,
                        "complaint_ham": 80,
                        "complaint_spam": 90
                    }
                ],
                "scheme": "history"
            },
            {
                "find": [
                    {
                        "shingle": 15016842284610189427,
                        "create_date": 6600,
                        "date": 6660,
                        "ham": 100,
                        "spam": 90,
                        "complaint_ham": 30,
                        "complaint_spam": 20
                    }
                ],
                "scheme": "history"
            },
            {
                "find": [
                    {
                        "days": 6660
                    }
                ],
                "scheme": "current"
            }]
        )in");

        TUrlStatisticVector urls = GetUrls(false);
        UNIT_ASSERT_EQUAL(client.Get(curl, urls), RES_OK);

        const TUrlInfo& check = urls.front().checkUrl;
        UNIT_ASSERT_EQUAL(check.today.ham, 10);
        UNIT_ASSERT_EQUAL(check.today.spam, 9);
        UNIT_ASSERT_EQUAL(check.today.complaint_ham, 3);
        UNIT_ASSERT_EQUAL(check.today.complaint_spam, 2);

        UNIT_ASSERT_EQUAL(check.last_date, 5550);
        UNIT_ASSERT_EQUAL(check.first_date, 5500);
        UNIT_ASSERT_EQUAL(check.current_date, 6660);
        UNIT_ASSERT_EQUAL(check.history.ham, 10);
        UNIT_ASSERT_EQUAL(check.history.spam, 20);
        UNIT_ASSERT_EQUAL(check.history.complaint_ham, 80);
        UNIT_ASSERT_EQUAL(check.history.complaint_spam, 90);

        const TUrlInfo& longurl = urls.back().checkUrl;
        UNIT_ASSERT_EQUAL(longurl.today.ham, 1);
        UNIT_ASSERT_EQUAL(longurl.today.spam, 2);
        UNIT_ASSERT_EQUAL(longurl.today.complaint_ham, 8);
        UNIT_ASSERT_EQUAL(longurl.today.complaint_spam, 9);

        UNIT_ASSERT_EQUAL(longurl.last_date, 6660);
        UNIT_ASSERT_EQUAL(longurl.first_date, 6600);
        UNIT_ASSERT_EQUAL(longurl.current_date, 6660);
        UNIT_ASSERT_EQUAL(longurl.history.ham, 100);
        UNIT_ASSERT_EQUAL(longurl.history.spam, 90);
        UNIT_ASSERT_EQUAL(longurl.history.complaint_ham, 30);
        UNIT_ASSERT_EQUAL(longurl.history.complaint_spam, 20);
    }

    Y_UNIT_TEST(GetResolve) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "url": "url",
                        "resolved": "resolved"
                    },
                    {
                        "url": "other",
                        "resolved": "https://www.ya.ru/"
                    }
                ],
                "scheme": "resolve"
            }]
        )in");

        TUrlStatisticVector urls = GetUrls();
        urls.emplace_back("other", CODES_UTF8, true, false);
        urls.emplace_back("url", CODES_UTF8, true, false);
        urls.emplace_back("other", CODES_UTF8, true, false);
        UNIT_ASSERT_EQUAL(client.Resolve(curl, urls), RES_OK);

        UNIT_ASSERT_EQUAL(urls[0].longUrl.url, "long");
        UNIT_ASSERT_EQUAL(urls[1].longUrl.url, "ya.ru");
        UNIT_ASSERT_EQUAL(urls[2].longUrl.url, "resolved");
        UNIT_ASSERT_EQUAL(urls[3].longUrl.url, "ya.ru");
    }

    Y_UNIT_TEST(GetMalware) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "value": 2536124716374566878,
                        "tags": ["cbb77"]
                    },
                    {
                        "value": 3002720841138180907,
                        "tags": ["other", "fraud.phishing", "malware.malware"]
                    }
                ],
                "scheme": "malware"
            }]
        )in");

        TUrlStatisticVector urls = GetUrlsReal();
        UNIT_ASSERT_EQUAL(client.Get(curl, urls), RES_OK);

        UNIT_ASSERT_EQUAL(urls[0].checkUrl.IsPhishingCBB(), true);
        UNIT_ASSERT_EQUAL(urls[0].checkUrl.IsPhishingVDirect(), false);
        UNIT_ASSERT_EQUAL(urls[0].checkUrl.virus, false);

        UNIT_ASSERT_EQUAL(urls[1].longUrl.IsPhishingCBB(), false);
        UNIT_ASSERT_EQUAL(urls[1].longUrl.IsPhishingVDirect(), true);
        UNIT_ASSERT_EQUAL(urls[1].longUrl.virus, true);
    }

    Y_UNIT_TEST(RBLDNS) {
        TGetClient client(R"in(
            [{
                "find": [
                    {
                        "host": "yandex.xn--p1ai",
                        "alias": "dbl",
                        "resolved": ["127.0.0.64"]
                    },
                    {
                        "host": "mail.ya.ru",
                        "alias": "surbl",
                        "resolved": ["127.0.0.32", "127.0.0.16"]
                    }
                ],
                "scheme": "dns"
            }]
        )in");

        TUrlStatisticVector urls = GetUrlsReal();
        UNIT_ASSERT_EQUAL(client.Get(curl, urls), RES_OK);

        UNIT_ASSERT_EQUAL(urls[0].checkUrl.aliases.size(), 1);
        UNIT_ASSERT_EQUAL(urls[1].longUrl.aliases.size(), 1);

        UNIT_ASSERT_EQUAL(urls[0].checkUrl.aliases["dbl"], 64);
        UNIT_ASSERT_EQUAL(urls[1].longUrl.aliases["surbl"], 32);

        UNIT_ASSERT_EQUAL(urls[0].checkUrl.server_surbl, 0);
        UNIT_ASSERT_EQUAL(urls[1].checkUrl.server_surbl, 0);
        UNIT_ASSERT_EQUAL(urls[1].longUrl.server_surbl, 32);
    }
}
