#include <library/cpp/testing/unittest/registar.h>
#include <util/random/random.h>

#include "sender/tsenderrepenv2.h"

using namespace NSenderReputation;

Y_UNIT_TEST_SUITE(SendersClient) {
    Y_UNIT_TEST(TestEmail) {
        {
            TEmail email("test@yandex");

            UNIT_ASSERT_EQUAL_C(email.Email(), "test@yandex", email.Email());
            UNIT_ASSERT_EQUAL_C(email.Domain().Defined(), true, email.Domain());
            UNIT_ASSERT_EQUAL(*email.Domain(), "yandex");
        }
        {
            TEmail email("test_yandex");

            UNIT_ASSERT_EQUAL_C(email.Domain().Defined(), false, email.Domain());
        }
        {
            TEmail email("test_yandex@");
            UNIT_ASSERT_EQUAL_C(email.Domain().Defined(), false, email.Domain());
        }
        {
            TEmail email("");
            UNIT_ASSERT_EQUAL_C(email.Email(), TString(""), email.Email());
            UNIT_ASSERT_EQUAL_C(email.Domain().Defined(), false, email.Domain());
        }
    }
    Y_UNIT_TEST(TestShingle) {

        {
            TShingle shingle("test@yandex", TSenderBitset(TSenderType::Sender));

            UNIT_ASSERT_EQUAL_C(shingle.ShingleEmail().Defined(), true, shingle.ShingleEmail());
            UNIT_ASSERT_EQUAL_C(shingle.ShingleDomain().Defined(), false, shingle.ShingleDomain());
            UNIT_ASSERT_EQUAL_C(shingle.ShinglePSndr().Defined(), false, shingle.ShinglePSndr());
        }
        {
            TShingle shingle("test@yandex", TSenderBitset(TSenderType::Sender, TSenderType::Domain));

            UNIT_ASSERT_EQUAL_C(shingle.ShingleEmail().Defined(), true, shingle.ShingleEmail());
            UNIT_ASSERT_EQUAL_C(shingle.ShingleDomain().Defined(), true, shingle.ShingleDomain());
            UNIT_ASSERT_EQUAL_C(shingle.ShinglePSndr().Defined(), false, shingle.ShinglePSndr());
        }
        {
            TShingle shingle("test@yandex", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));

            UNIT_ASSERT_EQUAL_C(shingle.ShingleEmail().Defined(), true, shingle.ShingleEmail());
            UNIT_ASSERT_EQUAL_C(shingle.ShingleDomain().Defined(), true, shingle.ShingleDomain());
            UNIT_ASSERT_EQUAL_C(shingle.ShinglePSndr().Defined(), false, shingle.ShinglePSndr());
        }
        {
            TShingle shingle("test@yandex", "paysender", TSenderBitset(TSenderType::Sender, TSenderType::Domain));

            UNIT_ASSERT_EQUAL_C(shingle.ShingleEmail().Defined(), true, shingle.ShingleEmail());
            UNIT_ASSERT_EQUAL_C(shingle.ShingleDomain().Defined(), true, shingle.ShingleDomain());
            UNIT_ASSERT_EQUAL_C(shingle.ShinglePSndr().Defined(), false, shingle.ShinglePSndr());
        }
        {
            TShingle shingle("test@yandex", "paysender", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));

            UNIT_ASSERT_EQUAL_C(shingle.ShingleEmail().Defined(), true, shingle.ShingleEmail());
            UNIT_ASSERT_EQUAL_C(shingle.ShingleDomain().Defined(), true, shingle.ShingleDomain());
            UNIT_ASSERT_EQUAL_C(shingle.ShinglePSndr().Defined(), true, shingle.ShinglePSndr());
        }
        {
            TShingle shingle1("teSt@yAndex", "paysender", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));
            TShingle shingle2("tesT@yaNdex", "paysender", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));
            TShingle shingle3("tesT@yaNdex", "paySender", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));

            UNIT_ASSERT_EQUAL(shingle1 == shingle2, true);
            UNIT_ASSERT_EQUAL(shingle3 == shingle2, false);
        }
        {
            TShingle shingle("test@yandex", "paysender", TSenderBitset(TSenderType::Sender, TSenderType::Domain, TSenderType::Paysender));

            UNIT_ASSERT_EQUAL(*shingle.ShingleEmail(), ShingleFromStroka("test@yandex"));
            UNIT_ASSERT_EQUAL(*shingle.ShingleDomain(), ShingleFromStroka("yandex"));
            UNIT_ASSERT_EQUAL(*shingle.ShinglePSndr(), ShingleFromStroka("paysender"));
        }
    }
    Y_UNIT_TEST(TestPutDataOutMail) {
        {
            TPutDataInMail data;

            UNIT_ASSERT_EQUAL(data.GetAbook(), false);
            UNIT_ASSERT_EQUAL(data.GetDkimSpam(), Nothing());
            UNIT_ASSERT_EQUAL(data.GetKooba(), false);
            UNIT_ASSERT_EQUAL(data.GetMask(), 0);
            UNIT_ASSERT_EQUAL(data.GetPers(), NSenderReputation::PA_NONE);
            UNIT_ASSERT_EQUAL(data.GetPop3Spam(), Nothing());
            UNIT_ASSERT_EQUAL(data.GetPsSpam(), Nothing());
            UNIT_ASSERT_EQUAL(data.GetSpam(), false);
        }
        {
            TPutDataInMail data;
            data.SetAbook(true);
            UNIT_ASSERT_EQUAL(data.GetAbook(), true);
        }
        {
            TPutDataInMail data;
            data.SetDkimSpam(false);
            UNIT_ASSERT_EQUAL(data.GetDkimSpam(), MakeMaybe(false));
        }
        {
            TPutDataInMail data;
            data.SetKooba(true);
            UNIT_ASSERT_EQUAL(data.GetKooba(), true);
        }
        {
            TPutDataInMail data;
            data.SetMask(42);
            UNIT_ASSERT_EQUAL(data.GetMask(), 42);
        }
        {
            TPutDataInMail data;
            data.SetPers(NSenderReputation::PA_PERSSPAM);
            UNIT_ASSERT_EQUAL(data.GetPers(), NSenderReputation::PA_PERSSPAM);
        }
        {
            TPutDataInMail data;
            data.SetPop3Spam(false);
            UNIT_ASSERT_EQUAL(data.GetPop3Spam(), MakeMaybe(false));
        }
        {
            TPutDataInMail data;
            data.SetPsSpam(true);
            UNIT_ASSERT_EQUAL(data.GetPsSpam(), MakeMaybe(true));
        }
        {
            TPutDataInMail data;
            data.SetSpam(true);
            UNIT_ASSERT_EQUAL(data.GetSpam(), true);
        }
    }

    Y_UNIT_TEST(TestComplaintData) {
        {
            TComplaintData data;
            UNIT_ASSERT_EQUAL(data.GetDKimSpam(), Nothing());
            UNIT_ASSERT_EQUAL(data.GetSpam(), false);
        }
        {
            TComplaintData data(true, false);
            UNIT_ASSERT_EQUAL(data.GetDKimSpam(), Nothing());
            UNIT_ASSERT_EQUAL(data.GetSpam(), true);
        }
        {
            TComplaintData data(true, true);
            UNIT_ASSERT_EQUAL(data.GetDKimSpam(), MakeMaybe(true));
            UNIT_ASSERT_EQUAL(data.GetSpam(), true);
        }
    }

    Y_UNIT_TEST(TestDbSenderParsing) {
        NJson::TJsonValue value;

        ui64 counter = 1;

        value["ct"] = counter++;
        value["ut"] = counter++;
        value["ad"] = counter++;
        value["hc"] = counter++;
        value["sc"] = counter++;
        value["ry"] = counter++;
//        value["ch"] = counter++;  check parse empty fields
        value["cs"] = counter++;
        value["fl"] = counter++;
        value["ph"] = counter++;
        value["ps"] = counter++;
        value["ab"] = counter++;
        value["ba"] = counter++;
        value["read"] = counter++;
        value["deleted"] = counter++;
        value["dkim_ham"] = counter++;
        value["dkim_spam"] = counter++;
        value["pop3_ham"] = counter++;
        value["pop3_spam"] = counter++;
        value["dkim_complaint_spam"] = counter++;
        value["dkim_del"] = counter++;
        value["ps_ham"] = counter++;
        value["ps_spam"] = counter++;

        TDBSender sender;

        sender.FromJson(value);

        UNIT_ASSERT_EQUAL(sender.empty, false);
        UNIT_ASSERT_EQUAL(sender.ps_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.ps_ham, --counter);
        UNIT_ASSERT_EQUAL(sender.dkim_del_wo_read, --counter);
        UNIT_ASSERT_EQUAL(sender.compl_dkim_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.pop3_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.pop3_ham, --counter);
        UNIT_ASSERT_EQUAL(sender.dkim_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.dkim_ham, --counter);
        UNIT_ASSERT_EQUAL(sender.del_wo_read, --counter);
        UNIT_ASSERT_EQUAL(sender.read, --counter);
        UNIT_ASSERT_EQUAL(sender.kooba_count, --counter);
        UNIT_ASSERT_EQUAL(sender.abook_count, --counter);
        UNIT_ASSERT_EQUAL(sender.personalspam, --counter);
        UNIT_ASSERT_EQUAL(sender.personalham, --counter);
        UNIT_ASSERT_EQUAL(sender.flags, --counter);
        UNIT_ASSERT_EQUAL(sender.complaint_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.complaint_ham, 0);
        UNIT_ASSERT_EQUAL(sender.rcvd_fromyandex_count, --counter);
        UNIT_ASSERT_EQUAL(sender.spam, --counter);
        UNIT_ASSERT_EQUAL(sender.ham, --counter);
        UNIT_ASSERT_EQUAL(sender.activedays, --counter);
        UNIT_ASSERT_EQUAL(sender.lasttime, --counter);
        UNIT_ASSERT_EQUAL(sender.firsttime, --counter);
        UNIT_ASSERT_EQUAL(counter, 1);
    }

    Y_UNIT_TEST(TestDbSender2WeeksParsing) {
        NJson::TJsonValue value;

        ui64 counter = 1;

        value["ham"] = counter++;
        value["spam"] = counter++;
        value["compl_ham"] = counter++;
        value["compl_spam"] = counter++;
//        value["pers_ham"] = counter++; check empty
        value["pers_spam"] = counter++;
        value["kooba"] = counter++;
        value["read"] = counter++;
        value["deleted"] = counter++;
        value["ps_ham"] = counter++;
        value["ps_spam"] = counter++;
        value["dkim_ham"] = counter++;
        value["dkim_spam"] = counter++;
        value["dkim_del"] = counter++;

        TDBPeriodicSender sender;

        sender.FromJson(value);

        UNIT_ASSERT_EQUAL(sender.dkim_del_wo_read, --counter);
        UNIT_ASSERT_EQUAL_C(sender.dkim_spam_count, --counter, sender.dkim_spam_count << " vs " << counter);
        UNIT_ASSERT_EQUAL(sender.dkim_ham_count, --counter);
        UNIT_ASSERT_EQUAL(sender.ps_spam, --counter);
        UNIT_ASSERT_EQUAL(sender.ps_ham, --counter);
        UNIT_ASSERT_EQUAL(sender.del_wo_read, --counter);
        UNIT_ASSERT_EQUAL(sender.read, --counter);
        UNIT_ASSERT_EQUAL(sender.kooba_count, --counter);
        UNIT_ASSERT_EQUAL(sender.pers_spam_count, --counter);
        UNIT_ASSERT_EQUAL(sender.pers_ham_count, 0);
        UNIT_ASSERT_EQUAL(sender.compl_spam_count, --counter);
        UNIT_ASSERT_EQUAL(sender.compl_ham_count, --counter);
        UNIT_ASSERT_EQUAL(sender.spam_count, --counter);
        UNIT_ASSERT_EQUAL(sender.ham_count, --counter);
    }

    Y_UNIT_TEST(TestDbSender2WeeksInterpolate) {
        TDBPeriodicSender curWeek;
        {
            curWeek.dkim_del_wo_read = RandomNumber<ui64>();
            curWeek.dkim_spam_count = RandomNumber<ui64>();
            curWeek.dkim_ham_count = RandomNumber<ui64>();
            curWeek.ps_spam = RandomNumber<ui64>();
            curWeek.ps_ham = RandomNumber<ui64>();
            curWeek.del_wo_read = RandomNumber<ui64>();
            curWeek.read = RandomNumber<ui64>();
            curWeek.kooba_count = RandomNumber<ui64>();
            curWeek.pers_spam_count = RandomNumber<ui64>();
            curWeek.pers_ham_count = RandomNumber<ui64>();
            curWeek.compl_spam_count = RandomNumber<ui64>();
            curWeek.compl_ham_count = RandomNumber<ui64>();
            curWeek.spam_count = RandomNumber<ui64>();
            curWeek.ham_count = RandomNumber<ui64>();
        }
        TDBPeriodicSender prevWeek;
        {
            prevWeek.dkim_del_wo_read = RandomNumber<ui64>();
            prevWeek.dkim_spam_count = RandomNumber<ui64>();
            prevWeek.dkim_ham_count = RandomNumber<ui64>();
            prevWeek.ps_spam = RandomNumber<ui64>();
            prevWeek.ps_ham = RandomNumber<ui64>();
            prevWeek.del_wo_read = RandomNumber<ui64>();
            prevWeek.read = RandomNumber<ui64>();
            prevWeek.kooba_count = RandomNumber<ui64>();
            prevWeek.pers_spam_count = RandomNumber<ui64>();
            prevWeek.pers_ham_count = RandomNumber<ui64>();
            prevWeek.compl_spam_count = RandomNumber<ui64>();
            prevWeek.compl_ham_count = RandomNumber<ui64>();
            prevWeek.spam_count = RandomNumber<ui64>();
            prevWeek.ham_count = RandomNumber<ui64>();
        }

        const auto day = static_cast<ui16>(kday_t(time(nullptr)));
        const auto interpolatedWeek = TDBPeriodicSender::Interpolate(curWeek, prevWeek, day);

        auto week2 = day / 14;

        const auto daysOnPrevWeek = (week2+1)*14 - day;

        UNIT_ASSERT_EQUAL(interpolatedWeek.dkim_del_wo_read, curWeek.dkim_del_wo_read + prevWeek.dkim_del_wo_read * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.dkim_spam_count, curWeek.dkim_spam_count + prevWeek.dkim_spam_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.dkim_ham_count, curWeek.dkim_ham_count + prevWeek.dkim_ham_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.ps_spam, curWeek.ps_spam + prevWeek.ps_spam * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.ps_ham, curWeek.ps_ham + prevWeek.ps_ham * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.del_wo_read, curWeek.del_wo_read + prevWeek.del_wo_read * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.read, curWeek.read + prevWeek.read * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.kooba_count, curWeek.kooba_count + prevWeek.kooba_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.pers_spam_count, curWeek.pers_spam_count + prevWeek.pers_spam_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.pers_ham_count, curWeek.pers_ham_count + prevWeek.pers_ham_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.compl_spam_count, curWeek.compl_spam_count + prevWeek.compl_spam_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.compl_ham_count, curWeek.compl_ham_count + prevWeek.compl_ham_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.spam_count, curWeek.spam_count + prevWeek.spam_count * daysOnPrevWeek / 14);
        UNIT_ASSERT_EQUAL(interpolatedWeek.ham_count, curWeek.ham_count + prevWeek.ham_count * daysOnPrevWeek / 14);
    }
};
