/*
 * TAnyValue_v2.cpp
 *
 *  Created on: 27 мар. 2017 г.
 *      Author: luckybug
 */

#include <util/generic/bt_exception.h>
#include <util/generic/iterator_range.h>
#include "AnyValue_v2.h"

namespace NAnyValue {
    const TString & TOid::get() const {
        return buf;
    }

    bool TOid::operator==(const TOid& oid) const {
        return buf == oid.buf;
    }

    TOid::TOid() = default;

    TOid::TOid(ui64 s)
    {
        TStringOutput out(buf);
        out << Hex(s, HF_FULL);
    }

    TOid::TOid(TString s) : buf(std::move(s)) {}

    size_t TScalar::Hash() const {
        struct THashVisitor{
            size_t operator()(ui16 v)                const { return THash<ui16>()(v); }
            size_t operator()(ui32 v)                const { return THash<ui32>()(v); }
            size_t operator()(ui64 v)                const { return THash<ui64>()(v); }
            size_t operator()(i64 v)                 const { return THash<i64>()(v); }
            size_t operator()(double v)              const { return THash<double>()(v); }
            size_t operator()(const TString & v)     const { return THash<TString>()(v); }
            size_t operator()(const TOid & v)        const { return THash<TOid>()(v); }
            size_t operator()(bool v)                const { return THash<bool>()(v); }
            size_t operator()(TNoType v)             const { return THash<TNoType>()(v); }
        };
        return std::visit(THashVisitor{}, instance);
    }

    bool TScalar::IsNull() const { return std::holds_alternative<TNoType>(instance); }

    TScalarType TScalar::GetType() const {
        struct TScalarTypeVisitor{
            TScalarType operator()(ui16)                const { return TScalarType::Ui16; }
            TScalarType operator()(ui32)                const { return TScalarType::Ui32; }
            TScalarType operator()(ui64)                const { return TScalarType::Ui64; }
            TScalarType operator()(i64)                 const { return TScalarType::I64; }
            TScalarType operator()(double)              const { return TScalarType::Double; }
            TScalarType operator()(const TString &)     const { return TScalarType::String; }
            TScalarType operator()(const TOid &)        const { return TScalarType::Oid; }
            TScalarType operator()(bool)                const { return TScalarType::Bool; }
            TScalarType operator()(TNoType)             const { return TScalarType::Notype; }
        };
        return std::visit(TScalarTypeVisitor{}, instance);
    }

    bool TScalar::operator==(const TScalar & s) const { return instance == s.instance; }

    struct TScalarWriteVisitor {
        template<typename T> void operator()(const T& v) const {
            if (key.IsInited())
                writer.Write(key, v);
            else
                writer.Write(v);
        }

        void operator()(const TOid & v) const {
            (*this)(v.get());
        }

        void operator()(const TNoType &) const {
            if(key.IsInited())
                writer.WriteNull(key);
            else
                writer.WriteNull();
        }

        explicit TScalarWriteVisitor(NJson::TJsonWriter & writer, TStringBuf key) : writer(writer), key(key) {}

        NJson::TJsonWriter & writer;
        TStringBuf key;
    };

    void TScalar::Write(NJson::TJsonWriter & writer, TStringBuf key) const {
        std::visit(TScalarWriteVisitor(writer, key), instance);
    }

    bool TAnyValue::operator==(const TAnyValue & s) const {
        return instance == s.instance;
    }

    struct TAnyValueTypeVisitor{
        TType operator()(const TScalar &) const { return TType::Scalar; }
        TType operator()(const TMap &) const { return TType::Map; }
        TType operator()(const TArray &) const { return TType::Array; }
        TType operator()(const TNoType &) const { return TType::Notype; }
    };

    TScalarMap TAnyValue::ToScalarMap() const {
        if(Y_UNLIKELY(!IsMap()))
            return {};

        TScalarMap map;
        for(const auto & p : AsMap()) {
            if(Y_UNLIKELY(!p.second.IsScalar()))
                continue;

            map.emplace(p.first, p.second.AsScalar());
        }

        return map;
    }

    TType TAnyValue::GetType() const {
        return std::visit(TAnyValueTypeVisitor{}, instance);
    }

    TAnyValue & TAnyValue::SetType(TType t) {
        switch (t) {
            case TType::Notype: instance = TNoType{}; break;
            case TType::Scalar: instance = TScalar{}; break;
            case TType::Map: instance = TMap{}; break;
            case TType::Array: instance = TArray{}; break;
        }
        return *this;
    }

    void TAnyValue::Write(NJson::TJsonWriter & writer, TStringBuf key) const {
        struct TVisitor{
            void operator()(const TMap & m) const {
                with_av_json_kmap(writer, key) {
                    for(const auto & v : m) {
                        v.second.Write(writer, v.first);
                    }
                }
            }

            void operator()(const TArray & a) const {
                with_av_json_karray(writer, key) {
                    for(const auto & v : a) {
                        v.Write(writer);
                    }
                }
            }

            void operator()(const TScalar & s) const {
                s.Write(writer, key);
            }

            void operator()(const TNoType &) const {
                if(key.IsInited())
                    writer.WriteNull(key);
                else
                    writer.WriteNull();
            }

            explicit TVisitor(NJson::TJsonWriter & writer, TStringBuf key) : writer(writer), key(key) {}

            NJson::TJsonWriter & writer;
            TStringBuf key;
        };

        std::visit(TVisitor(writer, key), instance);
    }

    size_t Hash(const NAnyValue::TScalarMap & value) {
        size_t res = 0;

        for(const auto & p : value) {
            res = CombineHashes(res, THash<TString>()(p.first));
            res = CombineHashes(res, p.second.Hash());
        }

        return res;
    }

    size_t Hash(const NAnyValue::TScalarVector & value) {
        size_t res = 0;

        for(const auto & p : value) {
            res = CombineHashes(res, p.Hash());
        }

        return res;
    }
    size_t Hash(const NAnyValue::TScalarVectorMap & value) {
        size_t res = 0;

        for(const auto & p : value) {
            res = CombineHashes(res, THash<TString>()(p.first));
            res = CombineHashes(res, Hash(p.second));
        }

        return res;
    }


    template<class T> struct TSetVisitor{
        void operator()(TMap * m) {
            m->emplace(lastKey, value);
        }
        void operator()(TArray * a) {
            a->emplace_back(value);
        }
        void operator()(TAnyValue * v) {
            *v = value;
        }

        TSetVisitor(TSetVisitor&&) noexcept = default;
        TSetVisitor(const typename TTypeTraits<T>::TFuncParam value, TStringBuf lastKey)
                : value(value), lastKey(lastKey) {}

        const typename TTypeTraits<T>::TFuncParam value;
        TStringBuf lastKey;
    };

    struct TVisitorOpenMap {
        TMap * operator()(TMap * m) {
            return &m->emplace(lastKey, TMap{}).first->second.AsMap();
        }
        TMap * operator()(TArray * a) {
            return &a->emplace_back(TMap{}).AsMap();
        }
        TMap * operator()(TAnyValue * v) {
            return &v->SetType(TType::Map).AsMap();
        }

        TVisitorOpenMap(TVisitorOpenMap&&) noexcept = default;

        explicit TVisitorOpenMap(TStringBuf lastKey) : lastKey(lastKey) {}
        TStringBuf lastKey;
    };

    bool TAnyValueParser::OnNull() {
        std::visit(TSetVisitor<TNoType>{{}, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnBoolean(bool v) {
        std::visit(TSetVisitor<bool>{v, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnInteger(long long v) {
        std::visit(TSetVisitor<i64>{v, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnUInteger(unsigned long long v) {
        std::visit(TSetVisitor<ui64>{v, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnDouble(double v) {
        std::visit(TSetVisitor<double>{v, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnString(const TStringBuf& v) {
        std::visit(TSetVisitor<TString>{TString{v}, lastKey}, stack.top());
        return true;
    }
    bool TAnyValueParser::OnStringNoCopy(const TStringBuf& v) {
        std::visit(TSetVisitor<TString>{TString{v}, lastKey}, stack.top());
        return true;
    }

    bool TAnyValueParser::OnOpenMap() {
        stack.emplace(std::visit(TVisitorOpenMap(lastKey), stack.top()));
        return true;
    }

    bool TAnyValueParser::OnMapKey(const TStringBuf& v) {
        lastKey = TString{v};
        return true;
    }

    bool TAnyValueParser::OnCloseMap() {
        stack.pop();
        return true;
    }

    struct TVisitorOpenArray {
        TArray * operator()(TMap * m) {
            return &m->emplace(lastKey, TArray{}).first->second.AsArray();
        }
        TArray * operator()(TArray * a) {
            return &a->emplace_back(TArray{}).AsArray();
        }
        TArray * operator()(TAnyValue * v) {
            return &v->SetType(TType::Array).AsArray();
        }

        TVisitorOpenArray(TVisitorOpenArray&&) noexcept = default;

        explicit TVisitorOpenArray(TStringBuf lastKey) : lastKey(lastKey) {}

        TStringBuf lastKey;
    };
    bool TAnyValueParser::OnOpenArray() {
        stack.emplace(std::visit(TVisitorOpenArray(lastKey), stack.top()));
        return true;
    }
    bool TAnyValueParser::OnCloseArray() {
        stack.pop();
        return true;
    }

    bool TAnyValueParser::OnMapKeyNoCopy(const TStringBuf& v) {
        lastKey = TString{v};
        return true;
    }
    void TWriter::Write(TStringBuf key, const TScalar & scalar) {
        scalar.Write(*this, key);
    }

    void TWriter::Write(const TScalar & scalar) {
        scalar.Write(*this);
    }

    void TWriter::Write(TStringBuf key, const TMap & map) {
        with_av_json_kmap(*this, key) {
            for(const auto & v : map)
                Write(v.first, v.second);
        }
    }

    void TWriter::Write(const TMap & map) {
        with_av_json_map(*this) {
            for(const auto & v : map)
                Write(v.first, v.second);
        }
    }

    void TWriter::Write(TStringBuf key, const TArray & array) {
        with_av_json_karray(*this, key) {
            for(const auto & v : array)
                Write(v);
        }
    }

    void TWriter::Write(const TArray & array) {
        with_av_json_array(*this) {
            for(const auto & v : array)
                Write(v);
        }
    }

    void TWriter::Write(TStringBuf key, const TAnyValue & value) {
        value.Write(*this, key);
    }
    void TWriter::Write(const TAnyValue & value) {
        value.Write(*this);
    }
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalar & value) {
    NAnyValue::TWriter writer(&stream, true);
    writer.Write(value);
    return stream;
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TArray & value) {
    NAnyValue::TWriter writer(&stream, true);
    writer.Write(value);
    return stream;
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TMap & value) {
    NAnyValue::TWriter writer(&stream, true);
    writer.Write(value);
    return stream;
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TAnyValue & value) {
    NAnyValue::TWriter writer(&stream, true);
    value.Write(writer);
    return stream;
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalarMap & value) {
    NAnyValue::TWriter writer(&stream, true);
    with_av_json_map(writer) {
        for(const auto & v : value)
            v.second.Write(writer, v.first);
    }
    return stream;
}

IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalarVectorMap & value) {
    NAnyValue::TWriter writer(&stream, true);
    with_av_json_map(writer) {
        for(const auto & p : value) {
            with_av_json_karray(writer, p.first) {
                for(const auto & v : p.second)
                    v.Write(writer);
            }
        }
    }
    return stream;
}
