#pragma once

#include <util/system/defaults.h>
#include <util/generic/hash.h>
#include <util/generic/deque.h>
#include <util/generic/vector.h>
#include <util/generic/variant.h>
#include <util/stream/format.h>
#include <util/generic/serialized_enum.h>
#include <util/generic/xrange.h>
#include <util/generic/yexception.h>
#include <util/generic/bt_exception.h>
#include <library/cpp/json/json_writer.h>
#include <array>
#include <util/generic/queue.h>
#include <util/generic/stack.h>
#include <util/system/type_name.h>


struct TNoType {
    inline void Save(IOutputStream*) const {}
    inline void Load(IInputStream*) {}
    bool operator==(const TNoType &) const { return true; }

    explicit operator size_t() const { return 0; }
};

namespace NAnyValue {
    enum class TScalarType : ui16 {
        Notype = 0,
        Ui16,
        Ui32,
        Ui64,
        I64,
        Double,
        String,
        Oid,
        Bool
    };

    enum class TType : ui16 {
        Notype = 0,
        Scalar,
        Map,
        Array
    };

    template<typename T>
    using make_signed_t = typename std::conditional_t<std::is_unsigned<T>::value &&
                                                      !std::is_same<bool, T>::value, std::make_signed<T>, std::common_type<T>>::type;

    struct TOid {
        const TString &get() const;

        bool operator==(const TOid &oid) const;

        explicit operator size_t() const { return THash<TString>()(buf); }

        explicit TOid(ui64 s);
        explicit TOid(TString s);

        TOid();
        TOid(const TOid &) = default;
        TOid(TOid &&) noexcept = default;
        TOid &operator=(const TOid &) = default;
        TOid &operator=(TOid &&) noexcept = default;

        Y_SAVELOAD_DEFINE(buf)

    private:
        TString buf = "0";
    };

#define GetSetAs(type, name)  \
    type & As ## name() { return As<type>(); }  \
    TTypeTraits<type>::TFuncParam As ## name() const { return As<type>(); }

#define CopyOp(base, copy)  \
    base& operator=(copy&& v) { instance = std::forward<copy>(v); return *this; }

    class TScalar {
    public:

        size_t Hash() const;

        bool IsNull() const;

        TScalarType GetType() const;

        bool operator==(const TScalar &s) const;

        void Write(NJson::TJsonWriter &writer, TStringBuf key = {}) const;

    private:
        template<typename T>
        struct TScalarRobustGetterVisitor {
            template<typename Q>
            T operator()(const Q& v) const { return static_cast<T>(v); }

            T operator()(const TString &v) const { return FromString<T>(v); }

            T operator()(const TOid &v) const { return FromString<T>(v.get()); }

            T operator()(TNoType) const { return T{}; }
        };

    public:
        template<typename T>
        decltype(auto) Visit(T&& visitor) const {
            return std::visit(std::forward<T>(visitor), instance);
        }

        template<typename T>
        decltype(auto) GetRobust() const {
            return Visit(TScalarRobustGetterVisitor<T>{});
        }

        template<class T>
        typename TTypeTraits<T>::TFuncParam As() const {
            if (!std::holds_alternative<T>(instance))
                ythrow TWithBackTrace<yexception>() << "isn't valid: " << TypeName<T>();

            return std::get<T>(instance);
        }

        template<class T>
        T &As() {
            if (!std::holds_alternative<T>(instance))
                ythrow TWithBackTrace<yexception>() << "isn't valid: " << TypeName<T>();

            return std::get<T>(instance);
        }

        GetSetAs(ui16, Ui16)
        CopyOp(TScalar, ui16)

        GetSetAs(ui32, Ui32)
        CopyOp(TScalar, ui32)

        GetSetAs(ui64, Ui64)
        CopyOp(TScalar, ui64)

        GetSetAs(i64, I64)
        CopyOp(TScalar, i64)

        GetSetAs(double, Double)
        CopyOp(TScalar, double)

        GetSetAs(TString, String)
        CopyOp(TScalar, TString)

        GetSetAs(TOid, Oid)
        CopyOp(TScalar, TOid)

        GetSetAs(bool, Bool)
        CopyOp(TScalar, bool)

        Y_SAVELOAD_DEFINE(instance)


        template<typename T, typename = std::enable_if_t<!std::is_same<std::decay_t<T>, TScalar>::value>>
        TScalar(T && v) : instance(std::forward<T>(v)) {}

        TScalar() : instance(TNoType{}) {}

        TScalar(TScalar &&) noexcept = default;

        TScalar(const TScalar &) = default;

        TScalar& operator=(const TScalar&) = default;

        TScalar& operator=(TScalar&&) = default;

    private:
        using TInstance = std::variant<TNoType, ui16, ui32, ui64, i64, double, bool, TOid, TString>;
        TInstance instance = TNoType{};
    };

    template<> struct TScalar::TScalarRobustGetterVisitor<TString> {
        template<typename Q>
        TString operator()(const Q& v) const { return ToString(v); }

        TString operator()(const TOid &v) const { return v.get(); }

        TString operator()(TNoType) const { return ""; }
    };
    template<> struct TScalar::TScalarRobustGetterVisitor<bool> {
        template<typename Q> bool operator()(const Q& v) const { return !(v == Q{}); }
    };

    class TAnyValue;

    using TMap = THashMap<TString, TAnyValue>;
    using TScalarMap = THashMap<TString, TScalar>;
    using TScalarVector = TVector<TScalar>;
    using TScalarVectorMap = THashMap<TString, TScalarVector>;
    using TArray = TVector<TAnyValue>;

    class TAnyValue {
    private:
        using TInstance = std::variant<TNoType, TScalar, TMap, TArray>;
        TInstance instance = TNoType{};
    public:

        TScalarMap ToScalarMap() const;

        template<TType type> bool Is() const { return GetType() == type; }
#define IsDef(name, type) bool Is##name() const { return Is<type>(); }

        IsDef(Map, TType::Map)

        IsDef(Array, TType::Array)

        IsDef(Scalar, TType::Scalar)

        IsDef(Null, TType::Notype)

#undef IsDef

        TType GetType() const;

        TAnyValue &SetType(TType t);

        void Write(NJson::TJsonWriter &writer, TStringBuf key = {}) const;

        bool operator==(const TAnyValue &s) const;

    private:
        template<class T> struct TAnyValueRobustGetterVisitor {
            T operator()(const TMap &v) const { return static_cast<T>(v.size()); }

            T operator()(const TArray &v) const { return static_cast<T>(v.size()); }

            T operator()(const TScalar &v) const { return v.GetRobust<T>(); }

            T operator()(TNoType) const { return {}; }
        };

    public:
        template<typename T> decltype(auto) Visit(T&& visitor) const {
            return std::visit(std::forward<T>(visitor), instance);
        }

        template<typename T>
        requires (!requires {TInstance{}.emplace<T>(); })
        T GetRobust(int = 0) const {
            if (!IsScalar())
                return {};
            return AsScalar().GetRobust<T>();
        };

        template<class T> typename TTypeTraits<T>::TFuncParam As() const { return std::get<T>(instance); }

        template<class T> T &As() { return std::get<T>(instance); }

        GetSetAs(TScalar, Scalar)

        GetSetAs(TMap, Map)

        GetSetAs(TArray, Array)

        Y_SAVELOAD_DEFINE(instance)

        template<typename T>
        std::enable_if_t<!std::is_same<std::decay_t<T>, TAnyValue>::value, TAnyValue&> operator=(T&& v) {
            instance = TScalar(std::forward<T>(v));
            return *this;
        }

        CopyOp(TAnyValue, TScalar)

        CopyOp(TAnyValue, TMap)

        CopyOp(TAnyValue, TArray)

        TAnyValue(TScalar v) : instance(std::move(v)) {}

        TAnyValue(TMap v) : instance(std::move(v)) {}

        TAnyValue(TArray v) : instance(std::move(v)) {}

        template<typename T> TAnyValue(T v) : instance(TScalar(std::move(v))) {}

        TAnyValue() = default;

        TAnyValue(TAnyValue &&) noexcept = default;
        TAnyValue(const TAnyValue &) = default;

        TAnyValue& operator=(TAnyValue &&) = default;
        TAnyValue& operator=(const TAnyValue &) = default;
    };

    size_t Hash(const NAnyValue::TScalarMap &value);

    size_t Hash(const NAnyValue::TScalarVector &value);

    size_t Hash(const NAnyValue::TScalarVectorMap &value);

    class TAnyValueParser : public NJson::TJsonCallbacks {
    public:
        explicit TAnyValueParser(TAnyValue &target) {
            stack.emplace(&target);
        }

    private:

        bool OnNull() final;

        bool OnBoolean(bool v) final;

        bool OnInteger(long long v) final;

        bool OnUInteger(unsigned long long v) final;

        bool OnDouble(double v) final;

        bool OnString(const TStringBuf &v) final;

        bool OnStringNoCopy(const TStringBuf &v) final;

        bool OnOpenMap() final;

        bool OnMapKey(const TStringBuf &v) final;

        bool OnCloseMap() final;

        bool OnOpenArray() final;

        bool OnCloseArray() final;

        bool OnMapKeyNoCopy(const TStringBuf &v) final;

        TString lastKey;
        using TStackSlice = std::variant<TMap *, TArray *, TAnyValue *>;
        TStack<TStackSlice> stack;

    };

    class TWriter : public NJson::TJsonWriter {
    public:
        using NJson::TJsonWriter::TJsonWriter;
        using NJson::TJsonWriter::Write;

        void Write(TStringBuf key, const TScalar & scalar);
        void Write(const TScalar & scalar);
        void Write(TStringBuf key, const TMap & map);
        void Write(const TMap & map);
        void Write(TStringBuf key, const TArray & array);
        void Write(const TArray & array);
        void Write(TStringBuf key, const TAnyValue & value);
        void Write(const TAnyValue & value);
    };

    struct TMapScope {
        TMapScope(NJson::TJsonWriter &writer, TStringBuf key = {}) : writer(writer), key(key) {}
        void Acquire() {
            if(key.IsInited())
                writer.OpenMap(key);
            else
                writer.OpenMap();
        }
        void Release() { writer.CloseMap(); }
        NJson::TJsonWriter &writer;
        TStringBuf key;
    };

    struct TArrayScope {
        TArrayScope(NJson::TJsonWriter &writer, TStringBuf key = {}) : writer(writer), key(key) {}
        void Acquire() {
            if(key.IsInited())
                writer.OpenArray(key);
            else
                writer.OpenArray();
        }
        void Release() { writer.CloseArray(); }
        NJson::TJsonWriter &writer;
        TStringBuf key;
    };
}

template<> inline NAnyValue::TOid FromStringImpl<NAnyValue::TOid, char>(const char* data, size_t len) {
    return NAnyValue::TOid({data, len});
}

#define with_av_json_map(writer) NAnyValue::TMapScope Y_CAT(SCOPE_, __LINE__)(writer); with_lock(Y_CAT(SCOPE_, __LINE__))
#define with_av_json_kmap(writer, key) NAnyValue::TMapScope Y_CAT(SCOPE_, __LINE__)(writer, key); with_lock(Y_CAT(SCOPE_, __LINE__))
#define with_av_json_array(writer) NAnyValue::TArrayScope Y_CAT(SCOPE_, __LINE__)(writer); with_lock(Y_CAT(SCOPE_, __LINE__))
#define with_av_json_karray(writer, key) NAnyValue::TArrayScope Y_CAT(SCOPE_, __LINE__)(writer, key); with_lock(Y_CAT(SCOPE_, __LINE__))


IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalar & value);
IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TArray & value);
IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TMap & value);
IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TAnyValue & value);
IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalarMap & value);
IOutputStream & operator << (IOutputStream & stream, const NAnyValue::TScalarVectorMap & value);

