#pragma once

/*
 * StorageBase.h
 *
 *  Created on: 31 мар. 2017 г.
 *      Author: luckybug
 */

#ifndef STORAGEBASEUT_H_
#define STORAGEBASEUT_H_

#include "StorageBase.h"

#include <library/cpp/testing/unittest/registar.h>

class StorageBaseTest: public TTestBase {
    UNIT_TEST_SUITE(StorageBaseTest);
    UNIT_TEST(RemoveAndCount);
    UNIT_TEST(UpdateWithoutUpsert);
    UNIT_TEST(UpdateWithUpsert);
    UNIT_TEST(SetOnInsert);
    UNIT_TEST(Updates);
    UNIT_TEST(ui64key);
    UNIT_TEST(FindSeveral);
    UNIT_TEST(UpdateSeveral);
    UNIT_TEST_SUITE_END();

    TStorageBase& storage;
    TString collectionName;

public:
    StorageBaseTest(TStorageBase& storage, TString collectionName)
        : storage(storage)
        , collectionName(collectionName)
    {
    }
    void RemoveAndCount() {
        storage.Remove(collectionName, TFindAction());

        UNIT_ASSERT_EQUAL(storage.Count(collectionName, TFindAction()), 0);
    }
    void UpdateWithoutUpsert() {
        TUpdateAction action;

        action.query.equals["id"] = 1u;
        action.update.sets["i16"] = 2u;
        action.upsert = false;

        storage.Update(collectionName, action);

        UNIT_ASSERT_EQUAL(storage.Count(collectionName, TFindAction()), 0);
    }

    void UpdateWithUpsert() {
        {
            TUpdateAction action;

            action.query.equals["id"] = 1u;
            action.update.sets["i16"] = 2u;
            action.upsert = true;

            storage.Update(collectionName, action);
        }

        UNIT_ASSERT_EQUAL(storage.Count(collectionName, TFindAction()), 1);

        {
            TFindAction findAction;
            findAction.query.equals["id"] = 1u;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL_C(result["id"].getLong(), 1u, result["id"].getLong());
            UNIT_ASSERT_EQUAL(result["i16"].getShort(), 2u);
        }
    }

    void SetOnInsert() {
        {
            TUpdateAction action;

            action.query.equals["id"] = 1u;
            action.update.setsOnInsert["i16"] = 3u;
            action.upsert = true;

            storage.Update(collectionName, action);
        }

        UNIT_ASSERT_EQUAL(storage.Count(collectionName, TFindAction()), 1);

        {
            TFindAction findAction;
            findAction.query.equals["id"] = 1u;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL(result["id"].getLong(), 1u);
            UNIT_ASSERT_EQUAL_C(result["i16"].getShort(), 2u, result["i16"].getShort());
        }

        {
            TUpdateAction action;

            action.query.equals["id"] = 2u;
            action.update.setsOnInsert["i16"] = 3u;
            action.upsert = true;

            storage.Update(collectionName, action);
        }

        UNIT_ASSERT_EQUAL(storage.Count(collectionName, TFindAction()), 2);

        {
            TFindAction findAction;
            findAction.query.equals["id"] = 2u;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL(result["id"].getLong(), 2u);
            UNIT_ASSERT_EQUAL(result["i16"].getShort(), 3u);
        }
    }
    void Updates() {
        {
            TUpdateAction action;

            action.query.equals["id"] = 1u;
            action.update.sets["i16"] = 5u;
            action.update.ors["i32"] = 4u;
            action.update.incrs["double"] = 5u;
            action.update.sets["text"] = "some text";

            storage.Update(collectionName, action);
        }

        {
            TFindAction findAction;
            findAction.query.equals["id"] = 1u;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL(result["id"].getLong(), 1u);
            UNIT_ASSERT_EQUAL(result["i16"].getShort(), 5u);
            UNIT_ASSERT_EQUAL(result["i32"].getInteger(), 4u);
            UNIT_ASSERT_EQUAL(result["double"].getDouble(), 5u);
            UNIT_ASSERT_STRINGS_EQUAL(result["text"].getString(), "some text");
        }

        {
            TUpdateAction action;

            action.query.equals["id"] = 1u;
            action.update.sets["i16"] = 6u;
            action.update.ors["i32"] = 8u;
            action.update.incrs["double"] = 6u;
            action.update.sets["text"] = "another text";

            storage.Update(collectionName, action);
        }

        {
            TFindAction findAction;
            findAction.query.equals["id"] = 1u;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL(result["id"].getLong(), 1u);
            UNIT_ASSERT_EQUAL(result["i16"].getShort(), 6u);
            UNIT_ASSERT_EQUAL(result["i32"].getInteger(), 12u);
            UNIT_ASSERT_EQUAL(result["double"].getDouble(), 11u);
            UNIT_ASSERT_STRINGS_EQUAL(result["text"].getString(), "another text");
        }
    }

    void ui64key() {
        ui64 id = ui64(-1);

        {
            TUpdateAction action;

            action.upsert = true;

            action.query.equals["id"] = id;
            action.update.sets["i16"] = 6u;

            storage.Update(collectionName, action);
        }

        {
            TFindAction findAction;
            findAction.query.equals["id"] = id;

            NAnyValue::HashMap result;
            storage.FindOne(collectionName, findAction, result);

            UNIT_ASSERT_EQUAL_C(result["id"].getLong(), id, result["id"].getLong() << " vs " << id);
            UNIT_ASSERT_EQUAL(result["i16"].getShort(), 6u);
        }
    }

    void FindSeveral() {
        TFindAction findAction;
        {
            TQueryData query;
            query.equals["id"] = 1u;
            findAction.query.ors.push_back(query);
        }
        {
            TQueryData query;
            query.equals["id"] = 2u;
            findAction.query.ors.push_back(query);
        }
        TFindResults results;
        storage.Find(collectionName, findAction, results);

        THashMap<ui64, NAnyValue::HashMap> resultsByIds;

        for (TFindResults::const_iterator it = results.begin(); it != results.end(); ++it) {
            const NAnyValue::HashMap& h = *it;
            resultsByIds[h["id"].getLong()] = h;
        }

        NAnyValue::HashMap result = resultsByIds[1];

        UNIT_ASSERT_EQUAL(result["id"].getLong(), 1u);
        UNIT_ASSERT_EQUAL(result["i16"].getShort(), 6u);
        UNIT_ASSERT_EQUAL(result["i32"].getInteger(), 12u);
        UNIT_ASSERT_EQUAL(result["double"].getDouble(), 11u);
        UNIT_ASSERT_STRINGS_EQUAL(result["text"].getString(), "another text");

        result = resultsByIds[2];

        UNIT_ASSERT_EQUAL(result["id"].getLong(), 2u);
        UNIT_ASSERT_EQUAL(result["i16"].getShort(), 3u);
    }

    void UpdateSeveral() {
        TActionSeries actions;

        {
            actions.push_back();
            TUpdateAction& action = actions.back();

            action.upsert = true;

            action.query.equals["id"] = 1u;
            action.update.incrs["i32"] = 6u;
            action.update.setsOnInsert["double"] = 456u;
        }
        {
            actions.push_back();
            TUpdateAction& action = actions.back();

            action.upsert = true;

            action.query.equals["id"] = 3u;
            action.update.incrs["i32"] = 6u;
            action.update.setsOnInsert["double"] = 456u;
        }

        storage.UpdateSeries(collectionName, actions, true);

        TFindAction findAction;
        {
            TQueryData query;
            query.equals["id"] = 1u;
            findAction.query.ors.push_back(query);
        }
        {
            TQueryData query;
            query.equals["id"] = 3u;
            findAction.query.ors.push_back(query);
        }
        TFindResults results;
        storage.Find(collectionName, findAction, results);

        THashMap<ui64, NAnyValue::HashMap> resultsByIds;

        for (TFindResults::const_iterator it = results.begin(); it != results.end(); ++it) {
            const NAnyValue::HashMap& h = *it;
            resultsByIds[h["id"].getLong()] = h;
        }

        NAnyValue::HashMap result = resultsByIds[1];

        UNIT_ASSERT_EQUAL(result["id"].getLong(), 1u);
        UNIT_ASSERT_EQUAL(result["i16"].getShort(), 6u);
        UNIT_ASSERT_EQUAL(result["i32"].getInteger(), 18u);
        UNIT_ASSERT_EQUAL(result["double"].getDouble(), 11u);
        UNIT_ASSERT_STRINGS_EQUAL(result["text"].getString(), "another text");

        result = resultsByIds[3];

        UNIT_ASSERT_EQUAL(result["id"].getLong(), 3u);
        UNIT_ASSERT_EQUAL(result["i32"].getInteger(), 6u);
        UNIT_ASSERT_EQUAL(result["double"].getDouble(), 456u);
    }
};

#endif /* STORAGEBASEUT_H_ */
