
#include <util/string/cast.h>
#include <util/string/split.h>
#include "anyvalue.h"

namespace nosql {
    AnyValue::AnyValue() {
        Clear();
    }

    AnyValue::AnyValue(double value) {
        operator=(value);
    }

    AnyValue::AnyValue(int value) {
        operator=(value);
    }

    AnyValue::AnyValue(ui32 value) {
        operator=(value);
    }

    AnyValue::AnyValue(i64 value) {
        operator=(value);
    }

    AnyValue::AnyValue(const TVector<char>& value) {
        operator=(value);
    }

    AnyValue::AnyValue(const TVector<AnyValue>& value) {
        operator=(value);
    }

    AnyValue::AnyValue(const HashMap& value) {
        operator=(value);
    }

    AnyValue::AnyValue(const TString& value) {
        operator=(value);
    }

    AnyValue::AnyValue(const char* value) {
        operator=(value);
    }

    AnyValue& AnyValue::operator=(double value) {
        Clear();
        m_double = value;
        m_int64 = static_cast<i64>(value);
        m_type = DOUBLE;
        return *this;
    }

    AnyValue& AnyValue::operator=(int value) {
        Clear();
        m_int64 = value;
        m_double = static_cast<double>(value);
        m_type = INTEGER;
        return *this;
    }

    AnyValue& AnyValue::operator=(ui32 value) {
        Clear();
        m_int64 = value;
        m_double = static_cast<double>(value);
        m_type = INTEGER;
        return *this;
    }

    AnyValue& AnyValue::operator=(i64 value) {
        Clear();
        m_int64 = value;
        m_double = static_cast<double>(value);
        m_type = INTEGER64;
        return *this;
    }

    AnyValue& AnyValue::operator=(const TVector<char>& value) {
        Clear();
        m_vector = value;
        m_type = VECTOR;
        return *this;
    }

    AnyValue& AnyValue::operator=(const TVector<AnyValue>& value) {
        Clear();
        m_list = value;
        m_type = LIST;
        return *this;
    }

    AnyValue& AnyValue::operator=(const HashMap& value) {
        Clear();
        m_hash = value;
        m_type = HASH;
        return *this;
    }

    AnyValue& AnyValue::operator=(const TString& value) {
        Clear();
        m_string = value;
        m_type = STRING;
        return *this;
    }

    AnyValue& AnyValue::operator=(const char* value) {
        Clear();
        m_string = value;
        m_type = STRING;
        return *this;
    }

    bool AnyValue::operator==(const AnyValue& obj) const {
        if (Type() != obj.Type())
            return false;

        switch (Type()) {
            case INTEGER:
            case INTEGER64:
                return m_int64 == obj.m_int64;
            case DOUBLE:
                return m_double == obj.m_double;
            case VECTOR:
                return m_vector == obj.m_vector;
            case STRING:
                return m_string == obj.m_string;
            case LIST:
                return m_list == obj.m_list;
            case NOTYPE:
            default:
                return false;
        }
    }

    double AnyValue::Double() const {
        return m_double;
    }

    ui32 AnyValue::Integer() const {
        return static_cast<int>(m_int64);
    }

    i64 AnyValue::Long() const {
        return m_int64;
    }

    const TVector<char>& AnyValue::Vector() const {
        return m_vector;
    }

    const TVector<AnyValue>& AnyValue::List() const {
        return m_list;
    }

    const HashMap& AnyValue::Hash() const {
        return m_hash;
    }

    const TString& AnyValue::String() const {
        return m_string;
    }

    AnyValue::TYPE AnyValue::Type() const {
        return m_type;
    }

    void AnyValue::Clear() {
        m_double = 0;
        m_int64 = 0;
        m_vector.clear();
        m_list.clear();
        m_string.clear();
        m_type = NOTYPE;
    }

    TDataCounters::TDataCounters() {
        Clear();
    }

    TDataCounters::~TDataCounters() {
    }

    void TDataCounters::Clear() {
        memset(count_by_type, 0, sizeof(count_by_type));
        empty_data = 0;
    }

    TString TDataCounters::GetCountersByType() {
        char tbuff[512];

        memset(tbuff, 0, sizeof(tbuff));
        snprintf(tbuff, sizeof(tbuff) - 1, "%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 ",%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "", count_by_type[nosql::AnyValue::INTEGER], count_by_type[nosql::AnyValue::INTEGER64], count_by_type[nosql::AnyValue::DOUBLE], count_by_type[nosql::AnyValue::STRING], count_by_type[nosql::AnyValue::NOTYPE], count_by_type[nosql::AnyValue::VECTOR], count_by_type[nosql::AnyValue::LIST], count_by_type[nosql::AnyValue::HASH]);

        return TString(tbuff);
    }

    TString AnyValueByTypeToStroka(HashMap::const_iterator& hit, TDataCounters& counters) {
        AnyValue::TYPE type = (*hit).second.Type();
        TString result = (*hit).first + "=#" + ToString(static_cast<int>(type)) + '\'';

        switch (type) {
            case AnyValue::INTEGER:
                result += ToString((*hit).second.Integer());
                break;
            case AnyValue::INTEGER64:
                result += ToString((*hit).second.Long());
                break;
            case AnyValue::DOUBLE:
                result += ToString((*hit).second.Double());
                break;
            case AnyValue::STRING: {
                TString temporary = (*hit).second.String();
                CGIEscape(temporary);
                result += temporary;
                break;
            }
            case AnyValue::LIST:
            case AnyValue::HASH:
            case AnyValue::VECTOR:
            case AnyValue::NOTYPE:
                break;
            default:
                return nullptr;
        };

        counters.count_by_type[type] = IncMax64(counters.count_by_type[type], 1);

        result += "'&";
        return result;
    }

    TString HashMapToStroka(HashMap& hash, TDataCounters& counters, TString& id) {
        TString recstr = "";
        bool exists_data = false;
        HashMap::const_iterator hit;

        id = "";

        hit = hash.find("_id");
        if (hit != hash.end()) {
            exists_data = true;
            id = AnyValueByTypeToStroka(hit, counters);
            recstr += id;
        }

        hit = hash.begin();
        while (hit != hash.end()) {
            if ((*hit).first == "_id") {
                ++hit;
                continue;
            }

            exists_data = true;
            recstr += AnyValueByTypeToStroka(hit, counters);

            ++hit;
        }
        if (!exists_data)
            counters.empty_data = IncMax64(counters.empty_data, 1);

        return recstr;
    }

    TString HashMapToStrokaBase(const HashMap& hash) {
        TString recstr = "";
        HashMap::const_iterator hit;
        TDataCounters counters;

        hit = hash.begin();
        while (hit != hash.end()) {
            recstr += AnyValueByTypeToStroka(hit, counters);

            ++hit;
        }

        return recstr;
    }

    namespace stroka_to_hash_map_detail {
        static TStringBuf GetPart(TStringBuf& data, const char delimeter, bool check_zero_size, const char* error_message) {
            size_t position = data.find(delimeter);
            if (position == TStringBuf::npos || (check_zero_size && position == 0))
                ythrow TFromStringException() << error_message << data;

            TStringBuf result = TStringBuf(data, 0, position);
            data.Skip(position + 1);
            return result;
        }

        static TStringBuf GetName(TStringBuf& data) {
            return GetPart(data, '=', true, ": can't find field name in ");
        }

        static AnyValue::TYPE GetType(TStringBuf& data) {
            data.Skip(1);
            TStringBuf value = GetPart(data, '\'', true, ": can't find field type in");
            return static_cast<AnyValue::TYPE>(FromString<ui32>(value));
        }

        static TStringBuf GetValueData(TStringBuf& data, AnyValue::TYPE type) {
            return GetPart(data, '\'', type != AnyValue::STRING, ": can't find value data in");
        }

        static void CheckForUnexpectedData(const TStringBuf& data) {
            if (!data.empty())
                ythrow TFromStringException() << ": find unexpected data: " << data;
        }

        static AnyValue GetValue(const TStringBuf& data, AnyValue::TYPE type) {
            switch (type) {
                case nosql::AnyValue::INTEGER:
                    return FromString<i32>(data);
                case nosql::AnyValue::INTEGER64:
                    return FromString<i64>(data);
                case nosql::AnyValue::DOUBLE:
                    return FromString<double>(data);
                case nosql::AnyValue::STRING: {
                    TString temporary = ToString(data);
                    CGIUnescape(temporary);
                    return temporary;
                }
                //non-supported types
                case nosql::AnyValue::VECTOR:
                case nosql::AnyValue::LIST:
                case nosql::AnyValue::HASH:
                default:
                    ythrow TFromStringException() << ": non supported type - " << static_cast<ui32>(type);
            };

            return AnyValue();
        }
    }

    bool pStrokaToHashMapBase(const char* pdatastr, size_t pdatastrlen, nosql::HashMap& hash, TString& err_str) try {
        using namespace stroka_to_hash_map_detail;
        if (pdatastr == nullptr || pdatastrlen == 0)
            ythrow TFromStringException() << "empty record data";

        HashMap result;

        for(auto it : StringSplitter(TStringBuf(pdatastr, pdatastrlen)).Split('&')) {
            TStringBuf item = it.Token();
            TStringBuf name = GetName(item);
            AnyValue::TYPE type = GetType(item);
            TStringBuf data = GetValueData(item, type);

            CheckForUnexpectedData(item);
            result[ToString(name)] = GetValue(data, type);
        }

        hash.swap(result);
        return true;
    } catch (const yexception& ye) {
        err_str = ye.what();
        return false;
    }

    bool StrokaToHashMapBase(const TString& datastr, nosql::HashMap& hash, TString& err_str) {
        return pStrokaToHashMapBase(datastr.c_str(), datastr.length(), hash, err_str);
    }
}
