#pragma once

#include <util/generic/map.h>
#include <util/generic/vector.h>

template <class T>
class THist {
public:
    struct TBin {
        bool operator==(const TBin &rhs) const {
            return val == rhs.val &&
                   count == rhs.count;
        }

        friend bool operator < (typename TTypeTraits<T>::TFuncParam another, const TBin & bin) { return another < bin.val; }
        bool operator < (const TBin & bin) const { return val < bin.val; }
        explicit TBin(T val) : val(std::forward<T>(val)) {}
        explicit TBin(T val, size_t count) : val(std::forward<T>(val)), count(count)  {}
        T val;
        size_t count{};
    };

    using TBins = TVector<TBin> ;

    void Clear() {
        for (auto & h : hist)
            h.count = 0;
        more = 0;
        total = 0;
    }

    TVector<size_t> GetValues() const {
        TVector<size_t> res;
        res.reserve(hist.size() + 1);
        for (const auto & h : hist) {
            res.emplace_back(h.count);
        }
        res.emplace_back(more);

        return res;
    }

    void Add(const typename TTypeTraits<T>::TFuncParam value) {
        total++;
        auto it = std::upper_bound(hist.begin(), hist.end(), value);
        if (hist.end() == it) {
            more++;
        } else {
            it->count++;
        }
    }

    template <size_t N>
    explicit THist(const T (&levels)[N]) {
        for (size_t i = 0; i < N; i++)
            hist.emplace_back(levels[i]);
        std::sort(hist.begin(), hist.end());
    }

    THist(std::initializer_list<T> list) {
        for(auto & item : list)
            hist.emplace_back(std::forward<T>(item));
        std::sort(hist.begin(), hist.end());
    }

    explicit THist(const TVector<T>& levels) {
        for(const auto & l : levels)
            hist.emplace_back(l);
        std::sort(hist.begin(), hist.end());
    }

    const TBins& GetBins() const {
        return hist;
    }

    size_t GetTotal() const {
        return total;
    }

    size_t GetMore() const {
        return more;
    }

    bool operator==(const THist &rhs) const {
        return hist == rhs.hist &&
               more == rhs.more &&
               total == rhs.total;
    }

private:
    TBins hist{};
    size_t more{};
    size_t total{};
};
