#pragma once

#include <util/stream/output.h>
#include <util/generic/bt_exception.h>

enum class TLimitedOutputOverflowPolicy{
    Silent,
    ThrowOnOverflow
};

template<TLimitedOutputOverflowPolicy OverflowPolicy>
class TLimitedOutput: public IOutputStream {
public:
    size_t Availiable() const noexcept {
        return Limit_ - Count_;
    }

    size_t Limit() const noexcept {
        return Limit_;
    }

    bool Exceed() const noexcept {
        return Limit_ == Count_;
    }

    void Reset() noexcept {
        Count_ = 0;
    }

    TLimitedOutput(IOutputStream& slave, size_t limit) noexcept
            : Slave_(slave)
            , Count_()
            , Limit_(limit) {}

    ~TLimitedOutput() override = default;

    TLimitedOutput(TLimitedOutput&&) noexcept = default;
    TLimitedOutput& operator=(TLimitedOutput&&) noexcept = default;

private:
    size_t SafeWrite(const void* buf, size_t len) {
        const auto toWrite = Min(Limit_ - Count_, len);
        if(toWrite) {
            Slave_.Write(buf, toWrite);
            Count_ += toWrite;
        }
        return toWrite;
    }

    void DoWrite(const void* buf, size_t len) override {
        const auto written = SafeWrite(buf, len);
        if constexpr (TLimitedOutputOverflowPolicy::ThrowOnOverflow == OverflowPolicy)
            if(written < len)
                ythrow TWithBackTrace<yexception>() << "exceed limit " << Limit_ << " on " << (len - written);
    }

private:
    IOutputStream& Slave_;
    size_t Count_;
    const size_t Limit_;
};
