#pragma once

#include <array>
#include <ctime>
#include <stdexcept>
#include <util/ysaveload.h>
#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/generic/xrange.h>
#include <util/system/rwlock.h>
#include <util/thread/pool.h>
#include <util/string/util.h>
#include <util/stream/file.h>
#include <util/generic/algorithm.h>

template <class item_t> class TShardedItems {
public:
    size_t Size() const { return items.size(); }

    template <class KeyT> item_t& getShard(const KeyT& val) noexcept {
        return items[val % items.size()];
    }
    template <class KeyT> const item_t& getShard(const KeyT& val) const noexcept {
        return items[val % items.size()];
    }

    bool operator==(const TShardedItems<item_t> & sharded) const {
        return items == sharded.items;
    }

    Y_SAVELOAD_DEFINE(items)

    explicit TShardedItems() {}
    explicit TShardedItems(size_t size) : items(size) {}

    TShardedItems(TShardedItems<item_t> &&) noexcept = default;
private:
    TVector<item_t> items;
};

namespace NCache {
    template <class KeyT, class ValueT> class TItem {
    public:
        typedef TItem<KeyT, ValueT> SelfType;
        typedef THashMap<KeyT, ValueT> ConT;

        struct TShard{
            inline void Save(IOutputStream* s) const {
                TReadGuard g(mutex);
                ::Save(s, data);
            }
            inline void Load(IInputStream* s) {
                TWriteGuard g(mutex);
                ::Load(s, data);
            }

            bool operator==(const TShard & another) const {
                return data == another.data;
            }

            TShard() = default;
            TShard(TShard && another) noexcept : data(std::move(another.data)) {}
            mutable TRWMutex mutex;
            ConT data;
        };

        struct TChangerStat {
            bool insertedNew{};
            TChangerStat() = default;
        };

        struct TChanger {
            virtual void Change(const KeyT& key, ValueT& val) noexcept = 0;
            virtual ~TChanger() = default;
        };

        TChangerStat Apply(const KeyT& key, TChanger&& changer) {
            TChangerStat stats;

            auto& shard = m_sharded_data.getShard(key);
            TWriteGuard guard(shard.mutex);

            auto it = shard.data.find(key);
            if (it == shard.data.end()) {
                it = shard.data.emplace(key, ValueT());
                stats.insertedNew = true;
            }

            changer.Change(key, it->second);

            return stats;
        }

        size_t Size() const {
            size_t size = 0;
            for (size_t i = 0; i < m_sharded_data.Size(); i++) {
                const auto& shard = m_sharded_data.getShard(i);
                TReadGuard guard(shard.mutex);
                size += shard.data.size();
            }
            return size;
        }

        bool Get(const KeyT& key, ValueT& val) const {
            const auto& shard = m_sharded_data.getShard(key);
            TReadGuard guard(shard.mutex);
            auto it = shard.data.find(key);
            if (it == shard.data.end())
                return false;

            val = it->second;
            return true;
        }

        bool Add(const KeyT& key, ValueT val) {
            auto& shard = m_sharded_data.getShard(key);
            TWriteGuard guard(shard.mutex);

            shard.data[key] = std::move(val);

            return true;
        }

        bool Exist(const KeyT& key) const {
            const auto& shard = m_sharded_data.getShard(key);
            TReadGuard guard(shard.mutex);
            return shard.data.contains(key);
        }

        void Remove(const KeyT& key) {
            auto& shard = m_sharded_data.getShard(key);
            TWriteGuard guard(shard.mutex);
            shard.data.erase(key);
        }

        bool operator==(const SelfType & another) const {
            return m_sharded_data == another.m_sharded_data;
        }

        Y_SAVELOAD_DEFINE(m_sharded_data)
        explicit TItem(size_t size = 31) : m_sharded_data(size) {}

    private:
        TShardedItems<TShard> m_sharded_data;
    };

    class TStat {
    public:
        ui32 GetChangeTime() const {
            return m_change_time;
        }
        ui32 GetLastCangeTime() const {
            return m_last_change_time;
        }
        ui32 GetLocalCount() const {
            return m_local_count;
        }
        ui32 GetRemainTime() const {
            return m_remain_time;
        }
        ui32 GetPrevLocalCount() const {
            return m_local_prev_count;
        }

        TStat()
            : m_change_time(0)
            , m_last_change_time(0)
            , m_remain_time(0)
            , m_local_count(0)
            , m_local_prev_count(0)
        {}
        TStat(ui32 m_change_time, ui32 m_last_change_time, ui32 m_remain_time, ui32 m_local_count, ui32 m_local_prev_count)
            : m_change_time(m_change_time)
            , m_last_change_time(m_last_change_time)
            , m_remain_time(m_remain_time)
            , m_local_count(m_local_count)
            , m_local_prev_count(m_local_prev_count)
        {}

    protected:
        ui32 m_change_time;
        ui32 m_last_change_time;
        ui32 m_remain_time;
        ui32 m_local_count;
        ui32 m_local_prev_count;
    };

    struct TConfig{
        TConfig & SetDuration(TDuration v) {
            duration = v;
            return *this;
        }
        TConfig & SetShardsNum(size_t v) {
            shardsNum = v;
            return *this;
        }

        TDuration duration = TDuration::Seconds(600);
        size_t shardsNum = 31;
    };

    template <class KeyT, class ValueT, size_t StepsCount = 2> class TCacheBase {
        static_assert(StepsCount >= 2, "incorrect step count");
    public:
        using TItem = TItem<KeyT, ValueT>;
        using TItemPtr = std::shared_ptr<TItem>;
        using TSelf = TCacheBase<KeyT, ValueT, StepsCount>;

        typename TItem::TChangerStat Apply(const KeyT& key, typename TItem::TChanger&& changer) {
            return std::atomic_load(&steps[0])->Apply(key, std::forward<TItem::TChanger>(changer));
        }

        const TDuration& GetTTL() const {
            return ttl;
        }

        size_t Size() const {
            size_t size = 0;
            for(auto & step : steps) {
                size += std::atomic_load(&step)->Size();
            }
            return size;
        }

        void Cleanup() {
            lastTouch.store(Now());
            for (size_t i = StepsCount - 1; i > 0; i--) {
                std::atomic_store(&steps[i], std::atomic_load(&steps[i - 1]));
            }
            std::atomic_store(&steps.front(), std::make_shared<TItem>(shardsNum));
        }

        void EventTick() {
            if (Expired())
                Cleanup();
        }

        TDuration TimeExpiredSinceLastTouch() const {
            return Now() - lastTouch.load();
        }

        bool Expired() const {
            return Now() > lastTouch.load() + ttl;
        }

        bool Add(const KeyT& key, const ValueT& val) {
            std::atomic_load(&steps.front())->Add(key, val);
            return true;
        }

        bool Add(const KeyT& key, ValueT && val) {
            auto local = std::atomic_load(&steps.front());
            local->Add(key, std::move(val));
            return true;
        }

        bool Get(const KeyT& key, ValueT& val) const {
            return AnyOf(steps, [&key, &val](const TItemPtr & step) {
                return std::atomic_load(&step)->Get(key, val);
            });
        }

        void Remove(const KeyT& key) {
            for(auto & step : steps)
                std::atomic_load(&step)->Remove(key);
        }

        void Clear() {
            for(auto & step : steps) {
                std::atomic_load(&step)->Clear();
            }
        }

        bool Exist(const KeyT& key) const {
            return AnyOf(steps, [&key](const TItemPtr & step) {
                return std::atomic_load(&step)->Exist(key);
            });
        }

        void Init(const TDuration& duration) {
            ttl = TDuration::MilliSeconds(duration.MilliSeconds() / StepsCount);
        }

        TStat GetStat() const {
            return {
                    static_cast<ui32>(ttl.Seconds()),
                    static_cast<ui32>(lastTouch.load().Seconds()),
                    static_cast<ui32>((ttl - TimeExpiredSinceLastTouch()).Seconds()),
                    static_cast<ui32>(std::atomic_load(&steps[0])->Size()),
                    static_cast<ui32>(std::atomic_load(&steps[1])->Size())
            };
        }

        bool operator==(const TSelf & another) const {
            if(lastTouch.load() != another.lastTouch.load() ||
                    ttl != another.ttl ||
                    shardsNum != another.shardsNum)
                return false;

            for(size_t i : xrange(StepsCount))
                if(!(*std::atomic_load(&steps[i]) == *std::atomic_load(&another.steps[i])))
                    return false;

            return true;
        }

        explicit TCacheBase(const TDuration& duration = TDuration::Seconds(600), size_t shardsNum = 31) : shardsNum(shardsNum) {
            lastTouch.store(Now());
            Init(duration);
            for(TItemPtr & step : steps)
                std::atomic_store(&step, std::make_shared<TItem>(shardsNum));
        }

        explicit TCacheBase(const TConfig & config) : shardsNum(config.shardsNum) {
            lastTouch.store(Now());
            Init(config.duration);
            for(TItemPtr & step : steps)
                std::atomic_store(&step, std::make_shared<TItem>(shardsNum));
        }

        void Save(IOutputStream* s) const {
            ::Save(s, shardsNum);
            ::Save(s, ttl);
            ::Save(s, lastTouch.load());
            for(auto & step : steps)
                ::Save(s, *std::atomic_load(&step));
        }
        void Load(IInputStream* s) {
            {
                size_t shardsNumTmp{};
                ::Load(s, shardsNumTmp);
                if(shardsNumTmp != shardsNum)
                    return;
            }
            {
                TDuration ttlTmp{};
                ::Load(s, ttlTmp);
                if(ttlTmp != ttl)
                    return;
            }
            {
                TInstant ls;
                ::Load(s, ls);
                lastTouch.store(ls);
            }
            for(size_t i : xrange(StepsCount)) {
                TItemPtr item = std::make_shared<TItem>(shardsNum);
                item->Load(s);
                std::atomic_store(&steps[i], std::move(item));
            }
        }
    private:
        size_t shardsNum;
        TDuration ttl;
        std::array<TItemPtr, StepsCount> steps;
        std::atomic<TInstant> lastTouch{};
    };

    struct TVoid {};

    template<class KeyT>
    using TVoidCache = TCacheBase<KeyT, TVoid>;
} //namespace NCache

