#include "parsers.h"

#include <util/stream/str.h>

#include <library/cpp/charset/codepage.h>
#include <library/cpp/langs/langs.h>
#include <util/string/strip.h>
#include <util/string/subst.h>

#include "sputil.h"

TStringBuf Parseout(TStringBuf sSource, const TStringBuf& sBegin, const TStringBuf& sEnd, size_t offset) {
    sSource.Skip(offset);
    if(!sBegin.StartsWith('&') || !sSource.SkipPrefix(sBegin.substr(1))) {
        const auto beginPos = sSource.find(sBegin);
        if (beginPos == TStringBuf::npos)
            return {};

        sSource.Skip(beginPos + sBegin.size());
    }
    return sSource.substr(0,  sSource.find(sEnd));
}

void CheckForSelfSentMessage(const TString & mfrm, const TString & rcpt, bool* bSameEmail, bool* bSameDomain, int* rcptCnt) {
    TString sPureMfrmEmail = mfrm.substr(0, mfrm.find_first_of(" \n\r"));
    ToLower(sPureMfrmEmail.begin(), sPureMfrmEmail.size());
    size_t atPos = sPureMfrmEmail.find("@");
    TString sPureMfrmDomain = atPos == TString::npos ? sPureMfrmEmail : sPureMfrmEmail.substr(atPos + 1);

    bool bOtherEmailFound = false;
    bool bOtherDomainFound = false;
    int rcpts = 0;

    char* buf = strdup(rcpt.c_str());
    char* tok;
    TString sTok;
    TString sPureRcptEmail;
    TString sPureRcptDomain;
#ifdef _MSC_VER
    for (tok = strtok(buf, ";"); tok; tok = strtok(nullptr, ";"))
#else
    char* brk;
    for (tok = strtok_r(buf, ";", &brk); tok; tok = strtok_r(nullptr, ";", &brk))
#endif

        // sender email found twice in rcptos list;
        // 1st for fake-addr for WMI Sent folder,
        // 2nd for actual delivery

    {
        sTok = tok;
        ToLower(sTok.begin(), sTok.size());
        sPureRcptEmail = sTok.substr(0, sTok.find_first_of(" ;"));
        atPos = sPureRcptEmail.find("@");
        sPureRcptDomain = atPos == TString::npos ? sPureRcptEmail : sPureRcptEmail.substr(atPos + 1);

        if (sPureRcptEmail != sPureMfrmEmail) {
            bOtherEmailFound |= true;
            rcpts++;
        }
        bOtherDomainFound |= sPureRcptDomain != sPureMfrmDomain;
    }

    if (buf)
        free(buf);

    *bSameEmail = !bOtherEmailFound;
    *bSameDomain = !bOtherDomainFound;

    if (rcptCnt)
        *rcptCnt = rcpts;
}

TTextSplitter::TTextSplitter(const char* clear_utf8_text, size_t len, size_t maxWords)
        : maxWords(maxWords)
        , wordsNumber()
        , text(clear_utf8_text)
        , start(reinterpret_cast<const unsigned char*>(text))
        , end(reinterpret_cast<const unsigned char*>(text) + len)
{
}

bool TTextSplitter::FindNextWord() {
    if (wordsNumber < maxWords && tryFindWord()) {
        wordsNumber++;
        return true;
    }
    return false;
}

const TTextSplitter::TSplitResult& TTextSplitter::Get() const {
    return current;
}

bool TTextSplitter::tryFindWord() {
    current = {};

    while (start < end) {
        wchar32 c = 0;
        if (ReadUTF8CharAndAdvance(c, start, end) != RECODE_OK) {
            ++start;
            continue;
        } else if (IsAlnum(c)) {
            current.word += c;
            const bool is_digit = IsDigit(c);
            current.has_2_or_more_digits |= current.has_digit && is_digit;
            current.has_digit |= is_digit;

            switch (ScriptByGlyph(c)) {
                case SCRIPT_LATIN:
                    current.has_latin = true;
                    break;
                case SCRIPT_CYRILLIC:
                    current.has_cyrillic = true;
                    break;
                default:
                    break;
            }

            continue;
        }

        if (!current.word.empty())
            return true;
    }

    return !current.word.empty();
}

TFrom::TFrom(TStringBuf from) {
    if (TStringBuf name, addr; from.TryRSplit(' ', name, addr)) {
        Name = StripString(name);
        Addr = FixAddress(StripString(addr));
    } else {
        Addr = FixAddress(StripString(from));
    }
}

TString FixAddress(const TStringBuf& addr) {
    TString res = to_lower(TString{addr});
    SubstGlobal(res, "<", "");
    SubstGlobal(res, ">", "");
    SubstGlobal(res, "\'", "");
    SubstGlobal(res, " @", "");

    {
        const auto plusPos = res.find('+');
        if(plusPos != TString::npos) {
            const auto delim = res.find_first_of("@;", plusPos + 1);
            res.remove(plusPos, delim == TString::npos ? TString::npos : delim - plusPos);
        }

    }

    if(res.EndsWith('.'))
        res.remove(res.size() - 1);

    return StripString(res);
}

IOutputStream& operator<<(IOutputStream& s, const TFrom& from) {
    return s << from.Name << ' ' << from.Addr;
}
