#include "phone_parser.h"
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/ptr.h>
#include <util/string/cast.h>
#include <library/cpp/charset/codepage.h>
#include <library/cpp/charset/recyr.hh>
#include <iostream>
#include <vector>

static const CodePage& cp_koi8 = *CodePageByCharset(CODES_KOI8);

bool TPhoneParser::TParseResult::IsMobile() const {
    return (flags & TPhoneParser::IsMobile);
}

const TString& TPhoneParser::TParseResult::GetPhone() const { // телефон, очищенный от всего лишнего
    return phone;
}

const TString& TPhoneParser::TParseResult::GetCountry() const {
    return country;
}

const TString& TPhoneParser::TParseResult::GetCity() const {
    return city;
}

const TString& TPhoneParser::TParseResult::GetCountryCode() const {
    return country_code;
}

const TString& TPhoneParser::TParseResult::GetCityCode() const {
    return city_code;
}

const TString& TPhoneParser::TParseResult::GetOperatorName() const {
    return opsos;
}

const unsigned char* TPhoneParser::TParseResult::OriginalPhoneStart() const {
    return original_phone_start;
}

const unsigned char* TPhoneParser::TParseResult::OriginalPhoneEnd() const {
    return original_phone_end;
}

TString TPhoneParser::TParseResult::OriginalPhone() const {
    if (original_phone_start == nullptr || original_phone_end == nullptr)
        return TString();
    return TString((char*)original_phone_start, (char*)original_phone_end);
}

ui32 TPhoneParser::TParseResult::GetWordsReplaceNum() const {
    return words_replace_cnt;
}

ui32 TPhoneParser::TParseResult::GetCharsReplaceNum() const {
    return char_replace_cnt;
}

TPhoneParser::TDigitGroupItem::TDigitGroupItem()
    : Offset(0)
    , Len(0)
    , GotPrefix(false)
    , GotPlus(false)
    , SeqStart(nullptr)
    , SeqEnd(nullptr)
    , WordsReplaceCount(0)
    , CharsReplaceCount(0)
    , LetterCount(0)
{ }

TPhoneParser::city_code::city_code(TString city_code) noexcept
    : code(std::move(city_code))
    , is_mobile(false)
{ }

TPhoneParser::city_code::city_code(TString city_code, TString city_name, bool is_mob) noexcept
    : code(std::move(city_code))
    , name(std::move(city_name))
    , is_mobile(is_mob)
{ }

bool TPhoneParser::city_code::operator==(const city_code& cc) const {
    return (cc.code == code);
}

TPhoneParser::country_info::country_info(TString code) noexcept
    : country_code(std::move(code))
{ }

TPhoneParser::country_info::country_info(TString code, ui8 len, TString name, TCityCodesPtr codes, bool plus_needed) noexcept
    : country_code(std::move(code))
    , phone_len(len)
    , country_name(std::move(name))
    , city_codes(std::move(codes))
    , must_have_plus(plus_needed)
{ }

bool TPhoneParser::country_info::operator==(const country_info& ci) const {
    return (ci.country_code == country_code);
}

TPhoneParser::context_t::context_t()
    : trash_symbol_counter(0)
    , digit_counter(0)
    , last_phone_end(nullptr)
    , prefix_start(nullptr)
    , need_more_data(false)
    , word_replacements_num(0)
    , char_replacements_num(0)
    , drop_group(false)
    , letter_counter(0)
{ }

TPhoneParser::TPhoneParser()
    : LastString(nullptr)
    , CallCounter(0)
{
    FillReplacementsList();
    FillSymbolTypes();
    FillPhoneCodes();
}

void TPhoneParser::FillSymbolTypes() {
    SymbolType.fill(0);
    symbol_to_digit_table.fill(0);

    for (ui8 chr = '0'; chr <= '9'; ++chr)
        SymbolType[chr] = Digit;

    for (ui8 chr = 'a'; chr <= 'z'; ++chr)
        SymbolType[chr] = Letter;

    for (ui8 chr = 'A'; chr <= 'Z'; ++chr)
        SymbolType[chr] = Letter;

    for (ui32 i = 192; i <= 255; ++i) //russian symbols
        SymbolType[i] = Letter;
    SymbolType[163] = Letter; //"ё"
    SymbolType[179] = Letter; //"Ё"

    //                                      0        1     2     3     4     5   6
    static const char* MayBeADigit[] = { "oOоОQ", "!iIl", "zZ", "зЗ", "чЧ", "", "б"};

    auto begin = std::begin(MayBeADigit);
    for (auto it = begin; it != std::end(MayBeADigit); ++it) {
        const size_t digit = std::distance(begin, it);
        const TString text = Recode(CODES_UTF8, CODES_KOI8, *it);
        for (unsigned char val: text) {
            SymbolType[val] |= LooksLikeDigit;
            symbol_to_digit_table[val] = static_cast<ui8>('0') + digit;
        }
    }
}

bool TPhoneParser::IsLetter(ui8 chr) const {
    return (SymbolType[chr] & Letter);
}

bool TPhoneParser::IsDigit(ui8 chr) const {
    return (SymbolType[chr] & Digit);
}

bool TPhoneParser::IsLooksLikeDigit(ui8 chr) const {
    return (SymbolType[chr] & LooksLikeDigit);
}

void TPhoneParser::FillReplacementsList() {
    const std::pair<TString, TString> replacemnet_info[] = {
        {"ноль",   "0"},
        {"один",   "1"},
        {"два",    "2"},
        {"три",    "3"},
        {"четыре", "4"},
        {"пять",   "5"},
        {"шесть",  "6"},
        {"семь",   "7"},
        {"восемь", "8"},
        {"девять", "9"},
        {"плюс",   "+"},
        {"plus",   "+"},

        {"десять",       "10"},
        {"одинадцать",   "11"},
        {"двенадцать",   "12"},
        {"тринадцать",   "13"},
        {"четырнадцать", "14"},
        {"пятнадцать",   "15"},
        {"шестнадцать",  "16"},
        {"семнадцать",   "17"},
        {"восемнадцать", "18"},
        {"девятнадцать", "19"},

        {"двадцать",    "20"},
        {"тридцать",    "30"},
        {"сорок",       "40"},
        {"пятьдесят",   "50"},
        {"шестьдесят",  "60"},
        {"семьдесят",   "70"},
        {"восемьдесят", "80"},
        {"девяносто",   "90"},

        {"сто",       "100"},
        {"двести",    "200"},
        {"триста",    "300"},
        {"четыреста", "400"},
        {"пятьсот",   "500"},
        {"шестьсот",  "600"},
        {"семьсот",   "700"},
        {"восемьсот", "800"},
        {"девятьсот", "900"},

        {"тысяча",   "1000"}
    };

    for (const auto& item: replacemnet_info)
        ReplacementsDict[Recode(CODES_UTF8, CODES_KOI8, item.first)] = item.second;
}

const TPhoneParser::prefix_container TPhoneParser::Prefixes = {
    Recode(CODES_UTF8, CODES_KOI8, "т"),
    Recode(CODES_UTF8, CODES_KOI8, "тел"),
    Recode(CODES_UTF8, CODES_KOI8, "телеф"),
    Recode(CODES_UTF8, CODES_KOI8, "телефон"),
    Recode(CODES_UTF8, CODES_KOI8, "телефоны"),
    Recode(CODES_UTF8, CODES_KOI8, "телефона"),
    Recode(CODES_UTF8, CODES_KOI8, "телефону"),
    Recode(CODES_UTF8, CODES_KOI8, "телефонам"),
    Recode(CODES_UTF8, CODES_KOI8, "fax"),
    Recode(CODES_UTF8, CODES_KOI8, "факс"),
    Recode(CODES_UTF8, CODES_KOI8, "факсу"),
    Recode(CODES_UTF8, CODES_KOI8, "факсам"),
    Recode(CODES_UTF8, CODES_KOI8, "факсы"),
    Recode(CODES_UTF8, CODES_KOI8, "tel"),
    Recode(CODES_UTF8, CODES_KOI8, "phone"),
    Recode(CODES_UTF8, CODES_KOI8, "telephone"),
    Recode(CODES_UTF8, CODES_KOI8, "mob"),
    Recode(CODES_UTF8, CODES_KOI8, "mobile"),
    Recode(CODES_UTF8, CODES_KOI8, "моб"),
    Recode(CODES_UTF8, CODES_KOI8, "мобильн"),
    Recode(CODES_UTF8, CODES_KOI8, "мобильный"),
    Recode(CODES_UTF8, CODES_KOI8, "звонить"),
    Recode(CODES_UTF8, CODES_KOI8, "звони"),
    Recode(CODES_UTF8, CODES_KOI8, "звоните"),
};

bool TPhoneParser::IsPrefix(const TString& word) {
    if (word.empty())
        return false;
    return Prefixes.contains(word);
}

void TPhoneParser::FillPhoneCodes() {
    TCityCodesPtr rus_codes(new TCityCodes({
        {"343", "Yekaterinburg", false},
        {"495", "Moscow", false},
        {"499", "Moscow", false},

        {"812", "St. Petersburg", false},
        {"831", "Nizhny Novgorod", false},
        {"843", "Kazan", false},
        {"863", "Rostov-on-Don", false},

        {"903", "Beeline", true},
        {"906", "Beeline", true},
        {"909", "Beeline", true},
        {"913", "MTS", true},
        {"916", "MTS", true},
        {"925", "Megafon", true},
        {"926", "Megafon", true},
    }));

    TCityCodesPtr ukr_codes(new TCityCodes({
        {"44", "Kiev", false}
    }));

    TCityCodesPtr bel_codes(new TCityCodes({
        {"25", "life", true},
        {"29", "Velcom, MTS or Diallog", true},
        {"33", "MTS", true},
        {"17", "Minsk", false}
    }));

    TCityCodesPtr nigeria_codes(new TCityCodes({
        {"1", "Lagos", false },
    }));

    default_country = rus_codes;
    TCityCodesPtr dummy(new TCityCodes());

    TCountriesInfo({
        {"007", 13, "RU", rus_codes, false}, //Russia
        {"7", 11, "RU", rus_codes, false},   //Russia
        {"8", 11, "RU", rus_codes, false},
        {"380", 12, "UA", ukr_codes, true},     //Ukraine
        {"375", 12, "BY", bel_codes, true},     //Belarus
        {"234", 13, "NG", nigeria_codes, true}, //Nigeria
        {"27", 14, "ZA", dummy, true},          //South Africa
        {"44", 12, "UK", dummy, true},          //United Kingdom
        {"66", 11, "TH", dummy, true},          //Thailand
        {"226", 11, "BF", dummy, true},         //Burkina Faso
        {"00226", 13, "BF", dummy, false},
        {"971", 13, "AE", dummy, true}, //United Arab Emirates
        {"86", 13, "CN", dummy, false}, //China
    }).swap(country_codes);
}

bool TPhoneParser::ReplaceWord(TString& word, context_t& context) const {
    if (word.empty())
        return false;
    replacements_container::const_iterator it = ReplacementsDict.find(word);
    if (it != ReplacementsDict.end()) {
        word = it->second;
        if (word != "+")
            context.word_replacements_num++;
    }
    return (it != ReplacementsDict.end());
}

ui32 TPhoneParser::CalcScore(ui32 flags) {
    return (flags & GotCountryCode ? 1 : 0) + (flags & GotRegionCode ? 1 : 0) + (flags & GotStandartPrefix ? 1 : 0) + (flags & StartsWithPlus ? 1 : 0) + (flags & MatchPhoneMask ? 1 : 0);
}

TPhoneParser::TCountriesInfo::const_iterator TPhoneParser::FindCountry(const TStringBuf& phone) const {
    ui32 country_code_lengths[] = {5, 3, 2, 1};

    for (ui32 length: country_code_lengths) {
        auto country = country_codes.find(country_info(TString(phone.substr(0, length))));
        if (country != country_codes.end())
            return country;
    }
    return country_codes.end();
}

TPhoneParser::TCityCodes::const_iterator TPhoneParser::FindCity(const TStringBuf& phone, const TCityCodes& region_codes, ui32 shift) {
    TCityCodes::const_iterator cit = region_codes.end();
    if (phone.size() >= 4 + shift) {
        cit = region_codes.find(city_code(TString(phone.substr(shift, 3))));
        if (cit == region_codes.end())
            cit = region_codes.find(city_code(TString(phone.substr(shift, 4))));
    }
    return cit;
}

size_t TPhoneParser::CalcPhoneLen(const TStringBuf& phone, TParseResult* res, bool got_plus) const {
    ui32 country_code_shift = 0;
    size_t phone_len = static_cast<size_t>(-1);

    bool got_country_code = false;
    bool got_region_code = false;

    TCountriesInfo::const_iterator country = FindCountry(phone);
    if (country != country_codes.end()) {
        if ((country->must_have_plus && got_plus) || !country->must_have_plus) {
            got_country_code = true;
            phone_len = country->phone_len;
            country_code_shift = country->country_code.size();
        } else {
            country = country_codes.end();
        }
    }

    TCityCodes& region_codes = (country != country_codes.end()) ? *(country->city_codes) : *default_country;

    TCityCodes::const_iterator cit = FindCity(phone, region_codes, country_code_shift);
    if (cit != region_codes.end())
        got_region_code = true;

    if ((country_code_shift == 1) && !got_region_code) {
        cit = FindCity(phone, region_codes, 0);
        if (cit != region_codes.end()) {
            got_country_code = false;
            got_region_code = true;
        } else {
            ui32 code_val = FromString<ui32>(phone.substr(1, 3));
            //no such region code in Russia
            if ((code_val < 300) || (code_val > 599 && code_val < 800)) {
                got_country_code = false;
                got_region_code = false;
            }
        }
    }

    res->flags = (got_country_code ? GotCountryCode : 0) |
                 (got_region_code ? GotRegionCode : 0);
    if (got_country_code) {
        res->country = country->country_name;
        res->country_code = country->country_code;
    }

    if (got_region_code) {
        ((cit->is_mobile) ? res->opsos : res->city) = cit->name;
        if (cit->is_mobile)
            res->flags |= IsMobile;
        res->city_code = cit->code;
    }

    // assume that phone is Russian but w/o +7
    if (!got_country_code && got_region_code)
        phone_len = 10;

    return phone_len;
}

bool TPhoneParser::PhoneLenIsFine(ui32 desired_len, TDigitGroupItem_arr::const_iterator it, TDigitGroupItem_arr::const_iterator end, TParseResult& res, std::vector<std::pair<int, TParseResult>>& possible_results) {
    static constexpr size_t pattern_size = 3;
    std::array<ui32, pattern_size> pattern = { {3, 2, 2} }; // for phones like xxx-xx-xx
    std::array<ui32, pattern_size> phone_pattern = { {0, 0, 0} };
    std::array<bool, 11> possible_lenghts = { { 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1 } };

    ui32 idx = 0;
    ui32 len = 0;
    ui32 letter_count = 0;
    res.words_replace_cnt = 0;
    res.char_replace_cnt = 0;
    for (; it != end && len < desired_len; it++) {
        phone_pattern[idx++ % pattern_size] = it->Len;
        len += it->Len;
        res.words_replace_cnt += it->WordsReplaceCount;
        res.char_replace_cnt += it->CharsReplaceCount;
        res.original_phone_end = it->SeqEnd;
        res.letter_count = letter_count;
        letter_count += it->LetterCount;

        bool phone_matches = true;
        for (ui32 i = 0; i < pattern_size; ++i)
            phone_matches &= (pattern[i] == phone_pattern[(idx + i) % pattern_size]);

        if (phone_matches)
            res.flags |= MatchPhoneMask;
        else
            res.flags &= ~MatchPhoneMask;

        if ((len < possible_lenghts.size()) && possible_lenghts[len])
            possible_results.emplace_back(len, res);
    }

    bool phone_matches = true;
    for (ui32 i = 0; i < 3; ++i) {
        phone_matches &= (pattern[i] == phone_pattern[idx++ % 3]);
    }
    if (phone_matches)
        res.flags |= MatchPhoneMask;

    return (len == desired_len);
}

//try to assemble correct phone from this digits group
TPhoneParser::TParseResults TPhoneParser::AssemblePhones(const TDigitGroup& gr, const unsigned char** last_phone_end) const {
    TParseResults results;
    bool prev_phone_had_prefix = false;
    bool prev_phone_had_code = false;

    ui32 range_end = 0;
    ui32 prev_score = 0;
    if (gr.data.empty() || gr.items.empty())
        return TParseResults();
    for (auto i = gr.items.begin(); i != gr.items.end(); ++i) {
        if (gr.data.size() - i->Offset < PhoneLenMin)
            break;
        const TStringBuf line = TStringBuf(gr.data).substr(i->Offset);
        TParseResult res;
        res.flags = 0;
        size_t len = CalcPhoneLen(line, &res, i->GotPlus);
        res.flags |= ((i->GotPrefix || (prev_phone_had_prefix && (i->Offset >= range_end))) ? GotStandartPrefix : 0); // if previous phone had standart prefix assume that next phone after it got prefix too

        res.flags |= ((prev_phone_had_code && (i->Offset >= range_end)) ? GotRegionCode : 0); // if previous phone had region code assume that next phone after it got code too

        res.flags |= (i->GotPlus ? StartsWithPlus : 0);

        std::vector<std::pair<int, TParseResult>> possible_results;
        bool phone_len_is_ok = PhoneLenIsFine(len, i, gr.items.end(), res, possible_results);

        //this seems to be a phone, but we dont know it's len so we'll have to guess it
        if (!phone_len_is_ok && ((res.flags & GotStandartPrefix) || prev_phone_had_code)) {
            if (!possible_results.empty()) {
                ui32 text_len = (gr.data.size() - i->Offset);
                size_t idx = possible_results.size() - 1;
                len = possible_results[idx].first;
                ui32 phone_chars_left = text_len - len;
                while (idx != 0 && !(phone_chars_left == 0 || phone_chars_left >= PhoneLenMin)) {
                    --idx;
                    len = possible_results[idx].first;
                    phone_chars_left = text_len - len;
                }

                res = possible_results[idx].second;
                phone_len_is_ok = true;
            }
        }

        // good phone is a phone with correct length and with at least one sign of beeing a phone
        ui32 score = CalcScore(res.flags);
        if (score > 0)
            score -= ((res.letter_count > 2) ? 1 : 0);

        if (phone_len_is_ok && score > 0) {
            res.phone.assign(line.substr(0, len));
            res.original_phone_start = i->SeqStart;
            bool better_phone_found = (i->Offset < range_end) && (score > prev_score);
            if (better_phone_found) {
                assert(!results.empty());
                results.pop_back();
            }

            if ((i->Offset >= range_end) || better_phone_found) {
                range_end = i->Offset + len;
                prev_score = score;
                results.push_back(res);
                prev_phone_had_prefix = res.flags & GotStandartPrefix;
                prev_phone_had_code = res.flags & GotRegionCode;
                *last_phone_end = res.original_phone_end;
            }
        }
    }
    return results;
}

void TPhoneParser::ResetDigitGroupItem(TDigitGroupItem& item) {
    item.GotPlus = false;
    item.GotPrefix = false;
    item.WordsReplaceCount = 0;
    item.CharsReplaceCount = 0;
}

TPhoneParser::TParseResults TPhoneParser::GetPhones(const TStringBuf& text) {
    const unsigned char* start = (unsigned char*)text.data();
    const unsigned char* end = start + text.size();
    prev_res.clear();
    ProcessString(start, end, end);
    return prev_res;
}

TPhoneParser::TParseResults TPhoneParser::ProcessSymbol(unsigned char symb, ptrdiff_t it, context_t& context, const unsigned char* real_it) const {
    const unsigned char*& last_phone_end = context.last_phone_end;
    TDigitGroup& group = context.group;
    TDigitGroupItem& digit_seq = context.digit_seq;

    ui32& trash_symbol_counter = context.trash_symbol_counter;
    ui32& digit_counter = context.digit_counter;
    ptrdiff_t& last_plus_pos = context.last_plus_pos;
    ptrdiff_t& last_prefix_pos = context.last_prefix_pos;

    if (symb == '+') {
        last_plus_pos = it;
        context.plus_start = real_it;
    }

    if (!IsDigit(symb) && !IsLooksLikeDigit(symb) || context.drop_group) {
        ++trash_symbol_counter;
        context.drop_group = false;
        if (digit_counter) {
            digit_seq.Len = digit_counter;
            digit_seq.SeqEnd = real_it;
            digit_seq.WordsReplaceCount = context.word_replacements_num;
            digit_seq.CharsReplaceCount = context.char_replacements_num;
            digit_seq.LetterCount = context.letter_counter;
            context.letter_counter = 0;
            context.word_replacements_num = 0;
            context.char_replacements_num = 0;
            group.items.push_back(digit_seq);
            ResetDigitGroupItem(digit_seq);
        }
        digit_counter = 0;
    }

    TParseResults results;
    if (trash_symbol_counter > TrashSymbolsNumMax) {
        results = AssemblePhones(group, &last_phone_end);

        digit_counter = 0;
        ResetDigitGroupItem(digit_seq);
        context.word_replacements_num = 0;
        context.char_replacements_num = 0;
        group.items.clear();
        group.data.clear();
        trash_symbol_counter = 0;
        context.need_more_data = false;
        if (symb == '\0')
            return results;
    }

    if (IsDigit(symb) || IsLooksLikeDigit(symb)) {
        if (digit_counter == 0) {
            digit_seq.SeqStart = real_it;
            digit_seq.Offset = group.data.size();

            int dist_to_plus = it - last_plus_pos;
            if (((ui32)dist_to_plus < TrashSymbolsNumMax) && (dist_to_plus > 0)) {
                digit_seq.GotPlus = true;
                digit_seq.SeqStart = context.plus_start;
            }

            int dist_to_prefix = it - last_prefix_pos;
            if ((ui32(dist_to_prefix) < 1.8 * TrashSymbolsNumMax) && (dist_to_prefix > 0)) {
                digit_seq.GotPrefix = true;
                digit_seq.SeqStart = context.prefix_start;
            }
        }

        if (IsLooksLikeDigit(symb)) {
            context.char_replacements_num++;
            group.data.append(symbol_to_digit_table[(ui8)symb]);
        } else {
            group.data.append(symb);
        }
        ++digit_counter;
        context.need_more_data = ((digit_counter < PhoneLenMax) && (group.data.size() < 5 * PhoneLenMax));

        trash_symbol_counter = 0;
        if (digit_counter == 1 && !group.items.empty())
            group.items.back().LetterCount += context.letter_counter;

        context.letter_counter = 0;
    }

    if (symb == '\0' && digit_seq.SeqStart != nullptr) {
        digit_seq.Len = digit_counter;
        digit_seq.SeqEnd = real_it;
        group.items.push_back(digit_seq);
        results = AssemblePhones(group, &last_phone_end);
    }
    return results;
}

const unsigned char* TPhoneParser::ProcessString(const unsigned char* start,
                                                 const unsigned char* end,
                                                 const unsigned char* buff_end) {
    assert(end - start >= 0);

    TString word;
    TParseResults results;
    context_t context;
    ResetDigitGroupItem(context.digit_seq);

    context.last_plus_pos = buff_end - start;
    context.last_prefix_pos = buff_end - start;

    ptrdiff_t insert_pos = 0;
    unsigned char symbol;

    const unsigned char* it;
    bool word_need_check = false;
    bool word_is_actually_a_digit = true;
    const unsigned char* word_start = nullptr;
    for (it = start; it != buff_end; ++it) {
        if (IsLetter(*it)) {
            if (word.empty())
                word_start = it;
            word.append(cp_koi8.ToLower(*it));
            word_is_actually_a_digit &= IsLooksLikeDigit(*it);
            if (it != buff_end - 1)
                continue;
            else
                word_need_check = ReplaceWord(word, context);
        } else {
            word_need_check = ReplaceWord(word, context);
        }

        if (IsPrefix(word)) {
            word_need_check = false;
            context.last_prefix_pos = insert_pos + word.size();
            context.prefix_start = it - word.size();
        }

        if (word_need_check || word_is_actually_a_digit) {
            for (TString::const_iterator word_it = word.begin(); word_it != word.end(); ++word_it) {
                symbol = cp_koi8.ToLower(*word_it);
                results = ProcessSymbol(symbol, insert_pos, context, word_start);
                copy(results.begin(), results.end(), back_inserter(prev_res));
                ++insert_pos;
            }
        } else {
            insert_pos += word.size();
            context.trash_symbol_counter += word.size();
            context.drop_group = true;
            context.letter_counter += word.size();
        }

        if ((it == buff_end - 1) && !word.empty()) {
            break;
        } else {
            symbol = *it;
            results = ProcessSymbol(symbol, insert_pos, context, it);
            copy(results.begin(), results.end(), back_inserter(prev_res));
            ++insert_pos;
        }

        if (!context.need_more_data && (it >= end))
            break;
        word.clear();
        word_is_actually_a_digit = true;
    }

    symbol = '\0';
    results = ProcessSymbol(symbol, insert_pos, context, it);
    copy(results.begin(), results.end(), back_inserter(prev_res));
    return context.last_phone_end;
}

bool TPhoneParser::GetPhone(const unsigned char* buff_start,          // начало буфера со строкой
                            const unsigned char* first_byte_to_check, //первый символ с которого надо начать проверку, возможно залезу назад в пределах радиуса
                            const unsigned char* buff_end,            // конец строки
                            TParseResult& res)                        // найденный телефон
{
    assert((first_byte_to_check >= buff_start) && (first_byte_to_check <= buff_end));
    if (++CallCounter >= MaxPhoneParserCalls)
        return false;

    const unsigned char* start = first_byte_to_check - CheckRadius;
    start = ((start > buff_start) ? start : buff_start);
    const unsigned char* end = first_byte_to_check + CheckRadius;
    end = ((end > buff_end) ? buff_end : end);

    bool already_processed_this = ((LastString == buff_start) && (first_byte_to_check < CacheEnd));
    if (already_processed_this) {
        if (prev_res.empty()) {
            return false;
        } else {
            res = prev_res.front();
            prev_res.erase(prev_res.begin());
            return true;
        }
    }

    if (LastString != buff_start) {
        CacheEnd = buff_start;
        LastString = buff_start;
        prev_res.clear();
    } else if (start < CacheEnd) {
        start = CacheEnd;
    }
    const unsigned char* last_phone_end = ProcessString(start, end, buff_end);

    CacheEnd = last_phone_end ? last_phone_end : first_byte_to_check + CheckRadius / 2;
    if (CacheEnd > buff_end)
        CacheEnd = buff_end;

    if (!prev_res.empty()) {
        res = prev_res.front();
        prev_res.erase(prev_res.begin());
        return true;
    }

    return false;
}

void TPhoneParser::NewMessage() {
    CallCounter = 0;
    LastString = nullptr;
}
