#pragma once

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/ptr.h>

#include <vector>
#include <deque>
#include <array>

class TPhoneParser {
public:
    // Выходная структура
    class TParseResult {
    public:
        bool IsMobile() const;
        const TString& GetPhone() const;
        const TString& GetCountry() const;
        const TString& GetCity() const;

        const TString& GetCountryCode() const;
        const TString& GetCityCode() const;
        const TString& GetOperatorName() const;

        const unsigned char* OriginalPhoneStart() const;
        const unsigned char* OriginalPhoneEnd() const;
        TString OriginalPhone() const;

        ui32 GetWordsReplaceNum() const;
        ui32 GetCharsReplaceNum() const;

    private:
        ui32 flags;
        ui32 words_replace_cnt;
        ui32 char_replace_cnt;
        TString phone;
        TString country;
        TString city;
        TString country_code;
        TString city_code;
        TString opsos;
        ui32 letter_count;

        const unsigned char* original_phone_start;
        const unsigned char* original_phone_end;

        friend class TPhoneParser;
    };
    typedef std::vector<TParseResult> TParseResults;

public:
    static const size_t MaxPhoneParserCalls = 100;

public:
    TPhoneParser();
    TParseResults GetPhones(const TStringBuf& text);
    bool GetPhone(const unsigned char* buff_start, const unsigned char* first_byte_to_check, const unsigned char* buff_end, TParseResult& res);
    void NewMessage();

private:
    enum PhoneFlags {
        GotCountryCode = 0x1,    // there is a country code in found phone
        GotRegionCode = 0x2,     // city or mobile operator code found (495 916, etc)
        GotStandartPrefix = 0x4, // phone got one of standart prefixes (tel: phone: call: etc...)
        StartsWithPlus = 0x8,    // phone starts with plus (+7 926 111 22 33)
        IsMobile = 0x10,
        MatchPhoneMask = 0x20 // phone conforms pattern like +x (xxx) xxx xx xx
    };

    enum SymbolTypes {
        Letter = 0x1,
        Digit = 0x2,
        LooksLikeDigit = 0x4
    };

    struct TDigitGroupItem {
        TDigitGroupItem();

        size_t Offset;
        size_t Len;
        bool GotPrefix;
        bool GotPlus;
        const unsigned char* SeqStart;
        const unsigned char* SeqEnd;
        ui32 WordsReplaceCount;
        ui32 CharsReplaceCount;
        ui32 LetterCount;
    };
    typedef std::deque<TDigitGroupItem> TDigitGroupItem_arr;

    struct TDigitGroup {
        TString data;
        TDigitGroupItem_arr items;
    };

    struct hasher;

    struct city_code {
        explicit city_code(TString city_code) noexcept;
        city_code(TString city_code, TString city_name, bool is_mob) noexcept;
        bool operator==(const city_code& cc) const;

        TString code;
        TString name;
        bool is_mobile;
    };
    typedef THashSet<city_code, hasher> TCityCodes;
    typedef TSimpleSharedPtr<TCityCodes> TCityCodesPtr;

    struct country_info {
        explicit country_info(TString code) noexcept;
        country_info(TString code, ui8 len, TString name, TCityCodesPtr codes, bool plus_needed) noexcept;
        bool operator==(const country_info& ci) const;

        TString country_code;
        ui8 phone_len{};
        TString country_name;
        TCityCodesPtr city_codes;
        bool must_have_plus{};
    };

    struct hasher {
        ui64 operator()(const city_code& cc) const {
            return ComputeHash(cc.code);
        }
        ui64 operator()(const country_info& ci) const {
            return ComputeHash(ci.country_code);
        }
    };

    typedef THashSet<TString> prefix_container;
    typedef THashSet<country_info, hasher> TCountriesInfo;
    typedef THashMap<TString, TString> replacements_container;

    struct context_t {
        context_t();

        TDigitGroup group;
        TDigitGroupItem digit_seq;

        ui32 trash_symbol_counter;
        ui32 digit_counter;
        ptrdiff_t last_plus_pos{};
        ptrdiff_t last_prefix_pos{};
        const unsigned char* last_phone_end;
        const unsigned char* prefix_start;
        const unsigned char* plus_start{};
        bool need_more_data;
        ui32 word_replacements_num;
        ui32 char_replacements_num;
        bool drop_group;
        ui32 letter_counter;
    };

private:
    static const ui32 TrashSymbolsNumMax = 5;
    static const size_t PhoneLenMax = 13;
    static const size_t PhoneLenMin = 5;

private:
    //cache stuff
    const unsigned char* LastString;
    ui32 CallCounter;
    const unsigned char* CacheEnd{};
    const static ptrdiff_t CheckRadius = 100; // radius of phones search

    static const prefix_container Prefixes;
    replacements_container ReplacementsDict;
    std::array<ui8, 256> symbol_to_digit_table{};
    std::array<ui8, 256> SymbolType{};
    TCountriesInfo country_codes;
    TCityCodesPtr default_country;
    TParseResults prev_res;

private:
    void FillReplacementsList();
    void FillSymbolTypes();
    void FillPhoneCodes();

    bool IsLetter(ui8 chr) const;
    bool IsDigit(ui8 chr) const;
    bool IsLooksLikeDigit(ui8 chr) const;

    static bool IsPrefix(const TString& word) ;
    bool ReplaceWord(TString& word, context_t& context) const;

    static ui32 CalcScore(ui32 flags) ;
    size_t CalcPhoneLen(const TStringBuf& phone, TParseResult* res, bool got_plus) const;
    static bool PhoneLenIsFine(ui32 desired_len, TDigitGroupItem_arr::const_iterator it, TDigitGroupItem_arr::const_iterator end, TParseResult& res, std::vector<std::pair<int, TParseResult>>& possible_results) ;
    TParseResults AssemblePhones(const TDigitGroup& gr, const unsigned char** last_phone_end) const;
    static void ResetDigitGroupItem(TDigitGroupItem& item) ;

    TCountriesInfo::const_iterator FindCountry(const TStringBuf& phone) const;
    static TCityCodes::const_iterator FindCity(const TStringBuf& phone, const TCityCodes& region_codes, ui32 shift) ;

    const unsigned char* ProcessString(const unsigned char* start, const unsigned char* end, const unsigned char* buff_end);
    TParseResults ProcessSymbol(unsigned char symb, ptrdiff_t it, context_t& context, const unsigned char* real_it) const;
};
