/*
 * prof.cpp
 *
 *  Created on: 22 дек. 2016 г.
 *      Author: luckybug
 */

#include <util/string/ascii.h>
#include <util/generic/list.h>
#include <util/generic/algorithm.h>
#include <util/string/builder.h>
#include "prof.h"

namespace NProf {
    void Profiler::CookLog(NJsonWriter::TBuf& jsonWriter, const Profiler& prof) {
        TVector<std::pair<std::reference_wrapper<const TString>, std::reference_wrapper<const Item>>> links(Reserve(prof.profs.size()));

        for (const auto& [name, item] : prof.profs) {
            if (item.GetDuration().MilliSeconds() == 0)
                continue;

            links.emplace_back(name, item);
        }

        Sort(links, [](const auto& p1, const auto& p2) {
            return AsciiCompareIgnoreCase(p1.first.get(), p2.first.get()) < 0;
        });

        jsonWriter.BeginObject();
        for (const auto& [name, itemRef] : links) {
            const Item& item = itemRef;
            if(item.GetCallsCount() == 1)
                jsonWriter.WriteKey(name.get()).WriteULongLong(item.GetDuration().MilliSeconds());
            else
                jsonWriter.WriteKey(name.get()).WriteString(TStringBuilder{} << item.GetDuration().MilliSeconds() << ':' << item.GetCallsCount());
        }

        for (const auto& subProf : prof.subProfs) {
            jsonWriter.WriteKey(subProf.first);
            CookLog(jsonWriter, subProf.second);
        }
        jsonWriter.EndObject();
    }

    IOutputStream& operator<<(IOutputStream& ss, const Profiler& prof) {
        {
            NJsonWriter::TBuf buf(NJsonWriter::HEM_DONT_ESCAPE_HTML, &ss);
            Profiler::CookLog(buf, prof);
        }
        return ss;
    }

    Item& Profiler::Prof(const TString& key) {
        return profs[key];
    }

    Profiler& Profiler::Sub(const TString& key) {
        return subProfs[key];
    }

    void Profiler::SetSub(const TString& key, Profiler&& p) {
        subProfs.emplace(key, std::move(p));
    }

    void Profiler::Extend(Profiler&& p) {
        for (auto& prof : p.profs)
            profs.emplace(std::move(prof));
        for (auto& subProf : p.subProfs)
            subProfs.emplace(std::move(subProf));
    }

    TDuration Profiler::GetItemDuration(const TString& itemName) const {
        profs_t::const_iterator it = profs.find(itemName);

        if (it == profs.end())
            throw yexception() << itemName << " not found in profs";

        return it->second.GetDuration();
    }

    Item& Item::operator+=(const Item& p) {
        duration += p.duration;
        callsCount += p.callsCount;
        return *this;
    }

    TDuration Item::GetDuration() const {
        return duration;
    }

    TInstant Item::GetStart() const {
        return start;
    }

    TInstant Item::GetEnd() const {
        return stop;
    }

    size_t Item::GetCallsCount() const {
        return callsCount;
    }

    void Item::Start() {
        start = Now();
    }

    void Item::Stop() {
        if (start == TInstant::Zero())
            return;

        callsCount++;

        stop = Now();
        duration += stop - start;

        start = TInstant::Zero();
    }
} // namespace NProf
