#pragma once

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/datetime/base.h>
#include <util/stream/str.h>
#include <library/cpp/json/writer/json.h>

namespace NProf {
    class Item {
    public:
        Item() noexcept = default;
        Item(const Item&) noexcept = default;
        bool operator<(const Item& item) const {
            return stop < item.stop;
        }
        bool operator<(TInstant t) const {
            return stop < t;
        }
        Item& operator+=(const Item& p);
        TDuration GetDuration() const;
        TInstant GetStart() const;
        TInstant GetEnd() const;
        size_t GetCallsCount() const;
        void Start();
        void Stop();

    private:
        TDuration duration = TDuration::Zero();
        TInstant start = TInstant::Zero();
        TInstant stop = TInstant::Zero();
        size_t callsCount{};
    };

    class TNamedItem : public Item {
    public:
        TNamedItem(const TNamedItem&) = default;
        const TString& GetName() const {
            return Name;
        }
        explicit TNamedItem(TString name) noexcept
            : Name(std::move(name)) {
        }

    private:
        TString Name;
    };

    class Profiler {
    public:
        friend IOutputStream& operator<<(IOutputStream& m, const Profiler& item);
        static void CookLog(NJsonWriter::TBuf& jsonWriter, const Profiler& prof);
        Item& Prof(const TString& key);
        Profiler& Sub(const TString& key);
        void SetSub(const TString& key, Profiler&& p);
        void Extend(Profiler&& p);

        TDuration GetItemDuration(const TString& itemName) const;

        Profiler() = default;
        Profiler(Profiler&&) noexcept = default;

    private:
        typedef THashMap<TString, Profiler> sub_profs_t;
        typedef THashMap<TString, Item> profs_t;
        sub_profs_t subProfs;
        profs_t profs;
    };
    using TGuard = TGuard<Item>;
} // namespace NProf

typedef NProf::Item CProfItem;
typedef NProf::Profiler CProf;

template <>
struct TCommonLockOps<CProfItem> {
    static inline void Acquire(CProfItem* t) noexcept {
        t->Start();
    }

    static inline void Release(CProfItem* t) noexcept {
        t->Stop();
    }
};

template <>
struct TCommonLockOps<NProf::TNamedItem> {
    static inline void Acquire(NProf::TNamedItem* t) noexcept {
        t->Start();
    }

    static inline void Release(NProf::TNamedItem* t) noexcept {
        t->Stop();
    }
};

typedef TGuard<CProfItem> CProfItemGuard;

template <class T = void>
class TResWithProf : public std::pair<T, CProf> {
public:
    using TBase = std::pair<T, CProf>;
    using TBase::TBase;
};

template <>
class TResWithProf<void> {
public:
    CProf prof;
};
