#include "receipt.h"
#include "sobase64.h"
#include "parsers.h"
#include "safe_recode.h"
#include <library/cpp/json/writer/json.h>
#include <library/cpp/logger/priority.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <mail/so/libs/syslog/so_log.h>

TString TReceipt::ToJson() const {
    NJsonWriter::TBuf js;
    {
        js.BeginObject();
        {
            js.WriteKey("timestamp").WriteULongLong(Timestamp.Seconds());
            js.WriteKey("data_size").WriteInt(DataSize);
            js.WriteKey("fromname").WriteString(From.GetName());
            js.WriteKey("fromaddr").WriteString(From.GetAddr());
            js.WriteKey("form_id").WriteString(FormId);
        }
        js.EndObject();
    }
    return js.Str();
}

TString TReceipt::CompressAndBase64() const {
    const auto js = ToJson();

    if (TString compressed, err; ::CompressAndBase64(js.c_str(), js.size(), compressed, err)) {
        return compressed;
    } else {
        throw TWithBackTrace<yexception>() << "error in compress receipt: " << err;
    }
}

std::pair<TReceipt, NJson::TJsonValue> TReceipt::Unbase64AndDecompress(const TStringBuf &data) {
    bool warningSize;
    TString result, error;
    if (!::Unbase64AndUncompress(data.data(), data.size(), warningSize, result, error)) {
        throw TWithBackTrace<yexception>() << "error in compress receipt: " << error;
    }

    const NJson::TJsonValue json = NJson::ReadJsonFastTree(result, false);
    return {FromJson(json), json};
}

TReceipt TReceipt::MakeReceipt(const TCgiParameters &cgi) {
    size_t realReceiptSize = 0;
    {
        const TString receipt = [&cgi] {
            NJsonWriter::TBuf test;
            test.BeginList();
            for (const auto&[key, params] : cgi) {
                test.BeginObject();
                test.WriteKey("k").WriteString(key);
                test.WriteKey("v").WriteString(params);
                test.EndObject();
            }
            test.EndList();
            return test.Str();
        }();

        if (TString compressed, err; ::CompressAndBase64(receipt.c_str(), receipt.size(), compressed, err)) {
            realReceiptSize = compressed.size();
        } else {
            Syslog(TLOG_ERR) << "error in compress receipt: " << err;
        }
    }

    return TReceipt(Now(),
                    realReceiptSize,
                    SafeRecode(cgi.Get("from"), CODES_WIN),
                    cgi.Get("form_id"));
}

TReceipt TReceipt::FromJson(const NJson::TJsonValue &json) {
    TReceipt receipt;
    if (const NJson::TJsonValue *value = json.GetValueByPath("timestamp")) {
        receipt.Timestamp = TInstant::Seconds(value->GetUIntegerRobust());
    }
    if (const NJson::TJsonValue *value = json.GetValueByPath("data_size")) {
        receipt.DataSize = value->GetUIntegerRobust();
    }

    {
        TString fromname, fromaddr;
        if (const NJson::TJsonValue *value = json.GetValueByPath("fromname")) {
            fromname = value->GetStringSafe();
        }
        if (const NJson::TJsonValue *value = json.GetValueByPath("fromaddr")) {
            fromaddr = value->GetStringSafe();
        }
        receipt.From = TFrom(std::move(fromname), std::move(fromaddr));
    }
    if (const NJson::TJsonValue *value = json.GetValueByPath("form_id")) {
        receipt.FormId = value->GetStringSafe();
    }
    return receipt;
}

IOutputStream &operator<<(IOutputStream &stream, const TReceipt &receipt) {
    return stream << "t:" << receipt.Timestamp
                  << ",s:" << receipt.DataSize
                  << ",from:" << receipt.From
                  << ",form_id:" << receipt.FormId
                  ;
}