#pragma once

#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/generic/yexception.h>
#include <util/generic/bt_exception.h>
#include <util/generic/variant.h>
#include <util/stream/str.h>

#include <utility>

namespace NRegexp {

    enum class EMatchResult {
        Match,
        NotMatch
    };

    class TError : public TWithBackTrace<yexception> {};

    using TMatches = TVector<TStringBuf>;

    struct TResult {
        TResult() = default;

        TResult(TStringBuf text, NRegexp::TMatches matches) noexcept
            : Text(text)
            , Matches(std::move(matches)) {
        }

        bool Empty() const {
            return Matches.empty();
        }

        template <class T>
        bool GetPattern(size_t ind, T& pnum) {
            if (auto match = GetPattern(ind)) {
                return TryFromString(match, pnum);
            }
            return false;
        }

        TStringBuf GetPattern(size_t ind) const {
            if (ind >= Matches.size())
                return nullptr;

            return Matches[ind];
        }

        bool GetPattern(size_t ind, TStringBuf& pattern) const {
            return bool(pattern = GetPattern(ind));
        }

        TStringBuf GetRestPattern() const {
            if (auto pattern = GetPattern(0)) {
                return {pattern.cend(), Text.cend()};
            }
            return nullptr;
        }

        std::pair<TStringBuf, TStringBuf> GetTextWithoutPattern(size_t ind) const {
            if (auto match = GetPattern(ind)) {
                TStringBuf until;
                {
                    const auto untilPatternOffset = std::distance(Text.cbegin(), match.cbegin());
                    if (untilPatternOffset > 0)
                        until = Text.substr(0, (size_t)untilPatternOffset);
                }

                TStringBuf after;
                {
                    const auto afterPatternOffset = std::distance(match.cend(), Text.cbegin() + Text.size());
                    if (afterPatternOffset > 0)
                        after = Text.substr(Text.size() - afterPatternOffset);
                }

                return {until, after};
            }
            return {};
        }

    private:
        TStringBuf Text;
        NRegexp::TMatches Matches;
    };

    class IExpression {
    public:
        [[nodiscard]] virtual EMatchResult Match(const TStringBuf &text) const = 0;

        virtual EMatchResult Match(const TStringBuf &text, size_t maxN, TMatches &matches) const = 0;

        TResult MatchAndGetResult(const TStringBuf &text, size_t maxN) const {
            if(TMatches matches; Match(text, maxN, matches) == EMatchResult::Match)
                return TResult(text, std::move(matches));
            return {};
        }

        virtual ~IExpression() = default;
    };
}
