#include "so_answer.h"
#include <util/stream/str.h>
#include <util/stream/format.h>
#include <util/string/join.h>
#include <util/string/builder.h>

using namespace google::protobuf;
using namespace mail::so::api::v1;

TCheckedMessage::TAsLegacy TCheckedMessage::AsLegacy(bool printPersonalReslutions) const {
    return {*this, printPersonalReslutions};
}

TCheckedMessage::TUidsResolutionsPrinter TCheckedMessage::UidsPrinter() const {
    return TUidsResolutionsPrinter(*this);
}

inline SoResolution SpClassToSoResolution(TSpClass spClass) {
    using namespace mail::so::api::v1;
    switch (spClass) {
        case TSpClass::UNKNOWN:
        case TSpClass::DLVR:
        case TSpClass::HAM:
            return SoResolution::SO_RESOLUTION_ACCEPT;
        case TSpClass::SPAM:
            return SoResolution::SO_RESOLUTION_SPAM;
        case TSpClass::MALIC:
            return SoResolution::SO_RESOLUTION_REJECT;
    }
}

SoResponse TCheckedMessage::AsSoResponse(bool addDlvLog, const TLog& logger) const {
    SoResponse response;

    response.set_resolution(SpClassToSoResolution(spClass));

    response.set_deny_graylist(DenyGreyListing);

    {
        OutParameters* outParameters = response.mutable_out_parameters();
        outParameters->set_foreign_mx(ForeignZone && ForeignZone != "RU");
        if (MXCode == "GRE") {
            outParameters->set_forward_type(FORWARD_TYPE_GRAY);
        } else if (MXCode == "WHI") {
            outParameters->set_forward_type(FORWARD_TYPE_WHITE);
        } else if (MXCode == "BLA") {
            outParameters->set_forward_type(FORWARD_TYPE_MXBACK);
        }
    }

    {
        RepeatedPtrField<ActivityInfo>* activityInfos = response.mutable_activity_infos();

        for(const auto& [uid, days]: UidsActivities) {
            ActivityInfo* activityInfo = activityInfos->Add();
            activityInfo->mutable_uid()->set_value(IntFromString<ui64, 10>(uid));
            activityInfo->set_activity_status(days);
        }
    }

    {
        RepeatedPtrField<TString>* soClasses = response.mutable_so_classes();

        for(TString soClass : SoClass) {
            soClasses->Add(std::move(soClass));
        }
    }

    {
        RepeatedPtrField<PersonalResolution>* personalResolutions = response.mutable_personal_resolutions();
        for (const auto& [uid, spClass] : UidsResolutions) {
            PersonalResolution* personalResolution = personalResolutions->Add();

            personalResolution->set_resolution(SpClassToSoResolution(spClass));

            RepeatedPtrField<TString>* soClasses = personalResolution->mutable_so_classes();

            for(TString soClass : SoClass) {
                soClasses->Add(std::move(soClass));
            }

            if(uid) {
                personalResolution->mutable_uid()->set_value(IntFromString<ui64, 10>(uid));
                if (UidsWithPf.contains(uid)) {
                    soClasses->Add(IsHam(spClass) ? TString("pf_ham") : TString("pf_spam"));
                }
            } else {
                logger << "meet empty uid while making SoResponse";
            }
        }
    }

    if(addDlvLog) {
        RepeatedPtrField<DeliveryLog>* deliveryLogs = response.mutable_delivery_logs();

        for(const TDeque<std::pair<TString, TString>>& dlvLog : DlvLog) {
            DeliveryLog* deliveryLog = deliveryLogs->Add();
            RepeatedPtrField<StringPair>* logItems = deliveryLog->mutable_log_items();
            for(const auto& [key, value]: dlvLog) {
                StringPair* pair = logItems->Add();
                pair->set_key(key);
                pair->set_value(value);
            }
        }
    }

    return response;
}

void TCheckedMessage::SetResHamAndAllNotPfResSpam() {
    spClass = TSpClass::HAM;
    for(auto& [uid, resolution]: UidsResolutions) {
        if(!UidsWithPf.contains(uid)) {
            resolution = TSpClass::SPAM;
        }
    }
}

void TCheckedMessage::CombineWith(TCheckedMessage&& another) {
    if(spClass != another.spClass) {
        if (Uids.size() == UidsWithPf.size() &&
            another.Uids.size() != another.UidsWithPf.size())
        {
            // All current uids resolutions set by PF
            // Better take new CheckedMessage resolution as more universal
            spClass = another.spClass;
        }
    }

    UidsActivities.insert(std::make_move_iterator(another.UidsActivities.begin()),
                          std::make_move_iterator(another.UidsActivities.end()));
    UidsResolutions.insert(std::make_move_iterator(another.UidsResolutions.begin()),
                           std::make_move_iterator(another.UidsResolutions.end()));
    SoClass.insert(std::make_move_iterator(another.SoClass.begin()),
                   std::make_move_iterator(another.SoClass.end()));
    SuidsWithPf.insert(std::make_move_iterator(another.SuidsWithPf.begin()),
                       std::make_move_iterator(another.SuidsWithPf.end()));
    UidsWithPf.insert(std::make_move_iterator(another.UidsWithPf.begin()),
                      std::make_move_iterator(another.UidsWithPf.end()));

    ShouldLogForKaspersky |= another.ShouldLogForKaspersky;
    DenyGreyListing |= another.DenyGreyListing;

    Uids.insert(Uids.cend(),
                std::make_move_iterator(another.Uids.begin()),
                std::make_move_iterator(another.Uids.end()));

    Suids.insert(Suids.cend(),
        std::make_move_iterator(another.Suids.begin()),
        std::make_move_iterator(another.Suids.end()));

    DlvLog.insert(DlvLog.cend(),
                  std::make_move_iterator(another.DlvLog.begin()),
                  std::make_move_iterator(another.DlvLog.end()));
    Mailish |= another.Mailish;
    TempFail |= another.TempFail;
}

IOutputStream& operator<<(IOutputStream& stream, const TCheckedMessage::TAsLegacy& asLegacy) {
    TStringStream resolution;

    const TCheckedMessage& checked = asLegacy.Master;

    resolution << TCheckedMessage::REJECT_TAG << ' ' << checked.Reject << '\n'
               << TCheckedMessage::REJECTSTR_TAG << ' ' << checked.RejectStr << '\n';
    {
        resolution << TCheckedMessage::SPAM_TAG << ' ' << int(IsSpam(checked.spClass));

        {
            bool firstInverted = true;
            bool firstPf = true;
            for (const TSuid &suid : checked.Suids) {
                if (checked.SuidsWithPf.contains(suid)) {
                    if(firstInverted) {
                        resolution << ' ';
                        firstInverted = false;
                    }
                    if(firstPf) {
                        resolution << ':';
                        firstPf = false;
                    } else {
                        resolution << ',';
                    }
                    resolution << suid;
                }
            }
        }

        resolution << '\n';
    }

    if(asLegacy.PrintPersonalResolutions) {
        resolution << TCheckedMessage::SPAMPERS_TAG << " 0 " << int(IsSpam(checked.spClass)) << ' '
                   << MakeRangeJoiner(" ", checked.SoClass);

        for(const TUid& uid : checked.Uids) {
            if(const TSpClass* spClass = MapFindPtr(checked.UidsResolutions, uid)) {
                resolution << ", " << uid << ' ' << int(IsSpam(*spClass)) << ' '
                           << MakeRangeJoiner(" ", checked.SoClass);
                if(checked.UidsWithPf.contains(uid)) {
                    resolution << ' ' << (IsHam(*spClass) ? TStringBuf("pf_ham") : TStringBuf("pf_spam"));
                }
            }
        }
        resolution << '\n';
    }

    resolution << TCheckedMessage::SPAMSTR_TAG << " \n";

    resolution << "X-Spam-Ystatus: hits=" << Prec(checked.Report.Hits, PREC_POINT_DIGITS, 1) << '\n';
    resolution << "X-Spam-Flag: " << checked.spClass << '\n';
    resolution << "X-Spam-Yversion: Spamooborona-4.3\n";

    for(const TUid& uid : checked.Uids) {
        if(const int* daysSinceLastActivity = MapFindPtr(checked.UidsActivities, uid)) {
            resolution << "X-Yandex-UID-Active: " << uid << ' ' << *daysSinceLastActivity << '\n';
        }
    }

    if (checked.SoClass) {
        resolution << "X-Yandex-SO-Class: " << MakeRangeJoiner(",", checked.SoClass) << '\n';
    }

    if (checked.DenyGreyListing && IsSpam(checked.spClass)) {
        resolution << "X-Yandex-DenyGL: 1\n";
    }

    if (checked.ForeignZone) {
        resolution << "X-Yandex-ForeignMX: " << checked.ForeignZone << '\n';
    }

    if(checked.MXCode) {
        resolution << "X-Yandex-MXCode: " << checked.MXCode << '\n';
    }

    return stream << TCheckedMessage::ANSWER_TAG << ' '
                  << LeftPad(resolution.Size(), 5, ' ') << '\n'
                  << resolution.Str();
}

IOutputStream& operator<<(IOutputStream& stream, const TCheckedMessage::TUidsResolutionsPrinter& checked) {
    for (const auto& [uid, res] : checked.Master.UidsResolutions) {
        stream << uid << ':' << res << ';';
    }
    return stream;
}

[[nodiscard]] NJson::TJsonValue TCheckedMessage::TUidsResolutionsPrinter::AsJson() const {
    NJson::TJsonValue value;
    for (const auto& [uid, res] : Master.UidsResolutions) {
        value[uid] = TStringBuilder{} << res;
    }
    return value;
}
