#include "sobase64.h"

#include <contrib/libs/lzo2/include/lzo/lzo1x.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/generic/vector.h>
#include <util/memory/tempbuf.h>
#include <util/generic/yexception.h>

#define UNCOMPRESS_SUSPICIOUS_SIZE 5242880

namespace {
    TString LZO_E_to_string(int e) {
        switch (e) {
            case LZO_E_OK:
                return "OK";
            case LZO_E_ERROR:
                return "ERROR";
            case LZO_E_OUT_OF_MEMORY:
                return "OUT_OF_MEMORY";
            case LZO_E_NOT_COMPRESSIBLE:
                return "NOT_COMPRESSIBLE";
            case LZO_E_INPUT_OVERRUN:
                return "INPUT_OVERRUN";
            case LZO_E_OUTPUT_OVERRUN:
                return "OUTPUT_OVERRUN";
            case LZO_E_LOOKBEHIND_OVERRUN:
                return "LOOKBEHIND_OVERRUN";
            case LZO_E_EOF_NOT_FOUND:
                return "EOF_NOT_FOUND";
            case LZO_E_INPUT_NOT_CONSUMED:
                return "INPUT_NOT_CONSUMED";
            case LZO_E_NOT_YET_IMPLEMENTED:
                return "NOT_YET_IMPLEMENTED";
            case LZO_E_INVALID_ARGUMENT:
                return "INVALID_ARGUMENT";
            case LZO_E_INVALID_ALIGNMENT:
                return "INVALID_ALIGNMENT";
            case LZO_E_OUTPUT_NOT_CONSUMED:
                return "OUTPUT_NOT_CONSUMED";
            case LZO_E_INTERNAL_ERROR:
                return "INTERNAL_ERROR";
            default:
                ythrow yexception() << "unknown error code: " << e;
        }
    }

    //#define LZO_E_OK                    0
    //#define LZO_E_ERROR                 (-1)
    //#define LZO_E_OUT_OF_MEMORY         (-2)    /* [lzo_alloc_func_t failure] */
    //#define LZO_E_NOT_COMPRESSIBLE      (-3)    /* [not used right now] */
    //#define LZO_E_INPUT_OVERRUN         (-4)
    //#define LZO_E_OUTPUT_OVERRUN        (-5)
    //#define LZO_E_LOOKBEHIND_OVERRUN    (-6)
    //#define LZO_E_EOF_NOT_FOUND         (-7)
    //#define LZO_E_INPUT_NOT_CONSUMED    (-8)
    //#define LZO_E_NOT_YET_IMPLEMENTED   (-9)    /* [not used right now] */
    //#define LZO_E_INVALID_ARGUMENT      (-10)
    //#define LZO_E_INVALID_ALIGNMENT     (-11)   /* pointer argument is not properly aligned */
    //#define LZO_E_OUTPUT_NOT_CONSUMED   (-12)
    //#define LZO_E_INTERNAL_ERROR        (-99)
}

static const bool LzoInitOk = lzo_init() == LZO_E_OK;

bool LzoCompress(const char* instr, size_t len, TVector<ui8>& result, TString& errstr) {
    if (!LzoInitOk) {
        errstr = "Can not init LZO-engine";
        return false;
    }

    if (len == 0) {
        return true;
    }

    char workMem[LZO1X_999_MEM_COMPRESS + 1];
    const lzo_uint outlen = len + len / 64 + 16 + 3;
    result.resize(5 + outlen);

    result[0] = 0xf0;
    result[1] = (unsigned char)((len >> 24) & 0xff);
    result[2] = (unsigned char)((len >> 16) & 0xff);
    result[3] = (unsigned char)((len >> 8) & 0xff);
    result[4] = (unsigned char)(len & 0xff);

    lzo_uint newlen = outlen;
    auto e = lzo1x_999_compress(reinterpret_cast<const unsigned char*>(instr), len, &result[5], &newlen, workMem);
    if (e != LZO_E_OK) {
        errstr = TString("Error while compressing data: ") + LZO_E_to_string(e);
        return false;
    }
    if (newlen > outlen) {
        errstr = "Wrong predicted length of compressed data";
        return false;
    }

    result.resize(5 + newlen);

    return true;
}

bool LzoUncompress(const ui8* in, size_t len, TString& result, TString& errstr) {
    if (!LzoInitOk) {
        errstr = "Can not init LZO-engine";
        return false;
    }

    if (len == 0) {
        result = "";
        return true;
    }

    if (in[0] < 0xF0 || in[0] > 0xF1) {
        errstr = "Incorrect algorithm in LZO-compressed data";
        return false;
    }

    lzo_uint outlen = ui32(in[1]) << 24 | ui32(in[2]) << 16 | ui32(in[3]) << 8 | ui32(in[4]);
    if (outlen == 0) {
        result = "";
        return true;
    }

    char workMem[LZO1X_999_MEM_COMPRESS + 1];
    result.resize(outlen + 1);
    lzo_uint newlen = outlen;

    int err = lzo1x_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, workMem);
    if (err != LZO_E_OK) {
        errstr = "Error while uncompressing data: " + LZO_E_to_string(err);
        ;
        return false;
    }
    if (outlen != newlen) {
        errstr = "Wrong length in LZO-compressed message";
        return false;
    }

    result.erase(newlen);

    return true;
}

bool CompressAndBase64(const char* instr, size_t len, TString& result, TString& err, TString (*Base64EncodeFunc)(TStringBuf)) {
    TVector<ui8> compressed;
    if (!LzoCompress(instr, len, compressed, err))
        return false;

    if (compressed.empty()) {
        return true;
    }

    try{
        result = Base64EncodeFunc({(const char*)&compressed[0], compressed.size()});
        return true;
    } catch(const yexception & e) {
        return false;
    }
}

bool CompressAndBase64(const char* instr, size_t len, TString& result, TString& err) {
    return CompressAndBase64(instr, len, result, err, Base64Encode);
}

bool CompressAndBase64Url(const char* instr, size_t len, TString& result, TString& err) {
    return CompressAndBase64(instr, len, result, err, Base64EncodeUrl);
}

bool Unbase64AndUncompress(const char* instr, size_t len, bool& warning_size, TString& result, TString& err) {
    warning_size = false;

    if (len >= UNCOMPRESS_SUSPICIOUS_SIZE) {
        err = "Buffer too big > UNCOMPRESS_SUSPICIOUS_SIZE";
        warning_size = true;
        return false;
    }

    result.clear();

    if (len == 0) {
        return true;
    }

    try{
        const TString outstr = Base64DecodeUneven({instr, instr + len});

        if (!LzoUncompress((const ui8*)outstr.c_str(), outstr.size(), result, err)) {
            err = TString(instr) + " " + err;
            return false;
        }

        if (result.size() >= UNCOMPRESS_SUSPICIOUS_SIZE) {
            err = "Buffer too big > UNCOMPRESS_SUSPICIOUS_SIZE";
            warning_size = true;
            return false;
        }
        return true;
    } catch (const yexception & e) {
        err = e.AsStrBuf();
        return false;
    }
}
