
#include "sputil.h"

#include <util/system/getpid.h>

#include <util/stream/format.h>
#include <util/stream/printf.h>
#include <util/folder/path.h>
#include <util/generic/bt_exception.h>
#include <library/cpp/logger/all.h>

IOutputStream& operator<<(IOutputStream& s, const TRemoveForbiddenChars& r) {
    for (auto c : r.s)
        if ((c != '(') && (c != ')') && (c != '-'))
            s << c;
    return s;
}

bool GetFullName(TString& sFullName, const TStringBuf& d, const TStringBuf& n, TString& errorStr) {
    auto joined = JoinFsPaths(d, n);
    const TFsPath p(joined);
    if (!p.Exists()) {
        TStringOutput stringOutput(errorStr);
        stringOutput << p << " not exists";
        return false;
    } else if (!p.IsFile()) {
        TStringOutput stringOutput(errorStr);
        stringOutput << p << " is not file";
        return false;
    }
    sFullName = std::move(joined);
    return true;
}

bool GetDirByFileName(TString& sDirName, const TStringBuf& fn) {
    sDirName.assign("");

    if (!fn)
        return false;

    sDirName = TFsPath(fn).Dirname();
    return true;
}

FILE* SpFileOpen(const char* dn, const char* szName, const char* option, TString& errorStr) {
    if (!dn || !szName)
        return nullptr;

    TString s;

    GetFullName(s, dn, szName, errorStr);
    return SpFileOpen(s.c_str(), option, false, errorStr);
}

FILE* SpFileOpen(const char* fn, const char* option, bool fMessage, TString& errorStr) {
    if (!fn || !(*fn))
        return nullptr;

    FILE* fp = fopen(fn, option);

    if (fp)
        return fp;
    else if (fMessage) {
        TStringOutput s(errorStr);
        s << "Could not open: " << fn;
    }

    return nullptr;
}

TLogBackend * SpFileOpenLog(const TString &name, bool fMessage, TString &errorStr) {
    if (name.empty())
        return nullptr;

    try {
        return new TFileLogBackend(name);
    } catch (const yexception& e) {
        if (fMessage) {
            TStringOutput s(errorStr);
            s << "Could not open: " << name;
        }
    }

    return nullptr;
}

bool SpGetFileSize(const char* fn, ui32* size, TString& errorStr) {
    struct stat bufstat;

    if (stat(fn, &bufstat)) {
        errorStr = "Problem getting file information";
        return false;
    }

    *size = bufstat.st_size;

    return true;
}

TMaybe<TFileStat> FileStateWasChanged(const TStringBuf& fn, const TFileStat& oldStat, TString& errorStr) {
    if (!fn) {
        errorStr = "FileStateWasChanged: file name not defined";
        return Nothing();
    }

    TFileStat fileStat(fn);

    if (fileStat.IsNull()) {
        errorStr = "Could not read file state: ";
        errorStr += fn;
        return Nothing();
    }

    if (oldStat.MTime == fileStat.MTime && oldStat.Size == fileStat.Size)
        return Nothing();

    return fileStat;
}

bool IsNumber(char* sztext) {
    return IsNumber(sztext, strlen(sztext));
}
bool IsNumber(const char* ptext, int len) {
    if (!len)
        return false;

    for (int i = 0; i < len; i++)
        if (!isdigit(ptext[i]) &&
            ptext[i] != '.' &&
            ptext[i] != '+' &&
            ptext[i] != '-')
            return false;

    return true;
}

bool IsInt(const char* ptext, int len) {
    if (!len)
        return false;

    int i = (ptext[0] == '+' || ptext[0] == '-') ? 1 : 0;
    for (; i < len; i++)
        if (!isdigit(ptext[i])) {
            return false;
        }

    return true;
}

bool IsInt(const char* ptext) {
    char c = *ptext;
    return isdigit(c) ||
           ((c == '+' || c == '-') && isdigit(ptext[1]));
}

bool GetInt(const char* ptext, int len, int* pval) {
    if (!IsInt(ptext, len))
        return false;

    *pval = atoi(ptext);
    return true;
}

bool GetDouble(const char* pstr, int len, double* d) {
    int i = (*pstr == '+' || *pstr == '-') ? 1 : 0;
    int cPoint = 0;
    int cDig = 0;

    for (; i < len; i++) {
        if (pstr[i] >= '0' && pstr[i] <= '9')
            cDig++;
        else if (pstr[i] == '.')
            ++cPoint;
        else
            return false;
    }

    if (cPoint > 1 || cDig == 0)
        return false;

    *d = atof(pstr);

    return true;
}

int GetDigitTextLen(const unsigned char* ptext) {
    int len = 0;

    while (isdigit(ptext[len]))
        len++;
    return len;
}

// 10 - 12 - 2005
// 10-12-2005
// 10.12.2005
bool GetDMY(const char* pstr, int len, int* day, int* month, int* year) {
    int i;

    // scip spaces
    for (i = 0; i < len && !isdigit(pstr[i]); i++, pstr++)
        ;
    // scip digit
    for (; i < len && isdigit(pstr[i]); i++)
        ;

    if (!i || !GetInt(pstr, i, day))
        return false;

    pstr += i;
    // scip spaces
    for (i = 0; i < len && !isdigit(pstr[i]); i++, pstr++)
        ;
    // scip digit
    for (; i < len && isdigit(pstr[i]); i++)
        ;

    if (!i || !GetInt(pstr, i, month))
        return false;

    pstr += i;
    // scip spaces
    for (i = 0; i < len && !isdigit(pstr[i]); i++, pstr++)
        ;
    // scip digit
    for (; i < len && isdigit(pstr[i]); i++)
        ;

    if (!i || !GetInt(pstr, i, year))
        return false;

    return true;
}

//trasform dd.mm.yy in time_t
// 20 - 12 - 2005
// 20-12-2005
// 20.12.2005
// return 0 in error case
time_t GetDayMonthYear(const char* str, int len) {
    int day = 0, month = 0, year = 0;
    time_t time_point = 0;

    if (!GetDMY(str, len, &day, &month, &year))
        return 0;

    if (day < 1 || day > 31 || month < 1 || month > 12 || year < 1970 || year > 2050)
        return 0;

    struct tm tm;

    tm.tm_year = year - 1900;
    tm.tm_mon = month - 1;
    tm.tm_mday = day;
    tm.tm_hour = 0;
    tm.tm_min = 0;
    tm.tm_sec = 0;
    tm.tm_isdst = 0;

#if defined(__FreeBSD__)
    time_point = timegm(&tm);
#else
    time_point = mktime(&tm);
#endif

    return time_point;
}

time_t GetDayMonthYear(const char* str) {
    return GetDayMonthYear(str, strlen(str));
}
//trasform mm.dd.yy in time_t
// time_t
// 10 - 22 - 2005
// 10-22-2005
// 10.22.2005
// return 0 in error case
time_t GetMonthDayYear(const char* str, int len) {
    int day = 0, month = 0, year = 0;
    time_t time_point = 0;

    if (!GetDMY(str, len, &day, &month, &year))
        return 0;

    std::swap(day, month);

    if (day < 1 || day > 31 || month < 1 || month > 12 || year < 1970 || year > 2050)
        return 0;

    struct tm tm;

    tm.tm_year = year - 1900;
    tm.tm_mon = month - 1;
    tm.tm_mday = day;
    tm.tm_hour = 0;
    tm.tm_min = 0;
    tm.tm_sec = 0;
    tm.tm_isdst = 0;

#if defined(__FreeBSD__)
    time_point = timegm(&tm);
#else
    time_point = mktime(&tm);
#endif

    return time_point;
}

time_t GetMonthDayYear(const char* str) {
    return GetMonthDayYear(str, strlen(str));
}

//trasform yy.mm.dd in time_t
// time_t
// 2005 - 10 - 22
// 2005-10-22
// 2005.10.22
// return 0 in error case
time_t GetYearMonthDay(const char* str, int len) {
    int day = 0, month = 0, year = 0;
    time_t time_point = 0;

    if (!GetDMY(str, len, &day, &month, &year))
        return 0;

    std::swap(day, year);

    if (day < 1 || day > 31 || month < 1 || month > 12 || year < 1970 || year > 2050)
        return 0;

    struct tm tm;

    tm.tm_year = year - 1900;
    tm.tm_mon = month - 1;
    tm.tm_mday = day;
    tm.tm_hour = 0;
    tm.tm_min = 0;
    tm.tm_sec = 0;
    tm.tm_isdst = 0;

#if defined(__FreeBSD__)
    time_point = timegm(&tm);
#else
    time_point = mktime(&tm);
#endif

    return time_point;
}

time_t GetYearMonthDay(const char* str) {
    return GetYearMonthDay(str, strlen(str));
}

bool IsSpace(ui8 c) {
    if (c == ' ' || c == '\t')
        return true;

    return false;
}

char* STRNCPY(char* strDest, const char* strSource, size_t DestSize) {
    int len = strlen(strSource);
    if (len <= (int)DestSize - 1)
        strcpy(strDest, strSource);
    else {
        len = DestSize - 1;
        strncpy(strDest, strSource, len);
        strDest[len] = 0;
    }
    return strDest;
}

char* STRNCPY(char* strDest, const char* strSource, size_t DestSize, size_t count) {
    if (count > DestSize - 1)
        count = DestSize - 1;
    strncpy(strDest, strSource, count);
    strDest[count] = 0;
    return strDest;
}

char* STRNCPYLWR(char* strDest, const char* strSource, size_t DestSize) {
    STRNCPY(strDest, strSource, DestSize);
    strlwr(strDest);
    return strDest;
}

char* STRNCPYLWR(char* strDest, const char* strSource, size_t DestSize, size_t count) {
    STRNCPY(strDest, strSource, DestSize, count);
    strlwr(strDest);
    return strDest;
}

char* GetStrPid(char str[MAX_STR_PID]) {
    sprintf(str, "_%d_", getpid());
    return str;
}

bool StrToHex(const ui8* str, int len, ui32* value) {
    ui32 b = 0;
    int i;

    for (i = 0; i < len && isxdigit(str[i]); i++) {
        b = b << 4;
        b += (xdigit(str[i]));
    }

    *value = b;
    if (i == len)
        return true;

    return false;
}

void CheckDirSlash(char* dirname) {
    if (dirname) {
        int len = strlen(dirname);
        if (len && dirname[len - 1] != '\\' && dirname[len - 1] != '/')
            strcat(dirname, "/");
    }
}

void CheckDirSlash(TString& dirname) {
    int len = dirname.length();
    if (len && dirname.at(len - 1) != '\\' && dirname.at(len - 1) != '/')
#ifdef WIN32
        dirname.append("\\");
#else
        dirname.append("/");
#endif
}

// if pip != 0  calculate ip and class C net
// else calculate class C net only
// return true if success
bool IpToInt(const char* pstr, int len, ui32* pnet, ui32* pip) {
    char* pdot;
    int i;
    int c = pip ? 3 : 2;
    ui32 t[4];
    const int buf_sz = 64;
    char buf[buf_sz];

    for (i = 0; i < len; i++) {
        if (!isdigit(pstr[i]) && pstr[i] != '.')
            return false;
    }

    for (i = 0; i < c; i++) {
        pdot = (char*)strchr(pstr, '.');
        if (!pdot)
            return false;
        STRNCPY(buf, pstr, buf_sz, pdot - pstr);
        t[i] = atoi(buf);
        pstr = pdot + 1;
        if (t[i] > 255)
            return false;
    }

    if (!isdigit(*pstr))
        return false;
    t[c] = atoi(pstr);
    if (t[c] > 255)
        return false;

    *pnet = t[0] * 256 * 256 * 256;
    *pnet += t[1] * 256 * 256;
    *pnet += t[2] * 256;

    if (pip) {
        *pip = *pnet;
        *pip += t[3];
    }

    return true;
}

const char* IntToIp(ui32 ip, char sip[]) {
    ui32 ip4 = ip & 0xFF;
    ui32 ip3 = (ip >> 8) & 0xFF;
    ui32 ip2 = (ip >> 16) & 0xFF;
    ui32 ip1 = (ip >> 24);
    sprintf(sip, "%u.%u.%u.%u", ip1, ip2, ip3, ip4);
    return sip;
}

// reverse case insensitive compare str and substr
bool cmpdomensrevi(const TStringBuf& subdomen, const TStringBuf& domen) {
    if(!subdomen || !domen || subdomen.size() > domen.size())
        return false;

    if(subdomen.size() == domen.size())
        return AsciiEqualsIgnoreCase(subdomen, domen);

    switch(domen[domen.size() - subdomen.size() - 1]) {
        case '.':
        case '@':
            return AsciiEqualsIgnoreCase(subdomen, domen.Last(subdomen.size()));
        default:
            return false;
    }
}

void STRDUPLWR(char** dst, const char* src, size_t count) {
    STRDUP(dst, src, count);
    strlwr(*dst);
}

char* APPENDSTR(char* strDest, const char* strSource, size_t DestSize, size_t count) {
    int len = strlen(strDest);

    STRNCPY(strDest + len, strSource, DestSize - len, count);
    return strDest;
}

int str_remove(char* str, int len, char* prem, int lenrem) {
    int newlen = len - lenrem;
    int restlen = newlen - (prem - str);

    memmove(prem, prem + lenrem, restlen);
    str[newlen] = 0;
    return newlen;
}

int str_remove(char* str, int len, char* prem, char* premend) {
    int newlen = len - (premend - prem);
    int restlen = len - (premend - str);

    if (restlen > 0)
        memmove(prem, premend, restlen);
    str[newlen] = 0;
    return newlen;
}

char* rot13(const char* strsrc, char* strout) {
    unsigned int i;
    char c, s;

    for (i = 0; i <= strlen(strsrc); i++) {
        c = strsrc[i];
        if (c >= 'a' && c <= 'z')
            s = 'a';
        else if (c >= 'A' && c <= 'Z')
            s = 'A';
        else
            s = 0;
        if (s > 0)
            c = (c - s + 13) % 26;
        strout[i] = c + s;
    }
    return strout;
}

void SpTrim(const char* szText, const char** ppText, int* plen) {
    const ui8* p = (const ui8*)szText;
    const ui8* pbeg;
    while (*p && *p <= 32)
        p++;
    pbeg = p;

    p = (const ui8*)(szText + strlen(szText));

    while (*p <= 32 && p > pbeg)
        p--;

    if (*p > 32)
        p++;

    *ppText = (const char*)pbeg;
    *plen = p - pbeg;
}

void SpSkipSpaces(const char** ppstr) {
    const char* pstr = *ppstr;
    while (*pstr > 0 && *pstr <= 32)
        ++pstr;
    *ppstr = pstr;
}

bool SpIsNotSpace(char ch) {
    if (ch > 32 || ch < 0)
        return true;
    return false;
}

void SpSkipWord(const char** ppstr) {
    while (SpIsNotSpace(**ppstr))
        ++(*ppstr);
}

bool SpEnd_Of_Str(char ch) {
    if (ch == '\r' || ch == '\n' || ch == 0)
        return true;

    return false;
}

int SpGetFirstWord(const char** ppstr, const char** ppnextword) {
    SpSkipSpaces(ppstr);
    *ppnextword = *ppstr;

    if (SpEnd_Of_Str(**ppstr))
        return 0;

    int len = 0;

    while (SpIsNotSpace(**ppnextword)) {
        ++(*ppnextword);
        len++;
    }

    return len;
}

int SpGetNextWord(const char** ppstr, const char** ppnextword) {
    SpSkipSpaces(ppstr);
    *ppnextword = *ppstr;

    if (SpEnd_Of_Str(**ppstr))
        return 0;

    int len = 0;

    while (SpIsNotSpace(**ppnextword)) {
        ++(*ppnextword);
        len++;
    }

    return len;
}

void SpCurTimeStr(char* str, int len, const char* tag) {
    str[0] = 0;

#ifdef WIN32
    char* timeline;
    struct _timeb timebuffer;
    _ftime(&timebuffer);
    timeline = ctime(&(timebuffer.time));
    snprintf(str, len - 1, "%.15s.%03hu - ", &timeline[4], timebuffer.millitm);
#else // if SYSTEM_UNIX

    struct timeval tv;
    gettimeofday(&tv, nullptr);
    time_t sec = (time_t)tv.tv_sec;

    char ctimeBuf[26];
    char* timeline = ctime_r(&sec, ctimeBuf);
    snprintf(str, len - 1, "%.15s.%.03lu - ", &timeline[4], tv.tv_usec / 1000);
#endif

#ifndef SP_NOT_PRINT_PID
    char pidstr[MAX_STR_PID];
    snprintf(str + strlen(str), len - 1, " %s", GetStrPid(pidstr));
#endif

    if (!(tag == 0 || *tag == 0)) {
        int len_s = strlen(str);
        snprintf(str + len_s, len - len_s - 1, "%s", tag);
        str[len - 1] = 0;
    }
}

IOutputStream & operator << (IOutputStream & s, const TPidPrinter & printer) {
    return s << '_' << printer.pid << '_';
}

IOutputStream& operator<<(IOutputStream& s, const TCurTimeAndPid& p) {

//Jul 16 02:03:55
    return s << p.now.FormatGmTime("%b %d %T") << '.' << LeftPad(p.now.MicroSeconds() / 1000, 3, '0') << ' ' << p.pidPrinter;
}

TStringStream& operator<<(TStringStream& stream, const TTextAdder& ta) {
    if(!stream.empty())
        stream << ta.delim;
    stream << ta.text;
    return stream;
}

bool GetLogin(TString& sLogin, const char* pLogin) {
    bool res = false;

    sLogin.assign("");
    if (pLogin && *pLogin != 0) {
        char *p, *p_at;
        STRDUP(&p, pLogin, strlen(pLogin));
        if (p_at = strchr(p, '@')) {
            *p_at = 0;
            res = true;
            sLogin.assign(p);
        }
        DELETE_ARR(p);
    }

    return res;
}

// return domain length;
TStringBuf GetDomain(const TStringBuf & login) {
    const size_t at = login.find('@');

    if(at == TStringBuf::npos)
        return nullptr;

    return login.substr(at + 1).Before(' ');
}

bool IsYandexDomain(const TStringBuf& domain) {
    return
        domain.size() < 11 && AsciiHasPrefixIgnoreCase(domain, "yandex.") ||
        domain.size() < 10 && AsciiHasPrefixIgnoreCase(domain, "narod.") ||
        domain.size() < 7 && AsciiHasPrefixIgnoreCase(domain, "ya.");
}

bool calc_strcrc64(const char* pbuf, size_t len, char str[32]) {
    if (len < 2)
        return false;
    snprintf(str, 31, "%016" PRIx64, FnvHash<ui64>(pbuf, len));
    return true;
}

bool calc_strcrc64(const TStringBuf & b, char str[32]) {
    return calc_strcrc64(b.data(), b.length(), str);
}


TStringBuf IsHttp(TStringBuf url) {
    if (url.Size() < 7)
        return {};

    if(!AsciiHasPrefixIgnoreCase(url, "http"))
        return {};
    url.Skip(4);

    if(AsciiHasPrefixIgnoreCase(url, "s"))
        url.Skip(1);

    if(!url.StartsWith("://"))
        return {};
    return url.Skip(3);
}

TStringBuf IsMailto(TStringBuf url) {
    constexpr const TStringBuf UrlMailto = "mailto:";

    if (!AsciiHasPrefixIgnoreCase(url, UrlMailto))
        return nullptr;

    return url.Skip(UrlMailto.Size());
}

ptrdiff_t GetCharsNumBeforeSpace(const char* pText, int Len) {
    auto * p = (ui8*)pText;
    int i = 0;

    while (*p && !IsSpace(*p) && i++ < Len)
        p++;

    return std::distance((ui8*)pText, p);
}

ptrdiff_t GetCharsNumBeforeSpace(const char *pText) {
    auto * p = (ui8*)pText;

    while (*p && !IsSpace(*p))
        p++;

    return std::distance((ui8*)pText, p);
}

TStringBuf StringAfter(const TStringBuf& text, const TStringBuf& part) {
    const auto offset = std::distance(text.data(), part.data());
    if(Y_UNLIKELY(offset < 0))
        ythrow TWithBackTrace<yexception>() << "part \"" << part << "\" isn't in text \"" << text << '"';

    const size_t endIndex = offset + part.size();
    if(Y_UNLIKELY(endIndex > text.size()))
        ythrow TWithBackTrace<yexception>() << "part \"" << part << "\" isn't in text \"" << text << '"';

    return text.substr(endIndex);
}

ECharset GetDestinationEncoding(ELanguage lang) {
    switch (lang) {
        case LANG_RUS:
            return CODES_KOI8;
        case LANG_POL:
            return CODES_WIN_EAST;
        case LANG_TUR:
            return CODES_WINDOWS_1254;
        default:
            return CODES_KOI8;
    }
}
