#pragma once

#include <util/generic/queue.h>
#include <util/system/mutex.h>
#include "prof.h"
#include <library/cpp/threading/queue/mpsc_read_as_filled.h>
#include <mutex>
#include <shared_mutex>
#include <util/generic/list.h>
#include <util/system/rwlock.h>

template<class TItem> class TSemiOrderedConsumer{
public:
    using TItems = TDeque<TItem>;
public:
    template<class ...TArgs> void Add(TArgs&&... args) {
        auto item = MakeHolder<TItem>(std::forward<TArgs>(args)...);
        lfQueue.Push(item.Release());
    }

    void JustCleanUp(TInstant after) {
        auto newItems = PopNewEvents(after);
        auto g = Guard(itemsMutex);
        CleanUp(after);
        StoreNewItems(std::move(newItems));
    }

    TItems GetLastsAndCleanUp(TInstant after) {
        auto newItems = PopNewEvents(after);
        auto g = Guard(itemsMutex);
        CleanUp(after);
        StoreNewItems(std::move(newItems));
        return items;
    }

private:

    auto PopNewEvents(TInstant after) {
        TDeque<THolder<TItem>> newItems;
        THolder<TItem> item;

        {
            auto g = Guard(popQueueMutex);
            while ((item = THolder(lfQueue.Pop())))
                if (!(*item < after))
                    newItems.emplace_back(std::move(item));
        }

        return newItems;
    }

    void CleanUp(TInstant after) {
        auto it = std::lower_bound(items.cbegin(), items.cend(), after, [](const TItem& item, TInstant v){
            return item < v;
        });
        items.erase(items.cbegin(), it);
    }

    void StoreNewItems(TDeque<THolder<TItem>> && newItems) {
        for(auto & item : newItems)
            items.emplace_back(std::move(*item));
    }
private:
    TMutex itemsMutex, popQueueMutex;
    TItems items;
    NThreading::TReadAsFilledQueue<TItem> lfQueue;
};


struct TStatEvent{
    constexpr bool operator<(TInstant t) const noexcept {
        return timestamp < t;
    }
    explicit TStatEvent(TString name, const NProf::Item& prof) noexcept : name(std::move(name)), prof(prof) {}

    TString name;
    TInstant timestamp = Now();
    NProf::Item prof{};
};

TString YasmTagFromComment(const TStringBuf & comment);

class TStatsConsumer : public TSemiOrderedConsumer<TStatEvent> {};

class TStatsConsumerLock {
public:
    TStatsConsumerLock(TStatsConsumer& consumer, TString name);

    void Acquire();
    void Release();

private:
    NProf::Item prof;
    const TString name;
    TStatsConsumer& consumer;
};

using TStatsList = TList<NProf::TNamedItem>;
class TStatsListSafe{
public:
    void Add(TStatsList newList) {
        TReadGuard g(Mutex);
        for(auto& item: newList)
            if(item.GetEnd() != TInstant::Zero())
                List.emplace_back(std::move(item));
    }

    TStatsList CleanUpAndGet(TInstant after) {
        TWriteGuard g(Mutex);
        auto it = std::lower_bound(List.cbegin(), List.cend(), after);
        List.erase(List.cbegin(), it);

        return List;
    }
private:
    TRWMutex Mutex;
    TStatsList List;
};

