#pragma once

#include <functional>
#include <util/generic/ptr.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/datetime/base.h>
#include <util/system/rwlock.h>
#include <util/string/cast.h>
#include <mail/so/libs/scheduler/scheduler.h>

namespace re2 { class RE2; class StringPiece; }

namespace NTagSets {
    template<typename T, typename Tag>
    class TStorage {
    public:
        using TSet = THashSet<T>;
        using TTag = Tag;
        using TValue = T;

        void SetData(const TTag& tag, TSet data) {
            TReadGuard lock(m_mutex);
            TInfo& info = GetInfo(tag, lock);

            TWriteGuard lockInfo(info.second);
            info.first.swap(data);
        }

        void RemoveData(const TTag& tag) {
            TWriteGuard lock(m_mutex);
            m_data.erase(tag);
        }

        TVector<TTag> GetTags(const TValue& value) const {
            TVector<TTag> result;
            GetTagsImpl(value, result);
            return result;
        }

    private:
        using TInfo = std::pair<TSet, TRWMutex>;
        using TReadUnguard = TInverseGuard<TRWMutex, TReadGuardOps<TRWMutex>>;

        TInfo& GetInfo(const TTag& tag, const TReadGuard& lock) {
            auto it = m_data.find(tag);
            if (it != m_data.end())
                return it->second;

            TReadUnguard unlock(lock.GetMutex());
            TWriteGuard writelock(lock.GetMutex());

            return m_data[tag];
        }

        void GetTagsImpl(const TValue& value, TVector<TTag>& result) const {
            TReadGuard lock(m_mutex);
            for (const auto& item : m_data) {
                TReadGuard lockInfo(item.second.second);
                if (item.second.first.contains(value))
                    result.push_back(item.first);
            }
        }

        THashMap<TTag, TInfo> m_data;
        mutable TRWMutex m_mutex;
    };

    template<typename T, typename Tag>
    class TSource {
    public:
        using TStorage = TStorage<T, Tag>;
        using TDestination = TAtomicSharedPtr<TStorage>;

        TSource(const TDestination& destination) : m_destination(destination) {
            Y_ENSURE(m_destination);
        }
        virtual ~TSource() = default;

        virtual void Update() = 0;

    protected:
        virtual void SetData(const typename TStorage::TTag& tag, const typename TStorage::TSet& data) {
            m_destination->SetData(tag, data);
        }

    private:
        TDestination m_destination;
    };

    template<typename T, typename Tag>
    class TManager {
    public:
        using TSource = TSource<T, Tag>;
        using TSourcePtr = TAtomicSharedPtr<TSource>;
        using TStoragePtr = typename TSource::TDestination;

        TManager()
            : m_data(MakeAtomicShared<typename TStoragePtr::TValueType>())
        { }

        const TStoragePtr& Get() const {
            return m_data;
        }

        template<typename Q, typename ... Types>
        TSourcePtr& CreateSource(Types&& ... args) {
            return CreateScheduleSource<Q>(TDuration::Zero(), std::forward<Types>(args)...);
        }

        template<typename Q, typename ... Types>
        TSourcePtr& CreateScheduleSource(const TDuration& interval, Types&& ... args) {
            return m_sources.emplace_back(interval, MakeAtomicShared<Q>(m_data, std::forward<Types>(args)...)).second;
        }

        void SetSchedules(TSimpleScheduler& scheduler) {
            size_t count = 0; auto now = TInstant::Now();
            for (const auto& item : m_sources) {
                if (item.first != TDuration::Zero()) {
                    auto source = item.second;
                    scheduler.Add([source]() {source->Update(); }, now, item.first, "tag set source " + ToString(count++) + " update");
                }
            }
        }

    private:
        TStoragePtr m_data;
        TVector<std::pair<TDuration, TSourcePtr>> m_sources;
    };

    namespace NHelpers {
        template<typename From, typename Tag>
        class THashSource : public TSource<size_t, Tag> {
        public:
            using TBase = TSource<size_t, Tag>;
            THashSource(const typename TBase::TDestination& destination)
                : TBase(destination)
            { }

        protected:
            using TBase::SetData;
            using TSet = typename TSource<From, Tag>::TStorage::TSet;

            virtual void SetData(const typename TBase::TStorage::TTag& tag, const TSet& data) {
                typename TBase::TStorage::TSet result;
                for (const auto& v: data) {
                    result.insert(m_hasher(v));
                }
                this->SetData(tag, result);
            }

        private:
            THash<From> m_hasher;
        };

        class TRegexpReplaceImpl {
        public:
            using TReplaceItem = std::pair<TString, TString>;
            using TReplaceInfo = TVector<TReplaceItem>;

            TRegexpReplaceImpl(const TString& from, const TString& to);
            TRegexpReplaceImpl(const TReplaceItem& item);
            TRegexpReplaceImpl(const TReplaceInfo& info);
            ~TRegexpReplaceImpl();

            THashSet<TString> Replace(const THashSet<TString>& data) const;

        private:
            using TReplaceItemImpl = std::pair<THolder<re2::RE2>, THolder<re2::StringPiece>>;
            TVector<TReplaceItemImpl> m_info;
        };

        template<typename Tag, typename Base = TSource<TString, Tag> >
        class TRegexpReplaceSource : public Base, public TRegexpReplaceImpl {
        public:
            template<typename ... Types>
            TRegexpReplaceSource(const typename Base::TDestination& destination, const Types& ... args)
                : Base(destination)
                , TRegexpReplaceImpl(args...)
            { }

        protected:
            void SetData(const Tag& tag, const THashSet<TString>& data) override {
                const auto converted = Replace(data);
                Base::SetData(tag, converted);
            }
        };

        template<typename Tag> using THashRegexpReplaceSource = TRegexpReplaceSource<Tag, THashSource<TString, Tag> >;
    }
}
