#pragma once

#include <util/generic/string.h>
#include <util/system/mutex.h>
#include <util/system/rwlock.h>
#include <util/system/fstat.h>
#include <library/cpp/charset/codepage.h>
#include "tkipv6.h"
#include "kfunc.h"
#include "tlogsgroupbase.h"
#include <list>

#define K_PRINT_ONLY_STAT

namespace kipv6 {

    //********************************************************************************************************************
    //                                                 THashStat
    //********************************************************************************************************************

    struct THashStat {
        ui32 trace_count;
        ui32 no_trace_count;
        ui32 bad_count;
        ui32 dublicat_count;

        THashStat() {
            Clear();
        }

        void Clear() {
            trace_count = 0;
            no_trace_count = 0;
            bad_count = 0;
            dublicat_count = 0;
        }
    };

    //*********************************************************************************************************************
    //                                                 THashIP
    //*********************************************************************************************************************

    class THashIPv6 {
    private:
        static const int MAX_FILE_SIZE = 33000000;

    private:
        TString m_name;
        TString m_filename;
        TFileStat m_statHashIp;
        TRWMutex m_Mutex;
        TMutex m_MutexLoad; //�� ����� �������� ����� ������
        bool m_clear_before_load;

        static ui32 GetStr(const char* source, size_t source_size, char* destination, ui32 destination_size);
        void ReloadMemListA(const char* buff, size_t buff_size, THashStat& stat);
        virtual void ParseItem(const char* str, THashStat& stat) = 0;
        virtual void ClearData() = 0;
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay) = 0;
        virtual void TriggerFunction() = 0;
        virtual void PrefixFunction() = 0;
        virtual void PostfixFunction() = 0;
        bool ExistsFile();

    public:
        TLogsGroupBase* LogsGroup;
        TString m_ident;

        void Lock();
        void UnLock();
        void LockRead();
        void UnLockRead();
        void LockLoad();
        void UnLockLoad();
        static ui64 CalcShingle(const TString& str_data);
        static TString IntToIpV4(ui32 ip);
        static char* STRNCPY(char* strDest, const char* strSource, size_t DestSize, size_t count);
        bool IpToIntV4(const char* pstr, size_t len, ui32* pnet, ui32* pip);

    public:
        THashIPv6();
        THashIPv6(const TString& name);
        virtual ~THashIPv6();

        void InitBase(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA, bool clear_before_load);
        void ReloadFileList();
        void ReloadMemList(const TString& data);
    };

    //*********************************************************************************************************************
    //                                                 TWhiteIPv6
    //*********************************************************************************************************************

    typedef THashMap<TKIPv6, ui32> TKIPv6Hash;

    typedef THashMap<ui64, ui32> TKUidHash;
    typedef TKUidHash::iterator TKUidHashIt;

    typedef THashMap<TKIPv6, TKUidHash> TKIPv6TraceHash;

    class TWhiteIPv6 : public THashIPv6 {
    private:
        static const ui32 MAX_UID_COUNT = 10000;

    private:
        TKIPv6Hash data;
        TKIPv6TraceHash data_trace;

        TKIPv6Hash data_temp;
        TKIPv6TraceHash data_trace_temp;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();

    public:
        TWhiteIPv6();
        virtual ~TWhiteIPv6();

        void Init(const TString& filename, TLogsGroupBase* LogsGroupA);
        void Init(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA);
        bool IsWhiteIp(TKIPv6 ip);
        bool IsWhiteIpTrace(TKIPv6 ip, ui64 uid);
    };

    //*********************************************************************************************************************
    //                                                 TRblIPv6
    //*********************************************************************************************************************

    typedef THashMap<TKIPv6, ui8> TKIPv6HashM;

    class TRblIPv6Item : public THashIPv6 {
    private:
        TKIPv6HashM* m_data;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();

    public:
        TRblIPv6Item();
        virtual ~TRblIPv6Item();

        void Init(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA);
        void SetHashPointer(TKIPv6HashM* dataA) {
            m_data = dataA;
        }
    };

    class TRblIPv6ItemDiff : public THashIPv6 {
    private:
        TKIPv6HashM* m_data;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();

    public:
        TRblIPv6ItemDiff();
        virtual ~TRblIPv6ItemDiff();

        void Init(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA);
        void SetHashPointer(TKIPv6HashM* dataA) {
            m_data = dataA;
        }
    };

    class TRblIPv6 {
    private:
        TLogsGroupBase* LogsGroup;
        TKIPv6HashM* data;
        TKIPv6HashM* data_temp;
        TKIPv6HashM* data_diff;
        TKIPv6HashM* data_diff_temp;
        TString blacklist1;
        TString blacklist1to;
        TString blacklist2;
        TString blacklist2to;
        TString blacklist3;
        TString blacklist3to;
        TString blacklist4;
        TString blacklist4to;
        TString blacklistdiff;
        TRblIPv6Item m_rbl1;
        TRblIPv6Item m_rbl2;
        TRblIPv6Item m_rbl3;
        TRblIPv6Item m_rbl4;
        TRblIPv6ItemDiff m_rbl_diff;
        TMutex m_Mutex;
        TMutex m_MutexLoad;
        // struct stat m_statdiffIp;

        void Lock();
        void UnLock();
        void LockLoad();
        void UnLockLoad();
        static i64 copyfile(const char* source, const char* destination);
        void PrefixFunction();
        void PostfixFunction();
        void PrefixFunctionDiff();
        void PostfixFunctionDiff();

    public:
        TRblIPv6();
        ~TRblIPv6();

        void Init(const TString& blacklist1A,
                  const TString& blacklist1toA,
                  const TString& blacklist2A,
                  const TString& blacklist2toA,
                  const TString& blacklist3A,
                  const TString& blacklist3toA,
                  const TString& blacklist4A,
                  const TString& blacklist4toA,
                  const TString& blacklistdiffA,
                  TLogsGroupBase* LogsGroupA);

        bool RefreshBLFiles();
        bool RefreshBLMems(const TString& rbl1_txt, const TString& rbl2_txt, const TString& rbl3_txt, const TString& rbl4_txt);

        bool RefreshBLDiffFiles();
        bool RefreshBLDiffMems(const TString& data);

        bool GetRBL(TKIPv6 ip);
    };

    //*********************************************************************************************************************
    //                                                 TStringsLists
    //*********************************************************************************************************************

    typedef THashMap<ui64, TString> TStrokaHash;
    typedef TStrokaHash::iterator TStrokaHashIt;

    class TStringsLists : public THashIPv6 {
    private:
        TStrokaHash* data;
        TStrokaHash* data_temp;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();

    public:
        TStringsLists();
        virtual ~TStringsLists();

        void Init(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA);
        bool Find(const char* value);
        bool Find(const TString& value);
    };

    //*********************************************************************************************************************
    //                                                 TStringsNumberLists
    //*********************************************************************************************************************

    struct TStringNumber {
        TString m_text;
        int m_number;

        TStringNumber() {
            Clear();
        }

        TStringNumber(const TString& str, int numb) {
            m_text = str;
            m_number = numb;
        }

        void Clear() {
            m_text = "";
            m_number = 0;
        }
    };

    typedef THashMap<ui64, TStringNumber> TStrokaNumberHash;
    typedef TStrokaNumberHash::iterator TStrokaNumberHashIt;

    class TStringsNumberLists : public THashIPv6 {
    private:
        TStrokaNumberHash* data;
        TStrokaNumberHash* data_temp;
        int m_type;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();

    public:
        TStringsNumberLists();
        virtual ~TStringsNumberLists();

        void Init(const TString& ident, const TString& filename, TLogsGroupBase* LogsGroupA, int type);
        int Find(const char* value);
        int Find(const TString& value);
    };

    //*********************************************************************************************************************
    //                                                 TZoneStat
    //*********************************************************************************************************************

    typedef struct _V_SDH {
        ui32 cS;
        ui32 cD;
        ui32 cH;
        ui32 cDSN;
    } V_SDH;

    struct TZoneStatStruct {
        TString m_text;
        V_SDH m_data;

        TZoneStatStruct() {
            Clear();
        }

        TZoneStatStruct(const TString& value) {
            Clear();
            m_text = value;
        }

        void Clear() {
            m_text = "";
            memset(&m_data, 0, sizeof(m_data));
        }
    };

    typedef THashMap<ui64, TZoneStatStruct> TZoneStatStructHash;
    typedef TZoneStatStructHash::iterator TZoneStatStructHashIt;

    class TZoneStat : public THashIPv6 {
    private:
        TZoneStatStructHash* data;
        TZoneStatStructHash* data_temp;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();
        bool AddZone(const char* zone, TSpClass messclass, bool fDSN);

    public:
        TZoneStat();
        virtual ~TZoneStat();

        void Init(const TString& filename, TLogsGroupBase* LogsGroupA);
        bool AddZoneTwice(const char* zone, TSpClass messclass, bool fDSN);
        bool PrintZones(const char* filename, bool fClear = false);
        bool PrintZones(TLogClass* log, bool fClear = false);
    };

    //*********************************************************************************************************************
    //                                                 TWhiteNetv6
    //*********************************************************************************************************************

#define SYMB_RANGE_WN1 0x01
#define SYMB_RANGE_WN2 0x02
#define IS_SYMB_RANGE_WN1(flag) ((flag)&SYMB_RANGE_WN1)
#define IS_SYMB_RANGE_WN2(flag) ((flag)&SYMB_RANGE_WN2)

    struct TWIPv6 {
        TKIPv6 ip1;
        TKIPv6 ip2;

        TWIPv6() {
            ip1 = TKIPv6();
            ip2 = TKIPv6();
        }
    };

    typedef std::list<TWIPv6> TWIPv6List;
    typedef TWIPv6List::iterator TWIPv6ListIt;

    typedef THashMap<ui64, TString> TTStringHash;
    typedef TTStringHash::iterator TTStringHashIt;

    class TWhiteNetv6 : public THashIPv6 {
    public:
        enum TRecordType { TUNDEF,
                           THOST,
                           TIPNET,
                           TIPRANGE,
                           TIPADDRESS };

    private:
        TTStringHash* host_data;
        TTStringHash* host_data_temp;
        TWIPv6List* range_data;
        TWIPv6List* range_data_temp;
        ui8 SymbTable[256]{};
        TString idents;

        virtual void ParseItem(const char* str, THashStat& stat);
        virtual void ClearData();
        virtual TString GetResultPrint(THashStat& stat, ui32 timedelay);
        virtual void TriggerFunction();
        virtual void PrefixFunction();
        virtual void PostfixFunction();
        static TString NormalizeStr(const char* str, size_t strlength);
        TRecordType GetRecordType(const char* str, size_t strlength);
        void InitTable();
        bool IsAllowSymbolRange1(char symb);
        bool IsAllowSymbolRange2(char symb);
        bool ParseRangeType1(const TString& text, TWIPv6& value);
        static bool ParseRangeType2(const TString& text, TWIPv6& value);

    public:
        TWhiteNetv6();
        TWhiteNetv6(const TString& ident);
        virtual ~TWhiteNetv6();

        void Init(const TString& filename, TLogsGroupBase* LogsGroupA);
        bool IsWhiteNet(const char* phost);
        bool IsWhiteNet(TKIPv6 ip);
        bool IsWhiteNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason);

        TRecordType GetAddresses(const TString& addr_text, TWIPv6& value);
    };

    //**********************************************************************************************************************

} // namespace kipv6
