#include "tkipv6.h"
#include "kfunc.h"
#include <array>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <library/cpp/ipv6_address/ipv6_address.h>
#include <netinet/in.h>
#include <mail/so/libs/syslog/so_log.h>

//******************************************************************************
//                                TKIPv6
//******************************************************************************

TKIPv6::TKIPv6()
    : TKIPv6(0, 0)
{ }

TKIPv6::TKIPv6(const TString& ip)
    : TKIPv6(ip.c_str())
{ }

TKIPv6::TKIPv6(const char* ip)
    : TKIPv6()
{
    TString ips;
    TStringBuf data(ip);
    std::copy_if(data.begin(), data.end(), std::back_inserter(ips), [](ui8 c) {
        return c > ' ';
    });

    ui32 ipaddr = 0;
    ui64 lpart = 0, hpart = 0;
    const bool has_point = ips.Contains('.');
    const bool has_colon = ips.Contains(':');

    if (has_point && has_colon) //IPv4
    {
        if ((IpToIntIPv4Mapped(ips, ipaddr)) && (ipaddr > 0))
            *this = TKIPv6(ipaddr);
        else if ((IpToIntIPv4Embedded(ips, hpart, lpart)) && ((hpart > 0) || (lpart > 0))) //IPv6 (11:11:11:11:11:11:127.0.0.1)
            *this = TKIPv6(hpart, lpart);
    } else if (has_point && !has_colon) //IPv4 (127.0.0.1)
    {
        if ((IpToIntIPv4(ips, ipaddr)) && (ipaddr > 0))
            *this = TKIPv6(ipaddr);
    } else if (!has_point && has_colon) // IPv6 (11:11:11:11:11:11:11:11)
    {
        if ((IpToIntIPv6(ips, hpart, lpart)) && ((hpart > 0) || (lpart > 0)))
            *this = TKIPv6(hpart, lpart);
    }
}

TKIPv6::TKIPv6(const ui64 highpart, const ui64 lowpart)
    : m_highpart(highpart), m_lowpart(lowpart)
{ }

TKIPv6::TKIPv6(const ui32 part1, const ui32 part2, const ui32 part3, const ui32 part4)
    : TKIPv6((static_cast<ui64>(part1) << 32) | part2, (static_cast<ui64>(part3) << 32) | part4)
{ }

TKIPv6::TKIPv6(const ui32 part)
    : TKIPv6(0, 0, 0xFFFF, part)
{ }

size_t TKIPv6::size() const {
    return (sizeof(m_highpart) + sizeof(m_lowpart));
}

bool TKIPv6::operator<(const TKIPv6& value) const {
    return (m_highpart == value.m_highpart) ? m_lowpart < value.m_lowpart : m_highpart < value.m_highpart;
}

bool TKIPv6::operator>(const TKIPv6& value) const {
    return value < *this;
}

bool TKIPv6::operator<=(const TKIPv6& value) const {
    return !(*this > value);
}

bool TKIPv6::operator>=(const TKIPv6& value) const {
    return !(*this < value);
}

bool TKIPv6::operator==(const TKIPv6& value) const {
    return (m_highpart == value.m_highpart) && (m_lowpart == value.m_lowpart);
}

bool TKIPv6::operator!=(const TKIPv6& value) const {
    return !(*this == value);
}

bool TKIPv6::IpToIntIPv4(const TStringBuf& data, ui32& ip) const
try {
    ip = 0;
    auto splitter = StringSplitter(data).Split('.');

    int c = 0;
    for(auto it : splitter){
        auto v = it.Token();
        ip = (ip << 8) + (v.empty()? 0: FromString<ui8>(v));
        c++;
    }

    return c == 4;
} catch (const std::exception&) {
    return false;
}

bool TKIPv6::IpToIntIPv4Mapped(const TStringBuf& data, ui32& ip) const {
    TStringBuf address(data);
    if (!address.SkipPrefix("::ffff:") && !address.SkipPrefix("0:0:0:0:0:FFFF:"))
        return false;

    return IpToIntIPv4(address, ip);
}

bool TKIPv6::IpToIntIPv4Embedded(const TStringBuf& data, ui64& highpart, ui64& lowpart) const {
    TStringBuf left, ipv4;
    if (!data.TryRSplit(':', left, ipv4))
        return false;

    ui32 ip = 0;
    const TString ipv6 = ToString(left) + ":0000:0000";
    if (!IpToIntIPv6(ipv6, highpart, lowpart) || !IpToIntIPv4(ipv4, ip))
        return false;

    lowpart |= ip;
    return true;
}

static TString ExpandIPv6(const TStringBuf& data)
{
    TStringBuf head, tail;
    if (!data.TrySplit("::", head, tail))
        return ToString(data);

    const size_t count = std::count(data.begin(), data.end(), ':');
    return (count > 7)? ToString(data): ToString(head) + TString(9 - count, ':') + ToString(tail);
}

bool TKIPv6::IpToIntIPv6(const TStringBuf& data, ui64& highpart, ui64& lowpart) const
try {
    const TString address = ExpandIPv6(data);

    auto splitter = StringSplitter(address).Split(':');

    int c = 0;

    for (auto it : splitter) {
        auto v = it.Token();
        highpart = (highpart << 16) + (v.empty() ? 0 : IntFromString<ui16, 16>(v));
        if(++c == 4)
            break;
    }

    if(c != 4)
        return false;

    c = 0;
    for (auto it : splitter) {
        auto v = it.Token();
        lowpart = (lowpart << 16) + (v.empty() ? 0 : IntFromString<ui16, 16>(v));
        c++;
    }
    return c == 4;
} catch (...) {
    Cerr << CurrentExceptionMessageWithBt() << Endl;
    return false;
}

TString TKIPv6::IntToIpIPv4(ui32 ip) const {
    std::array<ui32, 4> oct;
    oct[0] = (ip >> 24) & 0xFF;
    oct[1] = (ip >> 16) & 0xFF;
    oct[2] = (ip >> 8) & 0xFF;
    oct[3] = ip & 0xFF;

    return ToString(oct[0]) + '.' + ToString(oct[1]) + '.' + ToString(oct[2]) + '.' + ToString(oct[3]);
}

TString TKIPv6::IntToIpIPv6() const {
    std::array<ui16, 8> oct;
    oct[0] = (m_highpart >> 48) & 0xFFFF;
    oct[1] = (m_highpart >> 32) & 0xFFFF;
    oct[2] = (m_highpart >> 16) & 0xFFFF;
    oct[3] = m_highpart & 0xFFFF;
    oct[4] = (m_lowpart >> 48) & 0xFFFF;
    oct[5] = (m_lowpart >> 32) & 0xFFFF;
    oct[6] = (m_lowpart >> 16) & 0xFFFF;
    oct[7] = m_lowpart & 0xFFFF;

    std::array<char, 64> buffer; buffer.fill(0);
    snprintf(buffer.data(), buffer.size() - 1, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x", oct[0], oct[1], oct[2], oct[3], oct[4], oct[5], oct[6], oct[7]);
    return TString(buffer.data());
}

bool TKIPv6::Undefined() const {
    return m_highpart == 0 && m_lowpart == 0;
}

bool TKIPv6::IsIPv4() const {
    return m_highpart == 0 && (m_lowpart >> 32) == 0xFFFF;
}

bool TKIPv6::IsIPv6() const {
    return ((!Undefined()) && (!IsIPv4()));
}

TString TKIPv6::toStroka() const {
    TString res = "";

    if (Undefined())
        res = "-";
    else if (IsIPv4()) {
        ui32 ip = 0;

        ip = m_lowpart & 0xFFFFFFFF;
        res = "::ffff:" + IntToIpIPv4(ip);

    } else if (IsIPv6())
        res = IntToIpIPv6();

    return res;
}

TString TKIPv6::toStroka2() const {
    TString res = "";

    if (Undefined())
        res = "-";
    else if (IsIPv4()) {
        ui32 ip = 0;

        ip = m_lowpart & 0xFFFFFFFF;
        res = IntToIpIPv4(ip);

    } else if (IsIPv6())
        res = IntToIpIPv6();

    return res;
}

ui32 TKIPv6::IPv4Address() const {
    return IsIPv4()? m_lowpart & 0xFFFFFFFF: 0;
}

TString TKIPv6::IPv4AddressStr() const {
    return (IsIPv4() || Undefined())? IntToIpIPv4(IPv4Address()): nullptr;
}

TString TKIPv6::IPv6AddressStr() const {
    return IntToIpIPv6();
}

ui32 TKIPv6::AsShingle32() const {
    ui32 res = 0;
    char buff[sizeof(ui64) + sizeof(ui64) + 1];

    if (Undefined()) {
        res = 0;
    } else if (IsIPv4()) {
        res = IPv4Address();
    } else {
        memset(buff, 0, sizeof(buff));
        memcpy(buff, &m_highpart, sizeof(ui64));
        memcpy(buff + sizeof(ui64), &m_lowpart, sizeof(ui64));
        res = FnvHash<ui32>(buff, sizeof(ui64) + sizeof(ui64));
    }

    return res;
}

ui64 TKIPv6::AsShingle64() const {
    ui64 res = 0;

    if (Undefined()) {
        res = 0;

    } else {
        TString tstr = IntToIpIPv6();

        if (!tstr.empty()) {
            res = ShingleFromStroka(tstr);

        } else {
            res = 0;
        }
    }

    return res;
}

TString TKIPv6::AsShingle128() const {
    char buf[40];
    sprintf(buf, "%016" PRIX64 "%016" PRIX64, m_highpart, m_lowpart);
    return buf;
}

bool TKIPv6::FromShingle128(const TString& shingle) {
    ui64 highpart, lowpart;
    if (sscanf(shingle.c_str(), "%016" PRIX64 "%016" PRIX64, &highpart, &lowpart) != 2)
        return false;

    m_highpart = highpart;
    m_lowpart = lowpart;
    return true;
}

TKIPv6 TKIPv6::GetNextIP() {
    const ui64 high = (m_lowpart == std::numeric_limits<ui64>::max())? 1 : 0;
    return TKIPv6(m_highpart + high, m_lowpart + 1);
}

bool TKIPv6::IsLocalIP() {
    if (IsIPv4()) {
        ui32 host = (m_lowpart >> 16 ) & 0xFFFF;
        if (host == 0xC0A8 || host == 0xAC10 || host == 0xA9FE) //"192.168.", "172.16.", "172.16.", "169.254."
            return true;

        host = host >> 8;
        if (host == 127 || host == 10 || host == 0) //"127.0", "10.", "0."
            return true;
    }

    return false;
}

TKIPv6 TKIPv6::GetAddressByMask(ui8 mask) {
    if (mask == 0 || mask > 128 || Undefined())
        return TKIPv6();

    if (IsIPv4()) {
        if (mask > 32)
            return TKIPv6();

        if (mask == 32)
            return *this;

        const ui64 vv = std::numeric_limits<ui64>::max() << (32 - mask);
        return TKIPv6(0, LowPart() & vv);
    }

    if (mask == 128)
        return *this;

    if (mask == 64)
        return TKIPv6(HighPart(), 0);

    if (mask < 64) {
        const ui64 vv = std::numeric_limits<ui64>::max() << (64 - mask);
        return TKIPv6(HighPart() & vv, 0);
    }

    const ui64 vv = std::numeric_limits<ui64>::max() << (128 - mask);
    return TKIPv6(HighPart(), LowPart() & vv);
}
