#pragma once

#ifndef tkipv6_H
#define tkipv6_H

#include <util/generic/string.h>
#include <util/generic/hash.h>
#include <util/digest/multi.h>
#include <library/cpp/string_utils/quote/quote.h>
#include "kfunc.h"

#include <cstdio>
#include <list>
#include <limits>

enum TIpType { IPAll,
               IPv4,
               IPv6 };
enum TWhtIp { WHTALL,
              WHTYES,
              WHTNO };

//******************************************************************************
//                                TKIPv6
//******************************************************************************

class TKIPv6 {
private:
    //IPv4-mapped IPv6 address:
    //                     80бит               | 16 бит |   32бит
    //0000................................0000 |  FFFF  | IPv4 адрес
    //понимает адреса в виде:
    //127.0.0.1
    //::ffff:95.108.173.187
    //0:0:0:0:0:FFFF:129.144.52.38
    //1234:5678:90AB:CDEF:1234:5678:90AB:CDEF
    //fe80:0000:0000:0000:843e:ee57:1.125.115.226
    //2002:c000:0204::/48 (6to4: 192.0.2.4)

    ui64 m_highpart;
    ui64 m_lowpart;

    bool IpToIntIPv4(const TStringBuf& data, ui32& ip) const;
    bool IpToIntIPv4Mapped(const TStringBuf& data, ui32& ip) const;
    bool IpToIntIPv4Embedded(const TStringBuf& data, ui64& highpart, ui64& lowpart) const;
    bool IpToIntIPv6(const TStringBuf& data, ui64& highpart, ui64& lowpart) const;
    [[nodiscard]] TString IntToIpIPv4(ui32 ip) const;
    [[nodiscard]] TString IntToIpIPv6() const;

public:
    Y_SAVELOAD_DEFINE(m_highpart, m_lowpart)
    TKIPv6();
    explicit TKIPv6(const char* ip);
    explicit TKIPv6(const TString& ip);
    TKIPv6(ui64 highpart, ui64 lowpart);
    TKIPv6(ui32 part1, ui32 part2, ui32 part3, ui32 part4); //формируем ipv6 адрес (или mapped-ipv4)
    explicit TKIPv6(ui32 part);                                                        //формируем ipv4 адрес

    [[nodiscard]] size_t size() const;
    bool operator<(const TKIPv6& value) const;
    bool operator>(const TKIPv6& value) const;
    bool operator<=(const TKIPv6& value) const;
    bool operator>=(const TKIPv6& value) const;
    bool operator==(const TKIPv6& value) const;
    bool operator!=(const TKIPv6& value) const;
    operator size_t() const {
        return MultiHash(m_highpart, m_lowpart);
    }

    [[nodiscard]] bool Undefined() const;         //не определен (нулевой адрес)
    [[nodiscard]] bool IsIPv4() const;            //true - это ipv4 адрес
    [[nodiscard]] bool IsIPv6() const;            //true - это ipv6 адрес
    [[nodiscard]] TString toStroka() const;       //строковое представление адреса (для ipv4 вид "::ffff:127.0.0.1", для ipv6 вид "FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF", если адрес не определен, то "-")
    [[nodiscard]] TString toStroka2() const;      //строковое представление адреса (для ipv4 вид "127.0.0.1", для ipv6 вид "FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF", если адрес не определен, то "-")

    [[nodiscard]] ui32 IPv4Address() const;       //32-битное представление адреса для ipv4 (если не ipv4, то возвращает 0)
    [[nodiscard]] TString IPv4AddressStr() const; //текстовое представление адреса для ipv4 (если не ipv4, то возвращает "", если адрес не определен "0.0.0.0")
    [[nodiscard]] TString IPv6AddressStr() const; //текстовое представление адреса для ipv6 (если адрес не определен, то возвращает "0000:0000:0000:0000:0000:0000:0000:0000")

    [[nodiscard]] ui64 HighPart() const {
        return m_highpart;
    }
    [[nodiscard]] ui64 LowPart() const {
        return m_lowpart;
    }

    [[nodiscard]] ui32 AsShingle32() const;                         //представление в виде 32-битного шингла
    [[nodiscard]] ui64 AsShingle64() const;                         //представление в виде 64-битного шингла
    [[nodiscard]] TString AsShingle128() const;                     //представление в виде 128-биного шингла
    bool FromShingle128(const TString& shingle);      //загрузка ip-адреса из 128-битного шингла

    TKIPv6 GetNextIP(); //вернуть следующий ip (+1), если это максимальный ip, то возвращаем 0 (Undefined)

    bool IsLocalIP();                  //true - локальный ip
    TKIPv6 GetAddressByMask(ui8 mask); //вернуть адрес после наложения на его маски

    [[nodiscard]] TKIPv6 Net64() const {
        return TKIPv6(m_highpart, 0);
    } //возвращает сетку ip/64
    ui8 GetMinorByte() {
        return m_lowpart & 0xFF;
    } //возвращаем младший байт адреса
};

#endif
