#include "tnetipv6.h"
#include "shtime.h"

THashStatIPv6::THashStatIPv6()
    : trace_count(0)
    , no_trace_count(0)
    , bad_count(0)
    , dublicat_count(0) {
}

void THashStatIPv6::Clear() {
    *this = THashStatIPv6();
}

TFileStateIPv6::TFileStateIPv6()
    : mtime(0)
    , size(0) {
}

TMaskIem::TMaskIem()
    : m_hash()
    , addcount(0)
    , dublicat_count(0)
    , internal_error(0) {
}

TMaskIem::TMaskIem(TKMaskHash hash)
    : m_hash(std::move(hash))
    , addcount(1)
    , dublicat_count(0)
    , internal_error(0) {
}

TMaskItem2::TMaskItem2()
    : TMaskItem2(0, 0, 0, 0) {
}

TMaskItem2::TMaskItem2(ui8 maskA, ui32 addcountA, ui32 dublicat_countA, ui32 internal_errorA)
    : mask(maskA)
    , addcount(addcountA)
    , dublicat_count(dublicat_countA)
    , internal_error(internal_errorA) {
}

bool TMaskItem2::operator<(const TMaskItem2& value) const {
    return mask < value.mask;
}

//*********************************************************************************************************************
//                                                 THashIP
//*********************************************************************************************************************

THashIPv6Base::THashIPv6Base() {
    m_filename = "";
    m_Log = nullptr;
    m_ident = "";
    m_clear_before_load = true;
    m_loadmode = NUM_DEFAULT;
    m_load_ok = "";
    m_load_dublicat = "";
}

THashIPv6Base::THashIPv6Base(const THashIPv6Base& value) {
    m_filename = value.m_filename;
    m_Log = value.m_Log;
    m_ident = value.m_ident;
    m_clear_before_load = value.m_clear_before_load;
    m_statHashIp = value.m_statHashIp;
    m_loadmode = value.m_loadmode;
    m_load_ok = value.m_load_ok;
    m_load_dublicat = value.m_load_dublicat;
}

THashIPv6Base::~THashIPv6Base() {
}

void THashIPv6Base::Lock() const {
    m_Mutex.Acquire();
}

void THashIPv6Base::UnLock() const {
    m_Mutex.Release();
}

void THashIPv6Base::LockLoad() {
    m_MutexLoad.Acquire();
}

void THashIPv6Base::UnLockLoad() {
    m_MutexLoad.Release();
}

void THashIPv6Base::InitBase(const TString& ident, const TString& filename, TBaseLogClass* LogA, bool clear_before_load) {
    char buff[25];

    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff), "%15s", ident.c_str()); // was "% 15s"
    m_ident = TString(buff);

    m_filename = filename;
    m_Log = LogA;
    m_clear_before_load = clear_before_load;
}

bool THashIPv6Base::FileStateWasChangedK(const char* fn, TFileStateIPv6* poldbufstat) {
    struct stat bufstat;

    if (fn == 0 || *fn == 0) {
        return false;
    }

    if (stat(fn, &bufstat)) {
        return false;
    }

    if (poldbufstat->mtime == bufstat.st_mtime && poldbufstat->size == bufstat.st_size)
        return false;

    poldbufstat->mtime = bufstat.st_mtime;
    poldbufstat->size = bufstat.st_size;

    return true;
}

void THashIPv6Base::ReloadFileList() {
    LockLoad();

    m_load_ok = "";
    m_load_dublicat = "";

    UnLockLoad();

    if (!m_filename.empty()) {
        if (ExistsFile()) {
            if (!FileStateWasChangedK(m_filename.c_str(), &m_statHashIp)) {
                if (m_Log != nullptr)
                    m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s:    no changed.", m_ident.c_str());
                return;
            }

            FILE* handle = nullptr;
            char tbuff[1024];
            int len = 0;
            TString restxt = "";
            THashStatIPv6 stat;

            handle = fopen(m_filename.c_str(), "rb");
            if (handle != nullptr) {
                LockLoad();

                ui32 timepoint = CShingleTime::GetMs();

                PrefixFunction();

                while (fgets(tbuff, sizeof(tbuff), handle)) {
                    len = strlen(tbuff);
                    if (len > 0)
                        ReloadMemListA(tbuff, len, stat);
                }

                PostfixFunction();
                TriggerFunction();

                UnLockLoad();

                fclose(handle);

                timepoint = CShingleTime::GetMs() - timepoint;
                restxt = GetResultPrint(stat, timepoint);
                if (m_loadmode == THashIPv6Base::NUM_CBB)
                    restxt = restxt + " {OK='" + m_load_ok + "' dublicat='" + m_load_dublicat + "'}";
                if (m_Log != nullptr)
                    m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s", restxt.c_str());

            } else {
                if (m_Log != nullptr)
                    m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s:   error open file to read: '%s'.", m_ident.c_str(), m_filename.c_str());
            }
        } else {
            if (m_Log != nullptr)
                m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s:   file not found: '%s'.", m_ident.c_str(), m_filename.c_str());
        }
    } else {
        if (m_Log != nullptr)
            m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s:   no defined filename.", m_ident.c_str());
    }
}

void THashIPv6Base::ReloadFileListCBB() {
    m_loadmode = NUM_CBB;
    ReloadFileList();
}

ui32 THashIPv6Base::GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size) {
    long pos = -1;
    ui32 res = 0;
    ui8 dob = 0;

    for (ui32 i = 0; i < source_size; i++) {
        if ((source[i] == 0x0D) && ((i + 1) < source_size) && (source[i + 1] == 0x0A)) {
            pos = i;
            dob = 1;
            break;
        }
        if (source[i] == 0x0A) {
            pos = i;
            dob = 0;
            break;
        }
    }
    res = pos + 1 + dob;
    if (res == 0)
        res = source_size;
    if (res > 0) {
        memset(destination, 0, destination_size);
        if (res <= (destination_size - 1)) {
            memcpy(destination, source, res);
        } else {
            memcpy(destination, source, destination_size - 1);
        }
    }
    return res;
}

void THashIPv6Base::ReloadMemListA(const char* buff, size_t buff_size, THashStatIPv6& stat) {
    ui32 m_buff_pos = 0;
    char tbuff[65500];
    const char* sourcebuff = nullptr;
    size_t sourcebuffsize = 0;
    ui32 strlength = 0;
    TString restxt = "";
    bool valid_str = false;
    int identify_type = 0;

    sourcebuff = buff;
    sourcebuffsize = buff_size;
    m_buff_pos = 0;
    while (m_buff_pos < sourcebuffsize) {
        strlength = GetStr(sourcebuff + m_buff_pos, sourcebuffsize - m_buff_pos, tbuff, sizeof(tbuff));
        if (strlength > 0) {
            m_buff_pos += strlength;

            valid_str = false;
            for (size_t i = 0; i < strlength; i++) {
                if ((tbuff[i] < 0) || (tbuff[i] > ' ')) {
                    valid_str = true;
                    break;
                }
            }

            if (valid_str)
                ParseItem(false, tbuff, stat, identify_type);

        } else
            break;
    }
}

void THashIPv6Base::ReloadMemList(const TString& data) {
    THashStatIPv6 stat;
    TString restxt = "";

    LockLoad();

    m_load_ok = "";
    m_load_dublicat = "";

    ui32 timepoint = CShingleTime::GetMs();
    PrefixFunction();

    ReloadMemListA(data.c_str(), data.length(), stat);

    PostfixFunction();
    TriggerFunction();

    UnLockLoad();

    timepoint = CShingleTime::GetMs() - timepoint;
    restxt = GetResultPrint(stat, timepoint);
    if (m_loadmode == THashIPv6Base::NUM_CBB)
        restxt = restxt + " {OK='" + m_load_ok + "' dublicat='" + m_load_dublicat + "'}";
    if (m_Log != nullptr)
        m_Log->WriteMessageAndDataStatus(KMESSAGE, "%s", restxt.c_str());
}

void THashIPv6Base::ReloadMemListCBB(const TString& data) {
    m_loadmode = NUM_CBB;
    ReloadMemList(data);
}

bool THashIPv6Base::ExistsFile() {
    bool res = false;
    FILE* handle = nullptr;

    if (!m_filename.empty()) {
        handle = fopen(m_filename.c_str(), "rb");
        if (handle != nullptr) {
            res = true;
            fclose(handle);
        }
    }

    return res;
}

ui64 THashIPv6Base::CalcShingle(const TString& str_data) const {
    ui64 res = 0;
    TString str_data_t = "";

    if (!str_data.empty()) {
        str_data_t = Trim(str_data);
        if (!str_data_t.empty()) {
            char sshingle[32];
            TString stemp = "";

            stemp = str_data_t;
            to_lower_k(stemp);
            memset(sshingle, 0, sizeof(sshingle));
            calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
            sscanf(sshingle, "%" PRIx64, &res);
        }
    }

    return res;
}

TString THashIPv6Base::IntToIpV4(ui32 ip) {
    TString res = "";
    char buff[64];
    ui32 oct[4];

    memset(buff, 0, sizeof(buff));
    oct[0] = (ip >> 24) & 0xFF;
    oct[1] = (ip >> 16) & 0xFF;
    oct[2] = (ip >> 8) & 0xFF;
    oct[3] = ip & 0xFF;
    snprintf(buff, sizeof(buff) - 1, "%u.%u.%u.%u", oct[0], oct[1], oct[2], oct[3]);
    res = TString(buff);

    return res;
}

bool THashIPv6Base::IpToIntV4(const char* pstr, int len, ui32* pnet, ui32* pip) {
    const char* pdot;
    int i;
    int c = pip ? 3 : 2;
    ui32 t[4];
    const int buf_sz = 64;
    char buf[buf_sz];

    for (i = 0; i < len; i++) {
        if (!isdigit(pstr[i]) && pstr[i] != '.')
            return false;
    }

    for (i = 0; i < c; i++) {
        pdot = strchr(pstr, '.');
        if (!pdot)
            return false;
        STRNCPY(buf, pstr, buf_sz, pdot - pstr);
        t[i] = atoi(buf);
        pstr = pdot + 1;
        if (t[i] > 255)
            return false;
    }

    if (!isdigit(*pstr))
        return false;
    t[c] = atoi(pstr);
    if (t[c] > 255)
        return false;

    *pnet = t[0] * 256 * 256 * 256;
    *pnet += t[1] * 256 * 256;
    *pnet += t[2] * 256;

    if (pip) {
        *pip = *pnet;
        *pip += t[3];
    }

    return true;
}

//*********************************************************************************************************************
//                                              TFindIPClass
//*********************************************************************************************************************

TFindIPClass::TFindIPClass() {
}

void TFindIPClass::Lock() {
    m_Mutex.Acquire();
}

void TFindIPClass::UnLock() {
    m_Mutex.Release();
}

TString TFindIPClass::GetNetsStat() {
    TString res = "";
    TKMaskItemHashIt it;
    TString tstr = "";
    ui32 netcount_all = 0;
    ui32 add_all = 0;
    ui32 dublicat_all = 0;
    ui32 error_all = 0;
    ui32 mask_all = 0;
    bool first = true;
    TMaskItem2List statlist;
    TMaskItem2ListIt it2;

    Lock();

    it = m_masklist.begin();
    while (it != m_masklist.end()) {
        mask_all++;
        statlist.push_back(TMaskItem2((*it).first, (*it).second.addcount, (*it).second.dublicat_count, (*it).second.internal_error));
        if ((*it).second.m_hash) {
            netcount_all = IncMax32(netcount_all, (*it).second.m_hash.size());
            add_all = IncMax32(add_all, (*it).second.addcount);
            dublicat_all = IncMax32(dublicat_all, (*it).second.dublicat_count);
            error_all = IncMax32(error_all, (*it).second.internal_error);
        }

        ++it;
    }

    UnLock();

    statlist.sort();
    it2 = statlist.begin();
    while (it2 != statlist.end()) {
        if (first)
            first = false;
        else
            tstr = tstr + ", ";
        tstr = tstr + IntToStroka((*it2).mask) + ": add=" + IntToStroka((*it2).addcount) + " dubl=" + IntToStroka((*it2).dublicat_count) + " err=" + IntToStroka((*it2).internal_error);

        ++it2;
    }

    res = res + "mask_count=" + IntToStroka(mask_all) + " net_count=" + IntToStroka(netcount_all) + " add_all=" + IntToStroka(add_all) + " dublicat_all=" + IntToStroka(dublicat_all) + " error_all=" + IntToStroka(error_all) + "(" + tstr + ")";

    return res;
}

bool TFindIPClass::AddNet(const TString& nets, bool& dublicat) {
    bool res = false;
    TKMaskItemHashIt it;
    TKMaskHashIt itm;
    ui64 shingle = 0;
    TString netbody = 0;
    ui8 mask = 0;
    TString nets_m = "";
    const char* p = nullptr;
    int count = 0;
    TKIPv6 ip = TKIPv6();

    dublicat = false;
    nets_m = Trim(nets);
    if (!nets_m.empty()) {
        p = strchr(nets_m.c_str(), '/');
        if (p == nullptr)
            p = strchr(nets_m.c_str(), '\\');
        if (p != nullptr) {
            count = p - nets_m.c_str();
            if (count > 0) {
                mask = atoi(TString(p + 1).c_str());
                netbody = TString(nets_m.c_str(), count);
                ip = TKIPv6(netbody.c_str());
                if (ip.Undefined())
                    netbody = "";
                else
                    netbody = ip.GetAddressByMask(mask).toStroka();
            }
        }
    }
    if ((mask > 0) && (mask <= 128) && (!netbody.empty())) {
        shingle = ShingleFromStroka(netbody);
        if (shingle != 0) {
            Lock();

            it = m_masklist.find(mask);
            if (it != m_masklist.end()) {
                auto& maskhash = (*it).second.m_hash;
                if (maskhash) {
                    itm = maskhash.find(shingle);
                    if (itm == maskhash.end()) {
                        (maskhash)[shingle] = nets_m;
                        (*it).second.addcount = IncMax32((*it).second.addcount, 1);
                        res = true;

                    } else {
                        (*it).second.dublicat_count = IncMax32((*it).second.dublicat_count, 1);
                        dublicat = true;
                    }

                } else
                    (*it).second.internal_error = IncMax32((*it).second.internal_error, 1);

            } else {
                TKMaskHash maskhash;
                (maskhash)[shingle] = nets_m;
                m_masklist[mask] = TMaskIem(maskhash);
                res = true;
            }

            UnLock();
        }
    }

    return res;
}

bool TFindIPClass::IsIncludeNet(TKIPv6 ip) {
    TString netsource = "";
    ui8 mask = 0;

    return IsIncludeNet(ip, netsource, mask);
}

bool TFindIPClass::IsIncludeNet(TKIPv6 ip, TString& netsource, ui8& mask) {
    bool res = false;
    TKMaskItemHashIt it;
    TKMaskHashIt itm;
    ui64 shingle = 0;
    TString ipbody = "";

    Lock();

    mask = 0;
    it = m_masklist.begin();
    while (it != m_masklist.end()) {
        if ((*it).second.m_hash) {
            ipbody = ip.GetAddressByMask((*it).first).toStroka();
            if (!ipbody.empty()) {
                shingle = ShingleFromStroka(ipbody);
                if (shingle != 0) {
                    itm = (*it).second.m_hash.find(shingle);
                    if (itm != (*it).second.m_hash.end()) {
                        netsource = (*itm).second;
                        mask = (*it).first;
                        res = true;
                        break;
                    }
                }
            }
        }

        ++it;
    }

    UnLock();

    return res;
}

ui32 TFindIPClass::size() const {
    return m_masklist.size();
}

//*********************************************************************************************************************
//                                                 TNetIPv6
//*********************************************************************************************************************

TNetIPv6::TNetIPv6() {
    DefaultConstructor();
}

TNetIPv6::TNetIPv6(const TString& ident) {
    DefaultConstructor();
    idents = ident;
}

TNetIPv6::~TNetIPv6() {
    ClearData();
}

void TNetIPv6::DefaultConstructor() {
    idents = "";
    host_data = {};
    host_data_temp = {};
    range_data = {};
    range_data_temp = {};
    ipnetobj = {};
    ipnetobj_temp = {};
}

void TNetIPv6::Init(const TString& filename, TBaseLogClass* LogA) {
    InitTable();
    InitBase(idents, filename, LogA, true);
}

int TNetIPv6::ParseItem(bool item_add_mode, const char* str, THashStatIPv6& stat, int& identify_type) {
    int res = PRS_ERROR_UNKNOWN;
    TStrokaHashB* host_data_A = nullptr;
    TWIPv6BList* range_data_A = nullptr;
    TFindIPClass* ipnetobj_A = nullptr;

    if (item_add_mode) {
        host_data_A = &host_data;
        range_data_A = &range_data;
        ipnetobj_A = ipnetobj.Get();

    } else {
        host_data_A = &host_data_temp;
        range_data_A = &range_data_temp;
        ipnetobj_A = ipnetobj_temp.Get();
    }

    identify_type = TUNDEF;
    if (str != nullptr) {
        TString strdata = "";
        TRecordType rectype = TNetIPv6::TUNDEF;
        TWIPv6B wip;
        ui64 shingle = 0;
        TStrokaHashBIt it;

        switch (m_loadmode) {
            case THashIPv6Base::NUM_CBB:
                strdata = NormalizeUrl(str);
                break;
            default:
                strdata = NormalizeStr(str, strlen(str));
        };

        if ((!strdata.empty()) && (host_data_A != nullptr) && (range_data_A != nullptr) && (ipnetobj_A != nullptr)) {
            if (strdata[0] == '#') {
                identify_type = TCOMMENT;
                res = PRS_IS_COMMENT;

            } else {
                rectype = GetRecordType(strdata.c_str(), strdata.length());
                identify_type = rectype;
                switch (rectype) {
                    case TNetIPv6::THOST:
                        shingle = CalcShingle(strdata);
                        if (shingle != 0) {
                            it = host_data_A->find(shingle);
                            if (it != host_data_A->end()) {
                                stat.dublicat_count = IncMax32(stat.dublicat_count, 1);
                                res = PRS_ERROR_DUBLICAT;
                                if (m_loadmode == THashIPv6Base::NUM_CBB) {
                                    if (m_load_dublicat.empty())
                                        m_load_dublicat = m_load_dublicat + strdata + "[" + ShingleToStroka(shingle) + "]";
                                    else
                                        m_load_dublicat = m_load_dublicat + ", " + strdata + "[" + ShingleToStroka(shingle) + "]";
                                }

                            } else {
                                (*host_data_A)[shingle] = strdata;
                                stat.trace_count = IncMax32(stat.trace_count, 1);
                                res = PRS_OK;
                                if (m_loadmode == THashIPv6Base::NUM_CBB) {
                                    if (m_load_ok.empty())
                                        m_load_ok = m_load_ok + strdata + "[" + ShingleToStroka(shingle) + "]";
                                    else
                                        m_load_ok = m_load_ok + ", " + strdata + "[" + ShingleToStroka(shingle) + "]";
                                }
                            }

                        } else {
                            stat.bad_count = IncMax32(stat.bad_count, 1);
                            res = PRS_ERROR_BAD;
                        }
                        break;
                    case TNetIPv6::TIPNET: //range 1 (127.0.0.1/32)
                        if (ipnetobj_A != nullptr) {
                            bool dublicat = false;

                            if (ipnetobj_A->AddNet(strdata, dublicat)) {
                                stat.no_trace_count = IncMax32(stat.no_trace_count, 1);
                                res = PRS_OK;

                            } else {
                                if (dublicat) {
                                    stat.dublicat_count = IncMax32(stat.dublicat_count, 1);
                                    res = PRS_ERROR_DUBLICAT;

                                } else {
                                    stat.bad_count = IncMax32(stat.bad_count, 1);
                                    res = PRS_ERROR_BAD;
                                }
                            }
                        }
                        break;
                    case TNetIPv6::TIPRANGE: //range 2 (127.0.0.1 127.0.0.2)
                        if (ParseRangeType2(strdata, wip)) {
                            wip.source_net = strdata;
                            range_data_A->push_back(wip);
                            stat.no_trace_count = IncMax32(stat.no_trace_count, 1);
                            res = PRS_OK;

                        } else {
                            stat.bad_count = IncMax32(stat.bad_count, 1);
                            res = PRS_ERROR_BAD;
                        }
                        break;
                    default:
                        res = PRS_ERROR_UNKNOWN_TYPE;
                };
            }

        } else {
            if (strdata.empty())
                res = PRS_ERROR_EMPTY_NORMSTR;
            else if ((host_data_A == nullptr) || (range_data_A == nullptr) || (ipnetobj_A == nullptr))
                res = PRS_ERROR_NULL_STORAGE;
        }

    } else {
        res = PRS_ERROR_NULL_STR;
    }

    return res;
}

void TNetIPv6::ClearData() {
    Lock();

    range_data = {};
    ipnetobj = {};

    UnLock();

    LockLoad();

    host_data_temp = {};
    range_data_temp = {};
    ipnetobj_temp = {};

    UnLockLoad();
}

TString TNetIPv6::GetResultPrint(THashStatIPv6& stat, ui32 timedelay) {
    TString res = "";
    ui32 all = 0;

    all = IncMax32(all, stat.no_trace_count);
    all = IncMax32(all, stat.trace_count);
    all = IncMax32(all, stat.bad_count);
    all = IncMax32(all, stat.dublicat_count);
    res = res + m_ident + ":   READ LIST (" + IntToStroka(timedelay) + " ms), records: " + IntToStroka(all) + " (url=" + IntToStroka(stat.trace_count) + ", dublicat url=" + IntToStroka(stat.dublicat_count) + ", range=" + IntToStroka(stat.no_trace_count) + ", bad=" + IntToStroka(stat.bad_count) + ")";
    if (ipnetobj != nullptr)
        res = res + "; MASKHASH STATISTIK " + ipnetobj->GetNetsStat();

    return res;
}

void TNetIPv6::TriggerFunction() {
}

void TNetIPv6::PrefixFunction() {
    host_data_temp = TStrokaHashB();
    range_data_temp = TWIPv6BList();
    ipnetobj_temp = MakeHolder<TFindIPClass>();
}

void TNetIPv6::PostfixFunction() {
    Lock();

    host_data = std::exchange(host_data_temp, {});
    range_data = std::exchange(range_data_temp, {});
    ipnetobj = std::exchange(ipnetobj_temp, nullptr);

    UnLock();
}

TString TNetIPv6::NormalizeStr(const char* str, int strlength) {
    TString res = "";
    char symb = 0;
    char symb2 = 0;
    TString stemp = "";
    TString stemp2 = "";
    bool first_symb = true;
    bool next_slash = false;
    char last_symb = 0x00;
    const char* p = nullptr;
    long count = 0;

    stemp = TString(str, strlength);
    if (!stemp.empty()) {
        p = strchr(stemp.c_str(), '#');
        if (p != nullptr) {
            count = std::distance(stemp.c_str(), p);
            if (count > 0)
                stemp = TString(stemp.c_str(), count);
        }
        for (size_t i = 0; i < stemp.length(); i++) {
            symb = *(stemp.c_str() + i);
            if (symb <= ' ') {
                if (first_symb) {
                    continue;
                } else {
                    size_t j = 0;
                    next_slash = false;
                    for (j = (i + 1); j < stemp.length(); j++) {
                        symb2 = *(stemp.c_str() + j);
                        if (symb2 <= ' ') {
                            continue;
                        } else if ((symb2 == '\\') || (symb2 == '/')) {
                            next_slash = true;
                            break;
                        } else
                            break;
                    }
                    if ((!next_slash) && (last_symb != '/') && (last_symb != '\\') && (j < stemp.length()))
                        stemp2 = stemp2 + ' ';
                    i = j - 1;
                }
            } else {
                if (first_symb)
                    first_symb = false;
                stemp2 = stemp2 + symb;
                last_symb = symb;
            }
        }
        res = stemp2;
    }

    return res;
}

TNetIPv6::TRecordType TNetIPv6::GetRecordType(const char* str, int strlength) {
    TRecordType res = TUNDEF;
    char symb = 0;
    bool not_range1 = false;
    bool not_range2 = false;
    const char* pgood = nullptr;
    const char* pbad = nullptr;

    if ((str != nullptr) && (strlength > 0)) {
        for (int i = 0; i < strlength; i++) {
            symb = *(str + i);
            if (!IsAllowSymbolRange1(symb))
                not_range1 = true;
            if (!IsAllowSymbolRange2(symb))
                not_range2 = true;
            if (not_range1 && not_range2) {
                res = THOST;
                break;
            }
        }
    }
    if (!not_range1) //range 1 (127.0.0.1/32)
    {
        pgood = strchr(str, '/');
        if (pgood == nullptr)
            pgood = strchr(str, '\\');
        pbad = strchr(str, ' ');
        if ((pgood != nullptr) && (pbad == nullptr))
            res = TIPNET;
    }
    if ((res != TIPNET) && (!not_range2)) //range 2 (127.0.0.1 127.0.0.2)
    {
        pgood = strchr(str, ' ');
        pbad = strchr(str, '/');
        if (pbad == nullptr)
            pbad = strchr(str, '\\');
        if ((pgood != nullptr) && (pbad == nullptr))
            res = TIPRANGE;
    }
    if (res == TUNDEF) {
        TKIPv6 addr = TKIPv6(str);
        if (!addr.Undefined())
            res = TIPADDRESS;
    }

    return res;
}

void TNetIPv6::InitTable() {
    memset(SymbTable, 0, sizeof(SymbTable));

    //range 1 (127.0.0.1/32)
    SymbTable[int('.')] = (ui8)((SymbTable[int('.')] & 0xFE) + 0x01);
    SymbTable[int(':')] = (ui8)((SymbTable[int(':')] & 0xFE) + 0x01);
    SymbTable[int('/')] = (ui8)((SymbTable[int('/')] & 0xFE) + 0x01);
    SymbTable[int('\\')] = (ui8)((SymbTable[int('\\')] & 0xFE) + 0x01);
    for (int i = 'a'; i <= 'f'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFE) + 0x01);
    for (int i = 'A'; i <= 'F'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFE) + 0x01);
    for (int i = '0'; i <= '9'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFE) + 0x01);

    //range 2
    SymbTable[int('.')] = (ui8)((SymbTable[int('.')] & 0xFD) + 0x02);
    SymbTable[int(':')] = (ui8)((SymbTable[int(':')] & 0xFD) + 0x02);
    SymbTable[int(' ')] = (ui8)((SymbTable[int(' ')] & 0xFD) + 0x02);
    for (int i = 'a'; i <= 'f'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFD) + 0x02);
    for (int i = 'A'; i <= 'F'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFD) + 0x02);
    for (int i = '0'; i <= '9'; i++)
        SymbTable[i] = (ui8)((SymbTable[i] & 0xFD) + 0x02);
}

bool TNetIPv6::IsAllowSymbolRange1(char symb) {
    bool res = false;
    unsigned char uc = 0;
    ui8 value = 0;

    memcpy(&uc, &symb, sizeof(symb));
    value = SymbTable[uc];
    if (IS_SYMB_RANGE_NI1(value) > 0)
        res = true;

    return res;
}

bool TNetIPv6::IsAllowSymbolRange2(char symb) {
    bool res = false;
    unsigned char uc = 0;
    ui8 value = 0;

    memcpy(&uc, &symb, sizeof(symb));
    value = SymbTable[uc];
    if (IS_SYMB_RANGE_NI2(value) > 0)
        res = true;

    return res;
}

bool TNetIPv6::ParseRangeType1(const TString& text, TWIPv6B& value) {
    bool res = false;
    const char* p1 = nullptr;
    const char* p2 = nullptr;
    int ipsize = 0;
    ui32 ip = 0;
    ui32 net = 0;
    ui32 ipmaskai2 = 0;
    ui32 firstip = 0;
    ui32 lastip = 0;
    ui32 ipcount = 0;
    int ipmaskai = 0;
    TString ipaddress = "";
    TString ipmaska = "";
    TString firstip_s = "";
    TString lastip_s = "";

    if (!text.empty()) {
        p1 = strstr(text.c_str(), "\\");
        p2 = strstr(text.c_str(), "/");
        if (p1 != nullptr) {
            ipsize = p1 - text.c_str();
            if (ipsize > 0) {
                ipaddress = Trim(TString(text.c_str(), ipsize));
                ipmaska = Trim(TString(p1 + 1));
                ipmaskai = atoi(ipmaska.c_str());
            }
        } else if (p2 != nullptr) {
            ipsize = p2 - text.c_str();
            if (ipsize > 0) {
                ipaddress = TString(text.c_str(), ipsize);
                ipmaska = TString(p2 + 1);
                ipmaskai = atoi(ipmaska.c_str());
            }
        }
        if ((!ipaddress.empty()) && (!ipmaska.empty() && (ipmaskai > 0))) {
            TKIPv6 ipt = TKIPv6(ipaddress.c_str());

            if (ipt.IsIPv4()) {
                if (!IpToIntV4(ipaddress.c_str(), ipaddress.size(), &net, &ip))
                    ip = 0;

                if (ip > 0) {
                    ipmaskai2 = 0xFFFFFFFF << (32 - ipmaskai);
                    ipcount = 1 << (32 - ipmaskai);
                    firstip = ip & ipmaskai2;
                    lastip = firstip + ipcount - 1;

                    firstip_s = IntToIpV4(firstip);
                    lastip_s = IntToIpV4(lastip);

                    value.ip1 = TKIPv6(firstip_s.c_str());
                    value.ip2 = TKIPv6(lastip_s.c_str());
                    if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                        res = true;
                }

            } else if (ipt.IsIPv6()) {
                ui64 vv = 1;
                ui64 vm64_const = std::numeric_limits<ui64>::max();
                ui64 vm64 = vm64_const;
                ui64 vt = 0;
                ui64 vcount = 0;
                ui64 vfirstip = 0;
                ui64 vlastip = 0;
                int ipmaskai_t = 0;
                ui64 vipmaskai2 = 0;

                if (ipmaskai == 64) {
                    vt = ipt.HighPart();
                    value.ip1 = TKIPv6(vt, 0);
                    value.ip2 = TKIPv6(vt, vm64_const);
                    if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                        res = true;

                } else if (ipmaskai < 64) {
                    ipmaskai_t = ipmaskai;
                    vt = ipt.HighPart();
                    vipmaskai2 = vm64 << (64 - ipmaskai_t);
                    vcount = vv << (64 - ipmaskai_t);
                    vfirstip = vt & vipmaskai2;
                    vlastip = vfirstip + vcount - 1;
                    value.ip1 = TKIPv6(vfirstip, 0);
                    value.ip2 = TKIPv6(vlastip, vm64_const);
                    if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                        res = true;

                } else if (ipmaskai > 64) {
                    ipmaskai_t = ipmaskai - 64;
                    vt = ipt.LowPart();
                    vipmaskai2 = vm64 << (64 - ipmaskai_t);
                    vcount = vv << (64 - ipmaskai_t);
                    vfirstip = vt & vipmaskai2;
                    vlastip = vfirstip + vcount - 1;
                    value.ip1 = TKIPv6(ipt.HighPart(), vfirstip);
                    value.ip2 = TKIPv6(ipt.HighPart(), vlastip);
                    if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                        res = true;
                }
            }
        }
    }

    return res;
}

bool TNetIPv6::ParseRangeType2(const TString& text, TWIPv6B& value) {
    bool res = false;
    char ipbuff1[255];
    char ipbuff2[255];

    if (!text.empty()) {
        memset(ipbuff1, 0, sizeof(ipbuff1));
        memset(ipbuff2, 0, sizeof(ipbuff2));
        if (sscanf(text.c_str(), " %s %s ", ipbuff1, ipbuff2) == 2) {
            value.ip1 = TKIPv6(ipbuff1);
            value.ip2 = TKIPv6(ipbuff2);
            if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                res = true;
        }
    }

    return res;
}

TString TNetIPv6::NormalizeUrl(const TString& urlA) const {
    TString res = "";
    TString http_ident = "http://";
    TString https_ident = "https://";
    TString www_ident = "www.";
    TString doubleslash_ident = "//";
    TString copyurl = "";
    TString tstr = "";
    const char *p1 = nullptr, *p2 = nullptr, *p3 = nullptr, *p4 = nullptr, *pres = nullptr;
    int count = 0;
    int truncpos = 0;
    unsigned char uc = 0;

    copyurl = Trim(urlA);
    if (!copyurl.empty()) {
        truncpos = -1;
        for (size_t i = 0; i < copyurl.length(); i++) {
            memcpy(&uc, copyurl.c_str() + i, sizeof(uc));
            if (uc /*copyurl[i]*/ <= ' ') {
                truncpos = i;
                break;
            }
        }
        if (truncpos > 0)
            copyurl = TString(copyurl.c_str(), truncpos);

        tstr = copyurl;

        to_lower_k(tstr);
        p1 = strstr(tstr.c_str(), http_ident.c_str());
        p2 = strstr(tstr.c_str(), https_ident.c_str());
        p3 = strstr(tstr.c_str(), www_ident.c_str());
        p4 = strstr(tstr.c_str(), doubleslash_ident.c_str());

        if (p1 != nullptr) {
            p1 = p1 + http_ident.length();
            if (pres == nullptr) {
                pres = p1;

            } else {
                if (p1 > pres)
                    pres = p1;
            }
        }
        if (p2 != nullptr) {
            p2 = p2 + https_ident.length();
            if (pres == nullptr) {
                pres = p2;

            } else {
                if (p2 > pres)
                    pres = p2;
            }
        }
        if (p3 != nullptr) {
            p3 = p3 + www_ident.length();
            if (pres == nullptr) {
                pres = p3;

            } else {
                if (p3 > pres)
                    pres = p3;
            }
        }
        if (p4 != nullptr) {
            p4 = p4 + doubleslash_ident.length();
            if (pres == nullptr) {
                pres = p4;

            } else {
                if (p4 > pres)
                    pres = p4;
            }
        }
        if (pres != nullptr) {
            count = pres - tstr.c_str();
            if (count > 0) {
                res = TString(copyurl.c_str() + count);
                res = Trim(res);
            }
        } else {
            res = copyurl;
        }
    }

    if (!res.empty()) {
        count = 0;
        for (int i = res.length() - 1; i >= 0; i--) {
            if ((res[i] == '/') || (res[i] == '\\')) {
                count++;
            } else {
                break;
            }
        }
        if (count > 0)
            res = TString(res.c_str(), res.length() - count);
    }

    return res;
}

bool TNetIPv6::IsIncludeNetSerialAlgA(TKIPv6 ip, TString& netsource, TKIPv6& min_ip_from_diapason) const {
    bool res = false;
    TWIPv6B wip;

    netsource = "";
    min_ip_from_diapason = TKIPv6();
    if ((!ip.Undefined())) {
        auto it = range_data.begin();
        while (it != range_data.end()) {
            wip = *it;
            if ((ip >= wip.ip1) && (ip <= wip.ip2)) {
                min_ip_from_diapason = wip.ip1;
                res = true;
                break;
            }

            ++it;
        }
    }

    return res;
}

bool TNetIPv6::IsIncludeNetHashAlgA(TKIPv6 ip, TString& netsource, ui8& mask) const {
    bool res = false;

    if (!ip.Undefined()) {
        if (ipnetobj != nullptr)
            res = ipnetobj->IsIncludeNet(ip, netsource, mask);
    }

    return res;
}

bool TNetIPv6::IsIncludeNet(const char* phost) const {
    bool res = false;
    ui64 shingle = 0;
    TString st = "";

    if ((phost != nullptr) && (*phost != '-')) {
        st = TString(phost);
        shingle = CalcShingle(st);
        if (shingle != 0) {
            Lock();

            auto it = host_data.find(shingle);
            if (it != host_data.end()) {
                if ((*it).second == st)
                    res = true;
            }

            UnLock();
        }
    }

    return res;
}

bool TNetIPv6::IsIncludeNet(ui64 shingle_host) const {
    bool res = false;
    if ((shingle_host != 0)) {
        Lock();

        auto it = host_data.find(shingle_host);
        if (it != host_data.end())
            res = true;

        UnLock();
    }

    return res;
}

template <typename... Types>
bool IsIncludeNetImpl(const TNetIPv6& net, Types&&... args) {
    return net.IsIncludeNetHashAlg(std::forward<Types>(args)...) || net.IsIncludeNetSerialAlg(std::forward<Types>(args)...);
}

bool TNetIPv6::IsIncludeNet(TKIPv6 ip) const {
    return IsIncludeNetImpl(*this, ip);
}

bool TNetIPv6::IsIncludeNet(TKIPv6 ip, TString& netsource) const {
    return IsIncludeNetImpl(*this, ip, netsource);
}

bool TNetIPv6::IsIncludeNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const {
    return IsIncludeNetImpl(*this, ip, min_ip_from_diapason);
}

bool TNetIPv6::IsIncludeNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const {
    return IsIncludeNetImpl(*this, ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetSerialAlg(TKIPv6 ip) const {
    TString netsource;
    TKIPv6 min_ip_from_diapason;
    return IsIncludeNetSerialAlg(ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetSerialAlg(TKIPv6 ip, TString& netsource) const {
    TKIPv6 min_ip_from_diapason;
    return IsIncludeNetSerialAlg(ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetSerialAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const {
    TString netsource;
    return IsIncludeNetSerialAlg(ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetSerialAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const {
    bool res = false;

    netsource = "";
    min_ip_from_diapason = TKIPv6();
    if (!ip.Undefined()) {
        Lock();

        res = IsIncludeNetSerialAlgA(ip, netsource, min_ip_from_diapason);

        UnLock();
    }

    return res;
}

bool TNetIPv6::IsIncludeNetHashAlg(TKIPv6 ip) const {
    TString netsource;
    return IsIncludeNetHashAlg(ip, netsource);
}

bool TNetIPv6::IsIncludeNetHashAlg(TKIPv6 ip, TString& netsource) const {
    TKIPv6 min_ip_from_diapason;
    return IsIncludeNetHashAlg(ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetHashAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const {
    TString netsource = "";
    return IsIncludeNetHashAlg(ip, min_ip_from_diapason, netsource);
}

bool TNetIPv6::IsIncludeNetHashAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const {
    bool res = false;
    ui8 mask = 0;

    netsource = "";
    if (!ip.Undefined()) {
        Lock();

        res = IsIncludeNetHashAlgA(ip, netsource, mask);

        UnLock();
    }
    min_ip_from_diapason = ip.GetAddressByMask(mask);

    return res;
}

TNetIPv6::TRecordType TNetIPv6::GetAddresses(const TString& addr_text, TWIPv6B& value) {
    TRecordType res = TUNDEF;
    TWIPv6B wip = TWIPv6B();
    TString stemp = NormalizeStr(addr_text.c_str(), addr_text.length());
    TKIPv6 ip = TKIPv6();

    if (!addr_text.empty()) {
        if (addr_text[0] == '#') {
            res = TCOMMENT;

        } else {
            value = TWIPv6B();
            res = GetRecordType(stemp.c_str(), stemp.length());
            switch (res) {
                case TNetIPv6::THOST:
                    break;
                case TNetIPv6::TIPNET: //range 1 (127.0.0.1/32)
                    if (ParseRangeType1(stemp, wip))
                        value = wip;
                    else
                        res = TUNDEF;
                    break;
                case TNetIPv6::TIPRANGE: //range 2 (127.0.0.1 127.0.0.2)
                    if (ParseRangeType2(stemp, wip))
                        value = wip;
                    else
                        res = TUNDEF;
                    break;
                case TNetIPv6::TIPADDRESS:
                    ip = TKIPv6(stemp.c_str());
                    if (!ip.Undefined()) {
                        value.ip1 = ip;
                        value.ip2 = ip;
                    } else
                        res = TUNDEF;
                    break;
                default:
                    break;
            };
        }
    }

    return res;
}

void TNetIPv6::ClearAll() {
    ClearData();
}

TNetIPv6::TResultState TNetIPv6::AddIPItem(const TString& item_s) {
    TResultState res = PRS_ERROR_UNKNOWN;
    TRecordType rectype = TUNDEF;
    TString tmp_item_s = "";
    TRecordType identify_type = TUNDEF;
    TWIPv6B diapvalue;

    if (!item_s.empty()) {
        rectype = GetAddresses(item_s, diapvalue);
        switch (rectype) {
            case TCOMMENT:
                res = PRS_IS_COMMENT;
                break;
            case TIPNET:
                res = AddItem(item_s, identify_type);
                break;
            case TIPADDRESS:
                if (!diapvalue.ip1.Undefined()) {
                    if (diapvalue.ip1.IsIPv4())
                        tmp_item_s = diapvalue.ip1.toStroka2() + "/32";
                    else
                        tmp_item_s = diapvalue.ip1.toStroka2() + "/128";
                    res = AddItem(tmp_item_s, identify_type);

                } else {
                    res = PRS_ERROR_BAD;
                }

                break;
            default:
                res = RES_ERROR_ILLEGALTYPE;
        };

    } else {
        res = PRS_ERROR_NULL_STR;
    }

    return res;
}

TNetIPv6::TResultState TNetIPv6::AddItem(const TString& item_s, TRecordType& identify_type) {
    TResultState res = PRS_ERROR_UNKNOWN;
    THashStatIPv6 stat;
    int idtype = 0;

    Lock();
    res = (TResultState)ParseItem(true, item_s.c_str(), stat, idtype);
    identify_type = (TRecordType)idtype;

    UnLock();

    return res;
}

//**********************************************************************************************************************
