#pragma once

#ifndef tnetipv6_H
#define tnetipv6_H

#include "util/generic/string.h"
#include "util/system/mutex.h"
#include "util/generic/hash.h"

#include "tkipv6.h"
#include "kfunc.h"
#include "tlogclass.h"

//********************************************************************************************************************
//                                                 THashStatIPv6
//********************************************************************************************************************

struct THashStatIPv6 {
    ui32 trace_count;
    ui32 no_trace_count;
    ui32 bad_count;
    ui32 dublicat_count;

    THashStatIPv6();
    void Clear();
};

//*********************************************************************************************************************
//                                                 THashIPv6Base
//*********************************************************************************************************************

struct TFileStateIPv6 {
    time_t mtime;
    long size;

    TFileStateIPv6();
};

class THashIPv6Base {
public:
    static const int MAX_FILE_SIZE = 33000000;
    enum T_NORMALIZE_URL_MODE { NUM_DEFAULT,
                                NUM_CBB };

private:
    TString m_filename;
    TFileStateIPv6 m_statHashIp;
    bool m_clear_before_load;

    ui32 GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size);
    void ReloadMemListA(const char* buff, size_t buff_size, THashStatIPv6& stat);
    virtual int ParseItem(bool item_add_mode, const char* str, THashStatIPv6& stat, int& identify_type) = 0;
    virtual void ClearData() = 0;
    virtual TString GetResultPrint(THashStatIPv6& stat, ui32 timedelay) = 0;
    virtual void TriggerFunction() = 0;
    virtual void PrefixFunction() = 0;
    virtual void PostfixFunction() = 0;
    bool ExistsFile();
    bool FileStateWasChangedK(const char* fn, TFileStateIPv6* poldbufstat);

public:
    TBaseLogClass* m_Log;
    TString m_ident;
    T_NORMALIZE_URL_MODE m_loadmode;
    TString m_load_ok;
    TString m_load_dublicat;

    mutable TMutex m_Mutex;
    TMutex m_MutexLoad;

    void Lock() const;
    void UnLock() const;
    void LockLoad();
    void UnLockLoad();
    ui64 CalcShingle(const TString& str_data) const;
    TString IntToIpV4(ui32 ip);
    bool IpToIntV4(const char* pstr, int len, ui32* pnet, ui32* pip);

public:
    THashIPv6Base();
    THashIPv6Base(const THashIPv6Base& value);
    virtual ~THashIPv6Base();

    void InitBase(const TString& ident, const TString& filename, TBaseLogClass* LogA, bool clear_before_load);
    void ReloadFileList();
    void ReloadMemList(const TString& data);

    void ReloadFileListCBB();
    void ReloadMemListCBB(const TString& data);
};

//*********************************************************************************************************************
//                                              TFindIPClass
//*********************************************************************************************************************

typedef THashMap<ui64, TString> TKMaskHash;
typedef TKMaskHash::iterator TKMaskHashIt;

struct TMaskIem {
    TKMaskHash m_hash;
    ui32 addcount;
    ui32 dublicat_count;
    ui32 internal_error;

    TMaskIem();
    TMaskIem(TKMaskHash hash);
};

typedef THashMap<ui8, TMaskIem> TKMaskItemHash;
typedef TKMaskItemHash::iterator TKMaskItemHashIt;

struct TMaskItem2 {
    ui8 mask;
    ui32 addcount;
    ui32 dublicat_count;
    ui32 internal_error;

    TMaskItem2();
    TMaskItem2(ui8 maskA, ui32 addcountA, ui32 dublicat_countA, ui32 internal_errorA);

    bool operator<(const TMaskItem2& value) const;
};

typedef std::list<TMaskItem2> TMaskItem2List;
typedef TMaskItem2List::iterator TMaskItem2ListIt;

class TFindIPClass {
private:
    TKMaskItemHash m_masklist;
    TMutex m_Mutex;

    void Lock();
    void UnLock();

public:
    TFindIPClass();

    TString GetNetsStat();
    bool AddNet(const TString& nets, bool& dublicat);
    bool IsIncludeNet(TKIPv6 ip);
    bool IsIncludeNet(TKIPv6 ip, TString& netsource, ui8& mask);
    ui32 size() const;
};

//*********************************************************************************************************************
//                                                 TNetv6
//*********************************************************************************************************************

#define SYMB_RANGE_NI1 0x01
#define SYMB_RANGE_NI2 0x02
#define IS_SYMB_RANGE_NI1(flag) ((flag)&SYMB_RANGE_NI1)
#define IS_SYMB_RANGE_NI2(flag) ((flag)&SYMB_RANGE_NI2)

struct TWIPv6B {
    TKIPv6 ip1;
    TKIPv6 ip2;
    TString source_net;
};

typedef std::list<TWIPv6B> TWIPv6BList;
typedef TWIPv6BList::iterator TWIPv6BListIt;

typedef THashMap<ui64, TString> TStrokaHashB;
typedef TStrokaHashB::iterator TStrokaHashBIt;

//После создания объекта (выполнения конструктора) требуется выполнить:
//obj.ReloadFileList(); - если список сетей задан в файле
//либо
//obj.ReloadMemList(const Stroka &data); - если список сетей задан в виде строки

class TNetIPv6: public THashIPv6Base {
public:
    enum TRecordType { TUNDEF,
                       THOST,
                       TIPNET,
                       TIPRANGE,
                       TIPADDRESS,
                       TCOMMENT };
    enum TResultState { PRS_OK = 1,
                        PRS_IS_COMMENT = 2,
                        PRS_ERROR_UNKNOWN = -1,
                        PRS_ERROR_NULL_STR = -2,
                        PRS_ERROR_EMPTY_NORMSTR = -3,
                        PRS_ERROR_NULL_STORAGE = -4,
                        PRS_ERROR_DUBLICAT = -5,
                        PRS_ERROR_BAD = -6,
                        PRS_ERROR_UNKNOWN_TYPE = -7,
                        RES_ERROR_ILLEGALTYPE = -8 };

private:
    TStrokaHashB host_data;
    TStrokaHashB host_data_temp;
    TWIPv6BList range_data;
    TWIPv6BList range_data_temp;
    THolder<TFindIPClass> ipnetobj;
    THolder<TFindIPClass> ipnetobj_temp;
    ui8 SymbTable[256];
    TString idents;

    void DefaultConstructor();
    virtual int ParseItem(bool item_add_mode, const char* str, THashStatIPv6& stat, int& identify_type);
    virtual void ClearData();
    virtual TString GetResultPrint(THashStatIPv6& stat, ui32 timedelay);
    virtual void TriggerFunction();
    virtual void PrefixFunction();
    virtual void PostfixFunction();
    TString NormalizeStr(const char* str, int strlength);
    TRecordType GetRecordType(const char* str, int strlength);
    void InitTable();
    bool IsAllowSymbolRange1(char symb);
    bool IsAllowSymbolRange2(char symb);
    bool ParseRangeType1(const TString& text, TWIPv6B& value);
    bool ParseRangeType2(const TString& text, TWIPv6B& value);

    bool IsIncludeNetSerialAlgA(TKIPv6 ip, TString& netsource, TKIPv6& min_ip_from_diapason) const;
    bool IsIncludeNetHashAlgA(TKIPv6 ip, TString& netsource, ui8& mask) const;

public:
    //нормализуем урл для нужд ЦББ
    TString NormalizeUrl(const TString& urlA) const;

    //serial algorithm (не использовать непосредственно, лучше использовать обобщенный функции IsIncludeNet(...))
    bool IsIncludeNetSerialAlg(TKIPv6 ip) const;                                                   //true, если ip входит в состав сети, указанной в списке сетей (алгоритм: последовательный перебор)
    bool IsIncludeNetSerialAlg(TKIPv6 ip, TString& netsource) const;                               //true, если ip входит в состав сети, указанной в списке сетей; в netsource возвращается сеть, куда попал ip (алгоритм: последовательный перебор)
    bool IsIncludeNetSerialAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const;                     //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip (алгоритм: последовательный перебор)
    bool IsIncludeNetSerialAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const; //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip; в netsource возвращается сеть, куда попал ip (алгоритм: последовательный перебор)

    //hash algorithm (не использовать непосредственно, лучше использовать обобщенный функции IsIncludeNet(...))
    bool IsIncludeNetHashAlg(TKIPv6 ip) const;                                                   //true, если ip входит в состав сети, указанной в списке сетей (алгоритм на хэшах)
    bool IsIncludeNetHashAlg(TKIPv6 ip, TString& netsource) const;                               //true, если ip входит в состав сети, указанной в списке сетей; в netsource возвращается сеть, куда попал ip (алгоритм на хэшах)
    bool IsIncludeNetHashAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const;                     //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip (алгоритм на хэшах)
    bool IsIncludeNetHashAlg(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const; //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip; в netsource возвращается сеть, куда попал ip (алгоритм на хэшах)
public:
    TNetIPv6();
    TNetIPv6(const TString& ident);
    virtual ~TNetIPv6();

    //Способы задания сетей в файле (или памяти):
    //#                                    - комментарий
    //2a02:2f0c:3000::/28                  - сеть IPv6
    //2a02:2f0c:3000::1 2a02:2f0c:3000::2  - диапазон IPv6
    //127.0.0.1/32                         - сеть IPv4
    //127.0.0.1 127.0.0.2                  - диапазон IPv4
    //yandex.ru                            - хост

    void Init(const TString& filename, TBaseLogClass* LogA); //сети вида 127.0.0.1/24 грузятся в хэш, вида 127.0.0.1 127.0.0.2 в список для последовательного алгоритма
    //после выполнения Init(...) требуется вызвать ф-ии ReloadFileList() или ReloadMemList(const Stroka &data) для загрузки базы в объект соотвественно с файла или из списка в памяти, заданного строкой

    bool IsIncludeNet(const char* phost) const;     //true, если phost содержится в списке хостов
    bool IsIncludeNet(ui64 shingle_host) const;     //true, если хост (его шингл) содержится в списке хостов

    bool IsIncludeNet(TKIPv6 ip) const;                                                   //true, если ip входит в состав сети, указанной в списке сетей
    bool IsIncludeNet(TKIPv6 ip, TString& netsource) const;                               //true, если ip входит в состав сети, указанной в списке сетей; в netsource возвращается сеть, куда попал ip
    bool IsIncludeNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason) const;                     //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip
    bool IsIncludeNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason, TString& netsource) const; //true, если ip входит в состав сети, указанной в списке сетей; min_ip_from_diapason - нижняя граница сети, куда входит ip; в netsource возвращается сеть, куда попал ip

    TRecordType GetAddresses(const TString& addr_text, TWIPv6B& value); //по строке addr_text распознаем тип и указываем диапазон адресов (если имеет смысл)

    void ClearAll();                                                                   //очистить хранилище
    TNetIPv6::TResultState AddIPItem(const TString& item_s);                           //добавить элемент только типов 127.0.0.1/24(TIPNET) или 127.0.0.1(TIPADDRESS) (распарсится и добавится в хранилище)
    TNetIPv6::TResultState AddItem(const TString& item_s, TRecordType& identify_type); //добавить элемент (распарсится и добавится в хранилище, в identify_type возвращает тип, во что распарсилась строка)
};

//*********************************************************************************************************************

#endif
