#pragma once

#ifndef TNETIPV6_UT_HELPERS_H_
#define TNETIPV6_UT_HELPERS_H_

#include "tkipv6.h"

#include <iostream>
#include <exception>
#include <stdexcept>
#include <util/generic/noncopyable.h>
#include <util/random/random.h>

namespace helpers {
    class ip_difference {
    public:
        ip_difference()
            : m_overflow(0)
            , m_high(0)
            , m_low(0)
        { }

        ip_difference(const ui64& low)
            : m_overflow(0)
            , m_high(0)
            , m_low(low)
        { }

        ip_difference(const TKIPv6& ip)
            : m_overflow(0)
            , m_high(ip.HighPart())
            , m_low(ip.LowPart())
        { }

        TKIPv6 toIP() const {
            const ui64 max64 = std::numeric_limits<ui64>::max();

            return (m_overflow == 0) ? TKIPv6(m_high, m_low) : (m_overflow < 0) ? TKIPv6(0, 0) : TKIPv6(max64, max64);
        }

        ip_difference operator-() const {
            ip_difference result(~m_low);
            result.m_high = ~m_high;
            result.m_overflow = ~m_overflow;
            result += 1;

            return result;
        }

        ip_difference& operator+=(const ip_difference& v) {
            if (add(m_low, v.m_low) != 0)
                m_overflow += add(m_high, 1);
            m_overflow += add(m_high, v.m_high);
            m_overflow += v.m_overflow;
            return *this;
        }

        ip_difference operator+(const ip_difference& v) const {
            return ip_difference(*this) += v;
        }

        ip_difference& operator-=(const ip_difference& v) {
            ip_difference negate_v = -v;
            return operator+=(negate_v);
        }

        ip_difference operator-(const ip_difference& v) {
            return ip_difference(*this) -= v;
        }

        bool operator<(const ip_difference& v) const {
            if (m_overflow != v.m_overflow)
                return m_overflow < v.m_overflow;

            bool result = toIP() < v.toIP();
            return (m_overflow >= 0) ? result : !result;
        }

        ip_difference operator>>(const ui64 shift) const {
            if (shift == 0)
                return *this;

            if (shift >= (sizeof(m_overflow) + sizeof(m_high) + sizeof(m_low)) * 8)
                return ip_difference();

            if (shift > 64)
                return (*this >> 64) >> (shift - 64);

            ip_difference result;

            if (shift == 64) {
                result.m_high = static_cast<ui8>(m_overflow);
                result.m_low = m_high;
            } else {
                ui64 part = 0;
                result.m_overflow = static_cast<i8>(shift_impl(static_cast<ui8>(m_overflow), shift, 0, part));
                result.m_high = shift_impl(m_high, shift, part, part);
                result.m_low = shift_impl(m_low, shift, part, part);
            }

            return result;
        }

        ip_difference operator~() const {
            ip_difference result;

            result.m_overflow = ~m_overflow;
            result.m_high = ~m_high;
            result.m_low = ~m_low;

            return result;
        }

        ip_difference& operator&=(const ip_difference& other) {
            m_overflow &= other.m_overflow;
            m_high &= other.m_high;
            m_low &= other.m_low;

            return *this;
        }

        ip_difference& operator|=(const ip_difference& other) {
            m_overflow |= other.m_overflow;
            m_high |= other.m_high;
            m_low |= other.m_low;

            return *this;
        }

        ip_difference operator|(const ip_difference& other) const {
            return ip_difference(*this) |= other;
        }

    private:
        static ui64 shift_impl(const ui64 base, const ui64 shift, const ui64 from_prev, ui64& to_next) {
            const ui64 max64 = std::numeric_limits<ui64>::max();
            to_next = base & (max64 >> (64 - shift));
            return (base >> shift) | (from_prev << (64 - shift));
        }

        ui8 add(ui64& base, const ui64& v) {
            ui8 result = (v > (std::numeric_limits<ui64>::max() - base)) ? 1 : 0;
            base += v;
            return result;
        }

        i8 m_overflow;
        ui64 m_high, m_low;
    };

    template <typename Derived>
    class iteratorIPvBase
       : public std::iterator<std::random_access_iterator_tag, TKIPv6, ip_difference> {
        typedef iteratorIPvBase<Derived> this_type;

    public:
        iteratorIPvBase() {
        }

        template <typename T>
        iteratorIPvBase(const T& value)
            : m_value(value)
        {
        }

        static const Derived& begin() {
            static const Derived it_begin(Derived::begin_value());
            return it_begin;
        }

        static const Derived& end() {
            static const Derived it_end(Derived::end_value());
            return it_end;
        }

        bool equal(const this_type& other) const {
            return other.m_value == m_value;
        }

        TKIPv6& operator*() const {
            return const_cast<TKIPv6&>(m_value);
        }

        TKIPv6* operator->() const {
            return &operator*();
        }

        void increment() {
            static_cast<Derived*>(this)->advance(1);
        }

        void advance(const ip_difference& v) {
            m_value = (v + m_value).toIP();
        }

        ip_difference distance_to(const this_type& v) const {
            return ip_difference(v.m_value) - ip_difference(m_value);
        }

        Derived& operator++() {
            increment();
            return *static_cast<Derived*>(this);
        }

        Derived& operator+=(const ip_difference& v) {
            advance(v);
            return *static_cast<Derived*>(this);
        }

        Derived operator+(const ip_difference& v) const {
            return Derived(*this) += v;
        }

        ip_difference operator-(const this_type& v) const {
            return distance_to(v);
        }

        bool operator<(const this_type& other) const
        {
            return ip_difference(m_value) < ip_difference(other.m_value);
        }

    protected:
        static TKIPv6 begin_value() {
            return TKIPv6(0, 1);
        }

        static TKIPv6 end_value() {
            static const ui64 max64 = std::numeric_limits<ui64>::max();
            return TKIPv6(max64, max64);
        }

        TKIPv6 m_value;
    };

    class iteratorIPv4
       : public iteratorIPvBase<iteratorIPv4> {
        typedef iteratorIPvBase<iteratorIPv4> super;

    public:
        iteratorIPv4() {
        }

        template <typename T>
        iteratorIPv4(const T& value)
            : super(value)
        {
            if (m_value.IsIPv6() && m_value != end_value())
                throw std::runtime_error("not IPv4");
        }

        void advance(const ip_difference& v) {
            super::advance(v);
            validate();
        }

    private:
        friend iteratorIPvBase<iteratorIPv4>;
        static TKIPv6 begin_value() {
            return TKIPv6(ui32(1));
        }

        static TKIPv6 end_value() {
            return TKIPv6(0, 0xFFFFFFFFFFFFL + 1);
        }

        void validate() {
            if (m_value.IsIPv6())
                *this = end();
        }
    };

    class iteratorIPv6
       : public iteratorIPvBase<iteratorIPv6> {
        typedef iteratorIPvBase<iteratorIPv6> super;

    public:
        iteratorIPv6() {
        }

        template <typename T>
        iteratorIPv6(const T& value)
            : super(value)
        {
        }
    };

    template <typename T>
    inline T get_step(const T& v) {
        static const ui8 shift = 16;
        return std::max<T>(v >> shift, 1);
    }

    template <typename T>
    inline T get_random(const T& v) {
        return RandomNumber(v);
    }

    inline TKIPv6 get_random(const TKIPv6& v) {
        return (v.HighPart() == 0) ? TKIPv6(0, get_random(v.LowPart())) : TKIPv6(get_random(v.HighPart()), RandomNumber<ui64>());
    }

    inline ip_difference get_random(const ip_difference& v) {
        return get_random(v.toIP());
    }

    template <typename T>
    inline T get_random_start(const T& v) {
        return get_random(get_step(v));
    }

    inline ui64 make_ipv4(const ui64& v) {
        return v | 0xFFFF00000000L;
    }
}

#endif
