#include <util/stream/mem.h>
#include <util/stream/buffer.h>
#include "tpoolhttpclients.h"
#include "limited_output.h"
#include "shtime.h"

namespace poolhttpcl {

    TString TReqErrorToTString(TReqError err) {
        TString res = "";

        switch (err) {
            case KNONEERROR:
                res = "OK";
                break;
            case KHOSTPORTERROR:
                res = "HOSTPORTERR";
                break;
            case KWAITERROR:
                res = "WAITERR";
                break;
            case KCONNECTERROR:
                res = "CE";
                break;
            case KCONNECTTIMEOUTERROR:
                res = "CT";
                break;
            case KREQUESTERROR:
                res = "RE";
                break;
            case KREQUESTTIMEOUTERROR:
                res = "RT";
                break;
            case KREQUESTANSWERERROR:
                res = "RQSTANSWERR";
                break;
        };

        return res;
    }

    //****************************************************************************************************************************************
    //                                                     THttpBase
    //****************************************************************************************************************************************

    THttpBase::THttpBase() {
        Clear();
    }

    THttpBase::~THttpBase() {
        Clear();
    }

    void THttpBase::Clear() {
        m_rqst_paramlist.clear();

        m_res_empty = true;
        m_res_error = KNONEERROR;
        m_res_code = 0;
        m_res_error_txt = "";
        m_res_answer_from_server = "";
    }

    TParametrsList* THttpBase::GetRequest() {
        return &m_rqst_paramlist;
    }

    bool THttpBase::IsValid() {
        return true;
    }

    void THttpBase::SetMessage() {
        Clear();
    }

    void THttpBase::ADDField(const TString& param_ident, const char* data, size_t datasize) {
        TParametrsListIt it;
        const bool exists = m_rqst_paramlist.cend() != std::find_if(m_rqst_paramlist.cbegin(), m_rqst_paramlist.cend(), [&param_ident](const TParamItem& item) {
                                return item.name() == param_ident;
                            });

        if (!exists)
            m_rqst_paramlist.emplace_back(param_ident, TString{data, datasize});
    }

    void THttpBase::SetDublicatLoginMode(bool is_dublicat_login) {
        TString dubl_login = "";

        if (is_dublicat_login)
            dubl_login = "1";
        else
            dubl_login = "0";

        ADDField("is_dublicat_login", dubl_login.c_str(), dubl_login.length());
    }

    TString THttpBase::toLog() {
        TString res = "";

        if (m_res_empty)
            res = "empty,err=" + TReqErrorToTString(m_res_error);
        else
            res = "err=" + TReqErrorToTString(m_res_error) + ",data='" + m_res_answer_from_server + "'";

        return res;
    }

    //*********************************************************************************************************************
    //                                                    THttpClient
    //*********************************************************************************************************************

    THttpClient::THttpClient() {
        m_name = "";
        m_Log = NULL;
        m_port = 0;
        m_fullhost = "";
        m_only_ipv4 = false;
    }

    void THttpClient::Init(const TString& name, const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, TLogClass* LogA, bool only_ipv4, const TString& caPath) {
        m_name = name;
        m_only_ipv4 = only_ipv4;
        m_Log = LogA;
        SetHostData(host, port, ConnectTimeOutMSec, GetTimeOutMSec, caPath);
    }

    void THttpClient::SetHostData(const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, const TString& caPath) {
        m_HostDataMutex.Acquire();

        m_fullhost = host;
        m_port = port;

        m_ConnectTimeOut = TDuration::MilliSeconds(ConnectTimeOutMSec);
        m_GetTimeOut = TDuration::MilliSeconds(GetTimeOutMSec);

        if (caPath)
            curl.Setup(NCurl::TSSL().SetCACertFile(caPath));
        m_HostDataMutex.Release();
    }

    TKWTStringCl THttpClient::GetHost(const TString& host) {
        TKWTStringCl res;
        TString http_s1 = "http://";
        TString http_s2 = "HTTP://";
        const char* p = NULL;
        const char* ps = NULL;

        p = strstr(host.c_str(), http_s1.c_str());
        if (p == NULL) {
            p = strstr(host.c_str(), http_s2.c_str());
            if (p == NULL)
                p = host.c_str();
            else
                p = p + http_s2.length();
        } else
            p = p + http_s1.length();
        if (p != NULL) {
            ps = strstr(p, "/");
            if (ps != NULL) {
                res.host = TString(p, ps - p);
                if ((ps + 1) != NULL)
                    res.url = TString(ps + 1);
            } else {
                res.host = TString(p);
                res.url = "";
            }
        }

        return res;
    }

    void THttpClient::GetRequestToServerA(THttpBase& data, TPrintBuff& PBUFF, const TString& NumbRequest, ui32& connect_time, ui32& request_time) {
        connect_time = 0;
        request_time = 0;
        if (!m_fullhost.empty() && (m_port != 0)) {
            m_statdata.today.IncAllMailCount();
            data.m_res_error = KNONEERROR;
            m_HostDataMutex.Acquire();

            bool m_Result;
            ui32 BeginProcessTime = 0;
            ui16 m_HttpStatus = 0;
            TString request_url = "";
            TString full_request_url = "";
            //ui32                  datalen          = 0;
            //ui32                  shinglecount     = 0;
            TString txt = "";
            //time_t                dlv_time         = 0;
            int err = 0;
            TString printshlist = "";

            BeginProcessTime = CShingleTime::GetMs();
            connect_time = CShingleTime::GetMs();

            BeginProcessTime = CShingleTime::GetMs();

            TParametrsList* paramlistp = data.GetRequest();

            if (paramlistp != NULL) {
                //if (m_url.empty())
                //   m_url = data.GetRqstIdentDefault();

                request_time = CShingleTime::GetMs();

                TStringStream response;
                {
                    NCurl::TArtifacts artifacts(response);
                    NCurl::TRequestContext requestContext;
                    requestContext.SetHost(m_fullhost);
                    requestContext.SetPort(m_port);
                    requestContext.SetRequestTimeout(m_GetTimeOut);
                    requestContext.SetConnectTimeout(m_ConnectTimeOut);
                    requestContext.SetPostData(MakeCGIString(*paramlistp));


                    if (auto error = curl.Setup(std::move(requestContext)).Perform(artifacts)) {
                        m_Result = false;
                    } else {
                        m_Result = true;
                        m_HttpStatus = artifacts.code;
                    }
                }

                request_time = CShingleTime::GetMs() - request_time;

                BeginProcessTime = CShingleTime::GetMs() - BeginProcessTime;

                data.m_res_code = m_HttpStatus;

                if (true == m_Result) {
                    if (200 == m_HttpStatus) {
                        data.m_res_empty = false;

                        m_statdata.today.IncRequest200Count();
                        data.m_res_answer_from_server = response.Str();
                        data.m_res_error = KNONEERROR;
                        data.m_res_error_txt = "OK";
                        if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                            PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Mess! %u OK (HttpCode:200) - answer='%s'[%s]", NumbRequest.c_str(), m_name.c_str(), BeginProcessTime, response.Str().c_str(), data.toLog().c_str());

                    } else if ((400 <= m_HttpStatus) && (m_HttpStatus < 500)) {
                        data.m_res_error = KREQUESTANSWERERROR;
                        data.m_res_error_txt = IntToStroka(m_HttpStatus) + "(" + m_fullhost + ":" + IntToStroka(m_port) + ")";
                        if (m_HttpStatus == 404)
                            m_statdata.today.IncRequest404Count();
                        else
                            m_statdata.today.IncRequest400500Count();
                        if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                            PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Err! %u HttpCode:%u - '%s'", NumbRequest.c_str(), m_name.c_str(), BeginProcessTime, m_HttpStatus, full_request_url.c_str());

                    } else {
                        data.m_res_error = KREQUESTANSWERERROR;
                        data.m_res_error_txt = IntToStroka(m_HttpStatus) + "(" + m_fullhost + ":" + IntToStroka(m_port) + ")";
                        m_statdata.today.IncRequestOtherCount();
                        if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                            PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Err! %u HttpCode:%u - '%s'", NumbRequest.c_str(), m_name.c_str(), BeginProcessTime, m_HttpStatus, full_request_url.c_str());
                    }

                } else {
                    data.m_res_error = KREQUESTERROR;
                    data.m_res_error_txt = "RE(" + m_fullhost + ":" + IntToStroka(m_port) + ")";
                    m_statdata.today.IncRequestError();
                    if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                        PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Err! %u rqst error (%d) - '%s'", NumbRequest.c_str(), m_name.c_str(), BeginProcessTime, err, full_request_url.c_str());

                }
            }

            m_HostDataMutex.Release();
        } else {
            if (m_fullhost.empty()) {
                data.m_res_error = KHOSTPORTERROR;
                data.m_res_error_txt = "NOHOST";
                if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                    PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Err! no defined host", NumbRequest.c_str(), m_name.c_str());

            } else if (m_port == 0) {
                data.m_res_error = KHOSTPORTERROR;
                data.m_res_error_txt = "NOPORT";
                if ((PBUFF.printbuffsize - PBUFF.in_printbuff - 1) > 0)
                    PBUFF.in_printbuff += snprintf(PBUFF.printbuff.Data() + PBUFF.in_printbuff, PBUFF.printbuffsize - PBUFF.in_printbuff - 1, "%s %s: Err! no defined port", NumbRequest.c_str(), m_name.c_str());
            }
        }
    }

    TStatDataCl THttpClient::GetStat() {
        return m_statdata;
    }

    void THttpClient::Midnight() {
        m_statdata.Midnight();
    }

    bool THttpClient::NoDefinedHostOrPort() {
        bool res = false;

        m_HostDataMutex.Acquire();

        if ((m_fullhost.empty()) || (m_port == 0))
            res = true;

        m_HostDataMutex.Release();

        return res;
    }

    void THttpClient::RequestToServer(THttpBase& data, const TString& NumbRequest, ui32& connect_time, ui32& request_time, ui32& writelog_time) {
        writelog_time = 0;
        PBUFF.Clear();
        GetRequestToServerA(data, PBUFF, NumbRequest, connect_time, request_time);
        if (m_Log != NULL) {
            writelog_time = CShingleTime::GetMs();
            m_Log->WriteMessageAndData("%s", PBUFF.GetNormalizData().c_str()); //PBUFF.printbuff
            writelog_time = CShingleTime::GetMs() - writelog_time;
        }
    }

    //*********************************************************************************************************************
    //                                           TPoolHttpClients
    //*********************************************************************************************************************

    TPoolHttpClients::TPoolHttpClients(const TString& name) {
        m_Log = NULL;
        m_name = name;
    }

    void TPoolHttpClients::Init(int clcount, const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, TLogClass* LogA, bool only_ipv4, const TString& caPath) {
        THttpClientExt* clientext = NULL;

        m_Log = LogA;
        if (clcount > 0) {
            for (int i = 0; i < clcount; i++) {
                clientext = new THttpClientExt();
                if (clientext != NULL) {
                    clientext->client.Init(m_name, host, port, ConnectTimeOutMSec, GetTimeOutMSec, m_Log, only_ipv4, caPath);
                    clientext->busy = false;

                    cllist.emplace_back(clientext);
                }
            }
        }
    }

    void TPoolHttpClients::SetHostData(TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, const TString& caPath) {
        THttpClientExtListIt it;

        m_ActionMutex.Acquire();

        it = cllist.begin();
        while (it != cllist.end()) {
            if ((*it))
                (*it)->client.SetHostData(host, port, ConnectTimeOutMSec, GetTimeOutMSec, caPath);

            ++it;
        }

        m_ActionMutex.Release();
    }

    TStatDataCl TPoolHttpClients::GetStat() {
        TStatDataCl res;
        THttpClientExtListIt it;

        m_ActionMutex.Acquire();

        it = cllist.begin();
        while (it != cllist.end()) {
            if ((*it))
                res += (*it)->client.GetStat();

            ++it;
        }

        m_ActionMutex.Release();

        return res;
    }

    void TPoolHttpClients::Midnight() {
        THttpClientExtListIt it;

        m_ActionMutex.Acquire();

        it = cllist.begin();
        while (it != cllist.end()) {
            if ((*it))
                (*it)->client.Midnight();

            ++it;
        }

        m_ActionMutex.Release();
    }

    void TPoolHttpClients::RequestToServer(THttpBase& data, const TString& NumbRequest) {
        TDebugTiming debug_time = TDebugTiming();

        RequestToServer(data, NumbRequest, 100, debug_time);
    }

    void TPoolHttpClients::RequestToServer(THttpBase& data, const TString& NumbRequest, ui32 wait_item_period_max_ms, TDebugTiming& debug_time) {
        debug_time.m_fulltime_ms = CShingleTime::GetMs();

        THttpClientExtListIt it;
        THttpClientExt* clientext = NULL;
        ui32 proc_time = 0;
        ui32 proc_time_t = 0;
        ui32 wait_time = 0;
        ui32 wait_item_period_max = 50;

        if ((wait_item_period_max_ms > 0) && (wait_item_period_max != wait_item_period_max_ms))
            wait_item_period_max = wait_item_period_max_ms;

        if (data.IsValid()) {
            proc_time = CShingleTime::GetMs();
            wait_time = CShingleTime::GetMs();
            while (true) {
                proc_time_t = CShingleTime::GetMs();
                if ((proc_time_t - proc_time) > wait_item_period_max) {
                    data.m_res_error = KWAITERROR;
                    break;
                }

                m_ActionMutex.Acquire();

                it = cllist.begin();
                while (it != cllist.end()) {
                    if (*it && (!(*it)->busy)) {
                        clientext = (*it).Get();
                        clientext->busy = true;
                        break;
                    }

                    ++it;
                }

                m_ActionMutex.Release();

                if (clientext != NULL)
                    break;
                else
                    usleep(10000);
            }
            wait_time = CShingleTime::GetMs() - wait_time;
            debug_time.m_wait_ms = wait_time;

            if (clientext != NULL) {
                clientext->client.RequestToServer(data, NumbRequest, debug_time.m_connect_ms, debug_time.m_request_ms, debug_time.m_writelog_ms);

                m_ActionMutex.Acquire();
                clientext->busy = false;
                m_ActionMutex.Release();
            }
        }

        debug_time.m_fulltime_ms = CShingleTime::GetMs() - debug_time.m_fulltime_ms;
    }

} // namespace poolhttpcl
