#pragma once

#include <util/generic/string.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include "tlogclass.h"
#include "shconn.h"
#include "kfunc.h"
#include <list>
#include "ttrafficcontrol.h"
#include <mail/so/libs/curl/curl.h>

namespace poolhttpcl {

    //****************************************************************************************************************************************
    //                                                    THttpAnswer
    //****************************************************************************************************************************************

    enum TReqError { KNONEERROR,
                     KHOSTPORTERROR,
                     KWAITERROR,
                     KCONNECTERROR,
                     KCONNECTTIMEOUTERROR,
                     KREQUESTERROR,
                     KREQUESTTIMEOUTERROR,
                     KREQUESTANSWERERROR };

    TString TReqErrorToTString(TReqError err);

    //****************************************************************************************************************************************
    //                                                     THttpBase
    //****************************************************************************************************************************************

    class THttpBase {
    private:
        TParametrsList m_rqst_paramlist;

    public:
        bool m_res_empty;
        TReqError m_res_error;
        ui16 m_res_code;
        TString m_res_error_txt;
        TString m_res_answer_from_server;

    public:
        THttpBase();
        virtual ~THttpBase();

        void Clear();
        TParametrsList* GetRequest();
        bool IsValid();

        void SetMessage();
        void ADDField(const TString& param_ident, const char* data, size_t datasize);
        void SetDublicatLoginMode(bool is_dublicat_login);

        TString toLog();
    };

    //****************************************************************************************************************************************

    struct TPrintBuff {
        size_t printbuffsize;
        TBuffer printbuff;
        size_t in_printbuff;

        TPrintBuff() {
            printbuffsize = 65000;
            printbuff.Resize(printbuffsize);
            in_printbuff = 0;
        }

        void Clear() {
            printbuff.Resize(printbuffsize);
            printbuff.Fill(0, printbuffsize);
            in_printbuff = 0;
        }

        TString GetNormalizData() {
            TString res = "";

            for (size_t i = 0; i < in_printbuff; i++) {
                if (*(printbuff.Data() + i) < ' ')
                    res += " ";
                else
                    res += TString(*(printbuff.Data() + i));
            }

            return res;
        }
    };

    struct TStatDataItem {
        ui32 all_mail_count{};
        ui32 connect_error{};
        ui32 connect_timeout{};
        ui32 request_200_count{};
        ui32 request_404_count{};
        ui32 request_400500_count{};
        ui32 request_other_count{};
        ui32 request_error{};
        ui32 request_timeout{};
        ui32 request_buffer_overflow{};

        TStatDataItem() = default;

        void IncAllMailCount() {
            all_mail_count = IncMax32(all_mail_count, 1);
        }
        void IncConnectError() {
            connect_error = IncMax32(connect_error, 1);
        }
        void IncConnectTimeout() {
            connect_timeout = IncMax32(connect_timeout, 1);
        }
        void IncRequest200Count() {
            request_200_count = IncMax32(request_200_count, 1);
        }
        void IncRequest404Count() {
            request_404_count = IncMax32(request_404_count, 1);
        }
        void IncRequest400500Count() {
            request_400500_count = IncMax32(request_400500_count, 1);
        }
        void IncRequestOtherCount() {
            request_other_count = IncMax32(request_other_count, 1);
        }
        void IncRequestError() {
            request_error = IncMax32(request_error, 1);
        }
        void IncRequestTimeout() {
            request_timeout = IncMax32(request_timeout, 1);
        }
        void IncRequsetBufferOverflow() {
            request_buffer_overflow = IncMax32(request_buffer_overflow, 1);
        }

        void Clear() {
            all_mail_count = 0;
            connect_error = 0;
            connect_timeout = 0;
            request_200_count = 0;
            request_404_count = 0;
            request_400500_count = 0;
            request_other_count = 0;
            request_error = 0;
            request_timeout = 0;
            request_buffer_overflow = 0;
        }

        TStatDataItem& operator+=(const TStatDataItem& obj) {
            this->all_mail_count = IncMax32(this->all_mail_count, obj.all_mail_count);
            this->connect_error = IncMax32(this->connect_error, obj.connect_error);
            this->connect_timeout = IncMax32(this->connect_timeout, obj.connect_timeout);
            this->request_200_count = IncMax32(this->request_200_count, obj.request_200_count);
            this->request_404_count = IncMax32(this->request_404_count, obj.request_404_count);
            this->request_400500_count = IncMax32(this->request_400500_count, obj.request_400500_count);
            this->request_other_count = IncMax32(this->request_other_count, obj.request_other_count);
            this->request_error = IncMax32(this->request_error, obj.request_error);
            this->request_timeout = IncMax32(this->request_timeout, obj.request_timeout);
            this->request_buffer_overflow = IncMax32(this->request_buffer_overflow, obj.request_buffer_overflow);

            return *this;
        }

        TStatDataItem& operator-=(const TStatDataItem& obj) {
            this->all_mail_count = Minus32(this->all_mail_count, obj.all_mail_count);
            this->connect_error = Minus32(this->connect_error, obj.connect_error);
            this->connect_timeout = Minus32(this->connect_timeout, obj.connect_timeout);
            this->request_200_count = Minus32(this->request_200_count, obj.request_200_count);
            this->request_404_count = Minus32(this->request_404_count, obj.request_404_count);
            this->request_400500_count = Minus32(this->request_400500_count, obj.request_400500_count);
            this->request_other_count = Minus32(this->request_other_count, obj.request_other_count);
            this->request_error = Minus32(this->request_error, obj.request_error);
            this->request_timeout = Minus32(this->request_timeout, obj.request_timeout);
            this->request_buffer_overflow = Minus32(this->request_buffer_overflow, obj.request_buffer_overflow);

            return *this;
        }
    };

    struct TStatDataCl {
        ui16 compl_threads;
        bool enable_dlvclient;
        TStatDataItem today;
        TStatDataItem yesterday;

        TStatDataCl() {
            compl_threads = 0;
            enable_dlvclient = false;
        }

        void Midnight() {
            yesterday = today;
            today.Clear();
        }
        TStatDataCl& operator+=(const TStatDataCl& obj) {
            this->today += obj.today;
            this->yesterday += obj.yesterday;

            return *this;
        }
    };

    struct TKWTStringCl {
        TString host;
        TString url;

        TKWTStringCl() {
            host = "";
            url = "";
        }
    };

    //*********************************************************************************************************************
    //                                                  THttpClientProp
    //*********************************************************************************************************************

    struct THttpClientProp {
        THttpClientProp() = default;
        ui32 m_clcount{};
        TString m_host;
        ui32 m_port{};
        ui32 m_contimeout{};
        ui32 m_reqtimeout{};
        TString m_ca;
        bool m_onlyipv4{};
    };

    //*********************************************************************************************************************
    //                                                    THttpClient
    //*********************************************************************************************************************

    class THttpClient {
    private:

        TString m_name;
        NCurl::TCurl curl;

        TLogClass* m_Log;
        TMutex m_HostDataMutex;
        ui16 m_port;
        TDuration m_ConnectTimeOut = TDuration::MilliSeconds(100), m_GetTimeOut = TDuration::MilliSeconds(200);
        TString m_fullhost;
        TStatDataCl m_statdata;
        TPrintBuff PBUFF;
        bool m_only_ipv4;

        TKWTStringCl GetHost(const TString& host);
        void GetRequestToServerA(THttpBase& data, TPrintBuff& PBUFF, const TString& NumbRequest, ui32& connect_time, ui32& request_time);

    public:
        THttpClient();

        void Init(const TString& name, const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, TLogClass* LogA, bool only_ipv4, const TString& caPath = nullptr);
        void SetHostData(const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, const TString& caPath = nullptr);
        TStatDataCl GetStat();

        void Midnight();
        bool NoDefinedHostOrPort();
        void RequestToServer(THttpBase& data, const TString& NumbRequest, ui32& connect_time, ui32& request_time, ui32& writelog_time);
    };

    //*********************************************************************************************************************
    //                                           TPoolHttpClients
    //*********************************************************************************************************************

    struct THttpClientExt {
        THttpClient client;
        bool busy;

        THttpClientExt() {
            busy = false;
        }
    };

    struct TDebugTiming {
        ui32 m_fulltime_ms;
        ui32 m_wait_ms;
        ui32 m_connect_ms;
        ui32 m_request_ms;
        ui32 m_writelog_ms;

        TDebugTiming() {
            Clear();
        }

        void Clear() {
            m_fulltime_ms = 0;
            m_wait_ms = 0;
            m_connect_ms = 0;
            m_request_ms = 0;
            m_writelog_ms = 0;
        }
    };

    typedef std::list<THolder<THttpClientExt>> THttpClientExtList;
    typedef THttpClientExtList::iterator THttpClientExtListIt;

    class TPoolHttpClients {
    private:
        THttpClientExtList cllist;
        TString m_name;
        TMutex m_ActionMutex;
        TLogClass* m_Log;

    public:
        TPoolHttpClients(const TString& name);

        void Init(int clcount, const TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, TLogClass* LogA, bool only_ipv4, const TString& caPath = nullptr);
        void SetHostData(TString& host, ui16 port, ui32 ConnectTimeOutMSec, ui32 GetTimeOutMSec, const TString& caPath = nullptr);
        TStatDataCl GetStat();
        void Midnight();
        TString GetName() {
            return m_name;
        }

        void RequestToServer(THttpBase& data, const TString& NumbRequest);
        void RequestToServer(THttpBase& data, const TString& NumbRequest, ui32 wait_item_period_max_ms, TDebugTiming& debug_time);
    };

    //*********************************************************************************************************************

} // namespace poolhttpcl
