#include "trblhostclass.h"

#include "shtime.h"

//#ifdef SO_SERV_VERS
//*************************************************************************************************************************************
//                                                          TUdnsObjItem
//*************************************************************************************************************************************

TString TUdnsObjItemTypeToStroka(TUdnsObjItemType type) {
    TString res = "";

    switch (type) {
        case UDNS_RESOLV:
            res = "resolv";
            break;
        case UDNS_GEO:
            res = "geo";
            break;
        case UDNS_RBLHOST:
            res = "rblhost";
            break;
        case UDNS_OTHER:
            res = "other";
            break;
        default:
            res = "unknown";
    };

    return res;
}

TUdnsObjItem::TUdnsObjItem(const TString& nameA, TLogClass* logA, TUdnsPropData& udns_propA, TUdnsObjItemType typeA, const TSootvItemList& sootvlistA) {
    init_ok = false;
    udns_obj = nullptr;
    log = logA;
    name = nameA;
    udns_prop = udns_propA;
    type = typeA;
    sootvlist = sootvlistA;

    if (udns_prop.m_enable)
        init_ok = InitData(name, udns_prop.m_host, udns_prop.m_port, udns_prop.m_timeout, log);
}

bool TUdnsObjItem::InitData(const TString& nameA, const TString& host, ui16 port, ui32 timeout, TLogClass* log) {
    bool res = true;

    if (udns_obj == nullptr)
        udns_obj = MakeHolder<TUdnsClass>(nameA, log, timeout);
    if (udns_obj != nullptr) {
        if (!udns_obj->IsReady())
            res = udns_obj->Open(port, host);
    }

    return res;
}

bool TUdnsObjItem::GetData(const TString& Numbrequest, TKIPv6& ip, TMultiStringList& resvalues) {
    bool res = false;
    TIpAddr addr = TIpAddr(ip.toStroka());
    ui32 tick = 0;
    bool ok = false;
    TString res_str = "";
    TString str_host = "";
    TVector<TString> vectstr;
    bool debug = false;

    resvalues.clear();
    if (udns_prop.m_enable) {
        if ((addr.IsValid()) && (udns_obj != nullptr) && (udns_obj->IsReady())) {
            tick = CShingleTime::GetMs();
            if (type == UDNS_RESOLV) {
                res_str = "";
                ok = udns_obj->ResolveAddr(Numbrequest, addr, res_str);
                if (!res_str.empty())
                    resvalues.push_back(res_str);

            } else if (type == UDNS_GEO) {
                res_str = "";
                ok = udns_obj->ResolveDNSBL(Numbrequest, addr, udns_prop.m_zone.c_str(), res_str);
                if (!res_str.empty())
                    resvalues.push_back(res_str);

            } else if (type == UDNS_RBLHOST) {
                if (debug) {
                    if (udns_prop.m_zone == "combined-bl.yandex.ru")
                        vectstr.push_back("127.0.0.4");
                    else if (udns_prop.m_zone == "ywl.yandex.ru")
                        vectstr.push_back("127.0.0.2");
                    else if (udns_prop.m_zone == "returnpath-certifiedwhite")
                        vectstr.push_back("127.0.0.10");
                    ok = true;

                } else {
                    vectstr.clear();
                    str_host = addr.ToDNSBLString() + "." + udns_prop.m_zone;
                    ok = udns_obj->ResolveHost(Numbrequest, str_host.c_str(), &vectstr);
                }
                auto it = vectstr.begin();
                while (it != vectstr.end()) {
                    if (!(*it).empty())
                        resvalues.push_back(TString((*it)));

                    ++it;
                }
            }
            tick = CShingleTime::GetMs() - tick;
            AddTick(tick);
            if (ok) {
                res = true;
                m_ok.Increment();

            } else {
                res = false;
                m_failed.Increment();
            }

        } else {
            if (!addr.IsValid())
                m_ip_error.Increment();
            else
                m_error.Increment();
        }
    }

    return res;
}

TString TUdnsObjItem::GetInitStatus() {
    TString res = "";
    TSootvItemListIt it;
    TString sootvlist_txt = "";

    it = sootvlist.begin();
    while (it != sootvlist.end()) {
        if (!sootvlist_txt.empty())
            sootvlist_txt = sootvlist_txt + ", '" + (*it).m_key + "'->'" + (*it).m_value + "'";
        else
            sootvlist_txt = sootvlist_txt + "'" + (*it).m_key + "'->'" + (*it).m_value + "'";

        ++it;
    }
    res = res + "UDNSOBJ: type=" + TUdnsObjItemTypeToStroka(type) + ", enable=" + BoolToStroka2(udns_prop.m_enable) + ", init " + BoolToStroka3(init_ok) + ", name='" + name + "', host='" + udns_prop.m_host + ":" + IntToStroka(udns_prop.m_port) + "', zone='" + udns_prop.m_zone + "', timeout=" + IntToStroka(udns_prop.m_timeout) + ", sootv=(" + sootvlist_txt + ")";

    return res;
}

bool TUdnsObjItem::Resolv(const TString& NumbRequest, TKIPv6& ip, TMultiStringList& resvalues) {
    bool res = false;
    TSootvItemListIt it;
    TMultiStringList resvalues_t;
    TMultiStringList::iterator sit;
    TString tstr = "";
    TString tolog = "";
    TString tolog_item = "";
    ui32 tick = 0;

    resvalues.clear();
    tick = CShingleTime::GetMs();
    res = GetData(NumbRequest, ip, resvalues_t);
    tick = CShingleTime::GetMs() - tick;
    if (res) {
        sit = resvalues_t.begin();
        while (sit != resvalues_t.end()) {
            tstr = (*sit);
            if (!tstr.empty()) {
                if (!sootvlist.empty()) {
                    it = sootvlist.begin();
                    while (it != sootvlist.end()) {
                        if ((*it).m_key == tstr) {
                            tolog_item = (*it).m_value + "(" + tstr + ")";
                            resvalues.push_back((*it).m_value);
                            break;
                        }

                        ++it;
                    }

                } else {
                    resvalues.push_back(tstr);
                    tolog_item = tstr + "(" + tstr + ")";
                }
            }

            if (!tolog.empty())
                tolog = tolog + ", " + tolog_item;
            else
                tolog = tolog + tolog_item;

            ++sit;
        }
    }
    if ((!NumbRequest.empty()) && (log != nullptr) && (udns_prop.m_enable)) {
        log->WriteMessageAndDataStatus(KMESSAGE, "% 3u %s %s ip=%s res='%s'", tick, NumbRequest.c_str(), name.c_str(), ip.toStroka().c_str(), tolog.c_str());
    }

    return res;
}

TSummDataUdns TUdnsObjItem::GetSummData(const TString& NumbRequest, TKIPv6& ip) {
    TSummDataUdns res;
    ui32 tick = 0;
    TMultiStringList resvalues;
    TMultiStringList::iterator it;

    res.m_data.m_name = name;
    res.m_prop = udns_prop;
    tick = CShingleTime::GetMs();
    res.m_data.m_ok = Resolv(NumbRequest, ip, resvalues);
    it = resvalues.begin();
    while (it != resvalues.end()) {
        if (!(*it).empty())
            res.m_data.m_res.push_back((*it));

        ++it;
    }
    tick = CShingleTime::GetMs() - tick;
    res.m_data.m_tick = tick;

    return res;
}

//*************************************************************************************************************************************
//                                                          TRBLHostClass
//*************************************************************************************************************************************

TRBLHostClass::TRBLHostClass() {
    Log = nullptr;
    m_rblconf_filename = "";
    m_resolv_it = element_list.end();
    m_geo_it = element_list.end();
}

bool TRBLHostClass::InitByConfig(TKConfig* configobj, TLogClass* LogA) {
    bool res = true;

    if (configobj != nullptr) {
        //resolv
        bool resolv_enable = configobj->ReadBool("resolv", "enable", false);
        TString resolv_host = configobj->ReadStroka("resolv", "host", "");
        int resolv_port = configobj->ReadInteger("resolv", "port", 0);
        int resolv_timeout = configobj->ReadInteger("resolv", "timeout", 0);
        TString resolv_zone = configobj->ReadStroka("resolv", "zone", "");

        //geo
        bool geo_enable = configobj->ReadBool("geoserver", "enable", false);
        TString geo_host = configobj->ReadStroka("geoserver", "host", "");
        int geo_port = configobj->ReadInteger("geoserver", "port", 0);
        int geo_timeout = configobj->ReadInteger("geoserver", "timeout", 0);
        TString geo_zone = configobj->ReadStroka("geoserver", "zone", "");

        //rbl.conf
        bool rblconf_enable = configobj->ReadBool("rblconf", "enable", false);
        TString rblconf_host = configobj->ReadStroka("rblconf", "host", "");
        int rblconf_port = configobj->ReadInteger("rblconf", "port", 0);
        int rblconf_timeout = configobj->ReadInteger("rblconf", "timeout", 0);
        TString rblconf_zone = "";
        TString rblconf_filename = configobj->ReadStroka("rblconf", "filename", "");

        res = Init(LogA,
                   resolv_enable, resolv_host, resolv_port, resolv_timeout, resolv_zone,
                   geo_enable, geo_host, geo_port, geo_timeout, geo_zone,
                   rblconf_enable, rblconf_host, rblconf_port, rblconf_timeout, rblconf_filename);
    }

    return res;
}

bool TRBLHostClass::Init(TLogClass* LogA,
                         bool resolv_enable, const TString& resolv_host, int resolv_port, int resolv_timeout, const TString& resolv_zone,
                         bool geo_enable, const TString& geo_host, int geo_port, int geo_timeout, const TString& geo_zone,
                         bool rblconf_enable, const TString& rblconf_host, int rblconf_port, int rblconf_timeout, const TString& rblconf_filename) {
    bool res = true;

    Log = LogA;

#ifndef _WIN32
    dns_init(0, 0);
#endif

    //resolv
    m_resolv_prop.m_enable = resolv_enable;
    m_resolv_prop.m_host = resolv_host;
    m_resolv_prop.m_port = resolv_port;
    m_resolv_prop.m_timeout = resolv_timeout;
    m_resolv_prop.m_zone = resolv_zone;

    //geo
    m_geo_prop.m_enable = geo_enable;
    m_geo_prop.m_host = geo_host;
    m_geo_prop.m_port = geo_port;
    m_geo_prop.m_timeout = geo_timeout;
    m_geo_prop.m_zone = geo_zone;

    //rbl.conf
    m_rblconf_prop.m_enable = rblconf_enable;
    m_rblconf_prop.m_host = rblconf_host;
    m_rblconf_prop.m_port = rblconf_port;
    m_rblconf_prop.m_timeout = rblconf_timeout;
    m_rblconf_prop.m_zone = "";
    m_rblconf_filename = rblconf_filename;

    element_list.clear();
    element_list.emplace_back(TString("RESOLV"), Log, m_resolv_prop, UDNS_RESOLV, TSootvItemList());
    m_resolv_it = std::next(element_list.begin(), element_list.size() - 1);
    element_list.emplace_back(TString("GEO"), Log, m_geo_prop, UDNS_GEO, TSootvItemList());
    m_geo_it = std::next(element_list.begin(), element_list.size() - 1);

    if (!m_rblconf_filename.empty())
        res = ParseRBLHostFile(m_rblconf_filename, element_list, Log, m_rblconf_prop);

    return res;
}

void TRBLHostClass::GetInitStatus(TMultiStringList& status_list) {
    TUdnsObjItemListIt it;

    status_list.clear();
    it = element_list.begin();
    while (it != element_list.end()) {
        status_list.push_back((*it).GetInitStatus());
        ++it;
    }
}

bool TRBLHostClass::ParseRBLHostFile(const TString& filename, TUdnsObjItemList& elemlist, TLogClass* logA, TUdnsPropData& prop_base) {
    bool res = false;
    FILE* handle = nullptr;
    char tbuff[1024];
    char* p_str = nullptr;
    TString rule_prefix = "";
    TString zone = "";
    const char* pb = nullptr;
    const char* pe = nullptr;
    int count = 0;
    int field_number = 0;
    TString field_str = "";
    TSootvItemList sootvlist;
    TString key_s = "";
    TString value_s = "";
    TString tstr = "";
    ui32 rblhost_index = 0;
    TUdnsPropData prop_item;
    TUdnsObjItemListIt it;
    bool exists = false;

    if (!filename.empty()) {
        handle = fopen(filename.c_str(), "rb");
        if (handle != nullptr) {
            memset(tbuff, 0, sizeof(tbuff));
            p_str = fgets(tbuff, sizeof(tbuff) - 1, handle);
            while (p_str != nullptr) {
                tstr = Trim(tbuff);
                pb = tstr.c_str();
                if ((!tstr.empty()) && (pb != nullptr)) {
                    rblhost_index = IncMax32(rblhost_index, 1);
                    rule_prefix = "";
                    zone = "";
                    sootvlist.clear();

                    field_number = 0;
                    pe = strchr(pb, ' ');
                    while (pe != nullptr) {
                        count = pe - pb;
                        if (count > 0) {
                            field_str = Trim(TString(pb, count));
                            if (!field_str.empty()) {
                                field_number++;
                                if (field_number == 3) {
                                    rule_prefix = field_str;

                                } else if (field_number == 4) {
                                    zone = field_str;

                                } else if (field_number > 5) {
                                    if ((field_number % 2) == 0) //�����
                                    {
                                        key_s = field_str;

                                    } else //������������
                                    {
                                        value_s = field_str;
                                        sootvlist.push_back(TSootvItem(key_s, value_s));
                                        key_s = "";
                                        value_s = "";
                                    }
                                }
                            }
                        }

                        pb = pe + 1;
                        if (pb != nullptr) {
                            pe = strchr(pb, ' ');
                            if ((pe == nullptr) && (pb < (tstr.c_str() + tstr.length())))
                                pe = tstr.c_str() + tstr.length();
                        } else {
                            pe = nullptr;
                        }
                    }

                    if ((!rule_prefix.empty()) && (!zone.empty())) {
                        prop_item = prop_base;
                        prop_item.m_zone = zone;

                        exists = false;
                        it = elemlist.begin();
                        while (it != elemlist.end()) {
                            if ((*it).Name() == rule_prefix) {
                                exists = true;
                                break;
                            }

                            ++it;
                        }
                        if (!exists) {
                            elemlist.emplace_back(rule_prefix, logA, prop_item, UDNS_RBLHOST, sootvlist);
                        }
                    }
                }

                p_str = fgets(tbuff, sizeof(tbuff) - 1, handle);
            }

            fclose(handle);
            handle = nullptr;
        }
    }

    return res;
}

void TRBLHostClass::GetAllStat(const TString& NumbRequest, TKIPv6 ip, TSummDataUdnsList& stat_list) {
    TUdnsObjItemListIt it;

    stat_list.clear();
    it = element_list.begin();
    while (it != element_list.end()) {
        stat_list.push_back((*it).GetSummData(NumbRequest, ip));

        ++it;
    }
}

bool TRBLHostClass::ResolvEnable() {
    bool res = false;

    if (m_resolv_it != element_list.end()) {
        res = (*m_resolv_it).GetProp().m_enable;
    }

    return res;
}

TShortDataUdns TRBLHostClass::GetResolvData(const TString& NumbRequest, TKIPv6 ip) {
    TShortDataUdns res;

    res.Clear();
    if (m_resolv_it != element_list.end()) {
        res.m_name = (*m_resolv_it).Name();
        res.m_tick = CShingleTime::GetMs();
        res.m_ok = (*m_resolv_it).Resolv(NumbRequest, ip, res.m_res);
        res.m_tick = CShingleTime::GetMs() - res.m_tick;
    }

    return res;
}

bool TRBLHostClass::GetResolvData(const TString& NumbRequest, TKIPv6 ip, TString& resvalue) {
    bool res = false;
    TMultiStringList resvalues;
    TMultiStringListIt it;

    resvalue = "";
    if (m_resolv_it != element_list.end()) {
        res = (*m_resolv_it).Resolv(NumbRequest, ip, resvalues);
        it = resvalues.begin();
        if (it != resvalues.end())
            resvalue = (*it);
    }

    return res;
}

bool TRBLHostClass::GeoEnable() {
    bool res = false;

    if (m_geo_it != element_list.end()) {
        res = (*m_geo_it).GetProp().m_enable;
    }

    return res;
}

TShortDataUdns TRBLHostClass::GetGeoData(const TString& NumbRequest, TKIPv6 ip) {
    TShortDataUdns res;

    res.Clear();
    if (m_geo_it != element_list.end()) {
        res.m_name = (*m_geo_it).Name();
        res.m_tick = CShingleTime::GetMs();
        res.m_ok = (*m_geo_it).Resolv(NumbRequest, ip, res.m_res);
        res.m_tick = CShingleTime::GetMs() - res.m_tick;
    }

    return res;
}

bool TRBLHostClass::GetGeoData(const TString& NumbRequest, TKIPv6 ip, TString& resvalue) {
    bool res = false;
    TMultiStringList resvalues;
    TMultiStringListIt it;

    resvalue = "";
    if (m_geo_it != element_list.end()) {
        res = (*m_geo_it).Resolv(NumbRequest, ip, resvalues);
        it = resvalues.begin();
        if (it != resvalues.end())
            resvalue = (*it);
    }

    return res;
}

void TRBLHostClass::GetRBLHosts(const TString& NumbRequest, TKIPv6 ip, TShortDataUdnsList& reslist) {
    TUdnsObjItemListIt it;
    TShortDataUdns sdu;
    TMultiStringList resvalues;
    //    TMultiStringListIt mit;

    reslist.clear();
    it = element_list.begin();
    while (it != element_list.end()) {
        if ((*it).Type() == UDNS_RBLHOST) {
            sdu.m_name = (*it).Name();
            sdu.m_tick = CShingleTime::GetMs();
            sdu.m_ok = (*it).Resolv(NumbRequest, ip, sdu.m_res);
            sdu.m_tick = CShingleTime::GetMs() - sdu.m_tick;

            reslist.push_back(sdu);
        }

        ++it;
    }
}

void TRBLHostClass::GetWorkRules(const TString& NumbRequest, TKIPv6 ip, TRBLWorkRuleList& rulelist) {
    TShortDataUdnsList udnsreslist;
    TShortDataUdnsListIt it;
    TString rulename = "";
    TMultiStringListIt mit;
    THashMap<TString, ui32> rule_hash;
    THashMap<TString, ui32>::iterator rhit;

    rulelist.clear();
    udnsreslist.clear();
    GetRBLHosts(NumbRequest, ip, udnsreslist);
    it = udnsreslist.begin();
    while (it != udnsreslist.end()) {
        if ((*it).m_ok) {
            mit = (*it).m_res.begin();
            while (mit != (*it).m_res.end()) {
                rulename = (*it).m_name + "_" + (*mit);
                if (!rulename.empty()) {
                    rhit = rule_hash.find(rulename);
                    if (rhit != rule_hash.end())
                        (*rhit).second = IncMax32((*rhit).second, 1);
                    else
                        rule_hash[rulename] = 1;
                }

                ++mit;
            }
        }

        ++it;
    }

    rhit = rule_hash.begin();
    while (rhit != rule_hash.end()) {
        rulelist.push_back((*rhit).first);

        ++rhit;
    }
}

TString TRBLHostClass::GetMonData() {
    TString res = "";
    TUdnsObjItemListIt it;

    it = element_list.begin();
    while (it != element_list.end()) {
        res = res + (*it).GetMonData();

        ++it;
    }

    return res;
}

//#endif
//*************************************************************************************************************************************
