#include "tresolvcache.h"
#include "color_scheme.h"
#include "shtime.h"

//***************************************************************************************************************************
//                                                 TResolvStorageItem
//***************************************************************************************************************************

TResolvStorageItem::TResolvStorageItem() {
    m_index = -1;
    m_Log = nullptr;
    m_dump_filename = "";
    m_last_update_sec = 3600;
}

void TResolvStorageItem::Init(int index, TLogClass* Log, const TString& dump_filename, ui32 last_update_sec) {
    m_index = index;
    m_Log = Log;
    m_dump_filename = dump_filename;
    m_last_update_sec = last_update_sec;
}

void TResolvStorageItem::Cleanup() {
    TResolvRecord new_datastor;
    ui32 remain_count = 0;
    ui32 all_count = 0;

    ui32 tick = CShingleTime::GetMs();

    m_Mutex.Acquire();

    TResolvRecordIt it;
    ui32 current_time = time(nullptr);
    ui32 diff_t = 0;

    all_count = m_datastor.size();

    it = m_datastor.begin();
    while (it != m_datastor.end()) {
        if (current_time >= (*it).second.m_last_update) {
            diff_t = current_time - (*it).second.m_last_update;
            if (diff_t <= m_last_update_sec) {
                (new_datastor)[(*it).first] = (*it).second;
                remain_count = IncMax32(remain_count, 1);
            }
        }

        ++it;
    }

    m_datastor = std::move(new_datastor);

    m_Mutex.Release();

    tick = CShingleTime::GetMs() - tick;

    if (m_Log != nullptr)
        m_Log->WriteMessageAndDataStatus(KMESSAGE, "   RSLVSTOR%02u: cleanup %u from %u record to %u msec.", m_index, all_count - remain_count, all_count, tick);
}

void TResolvStorageItem::Midnight() {
    m_Mutex.Acquire();

    m_stat.Midnight();

    m_Mutex.Release();
}

TResolvStorageStat TResolvStorageItem::GetStat() {
    TResolvStorageStat res = TResolvStorageStat();

    m_Mutex.Acquire();

    res = m_stat;
    res.m_today.m_count = m_datastor.size();

    m_Mutex.Release();

    return res;
}

bool TResolvStorageItem::AddHostFromDump(TKIPv6 ip, ui32 lasttime, const TString& host) {
    bool res = false;
    ui64 ip_shingle = 0;

    ip_shingle = ip.AsShingle64();

    m_Mutex.Acquire();

    TResolvRecordIt it = m_datastor.find(ip_shingle);
    if (it != m_datastor.end()) {
        res = true;

    } else {
        m_datastor[ip_shingle] = TResolvRecordStruct(ip, lasttime, host);
    }

    m_Mutex.Release();

    return res;
}

void TResolvStorageItem::AddHost(TKIPv6 ip, const TString& host) {
    ui64 ip_shingle = 0;

    ip_shingle = ip.AsShingle64();

    m_Mutex.Acquire();

        ui32 currtime = time(nullptr);

        m_stat.m_today.m_add_all = IncMax32(m_stat.m_today.m_add_all, 1);
        if (host.empty())
            m_stat.m_today.m_add_empty = IncMax32(m_stat.m_today.m_add_empty, 1);

        TResolvRecordIt it = m_datastor.find(ip_shingle);
        if (it != m_datastor.end()) {
            (*it).second = TResolvRecordStruct(ip, currtime, host);
            m_stat.m_today.m_add_update = IncMax32(m_stat.m_today.m_add_update, 1);

        } else {
            m_datastor[ip_shingle] = TResolvRecordStruct(ip, currtime, host);
        }

    m_Mutex.Release();
}

TResolvRecordStructExt TResolvStorageItem::GetHost(TKIPv6 ip) {
    TResolvRecordStructExt res;
    ui64 ip_shingle = 0;

    ip_shingle = ip.AsShingle64();

    m_Mutex.Acquire();

        m_stat.m_today.m_get_all = IncMax32(m_stat.m_today.m_get_all, 1);

        TResolvRecordIt it = m_datastor.find(ip_shingle);
        if (it != m_datastor.end()) {
            res.m_notfound = false;
            res.m_data = (*it).second;
        }

        if (res.m_notfound) {
            m_stat.m_today.m_get_notfound = IncMax32(m_stat.m_today.m_get_notfound, 1);

        } else {
            if (res.m_data.m_host.empty())
                m_stat.m_today.m_get_empty = IncMax32(m_stat.m_today.m_get_empty, 1);
        }

    m_Mutex.Release();

    return res;
}

//***************************************************************************************************************************
//                                                 TResolvStorage
//***************************************************************************************************************************

void TResolvStorage::Init(TLogClass* Log, const TString& dump_filename, ui32 last_update_sec) {
    m_Log = Log;
    for (size_t i = 0; i < PART_COUNT; i++)
        m_data[i].Init(i, m_Log, dump_filename, last_update_sec);
}

ui32 TResolvStorage::PartByIP(TKIPv6 ip) {
    return ip.AsShingle64() % PART_COUNT;
}

void TResolvStorage::Cleanup() {
    if (m_Log != nullptr) {
        m_Log->WriteMessageAndDataStatus(KMESSAGE, "RSLVSTOR: Start cleanup...");
        m_Log->FFlush();
    }

    ui32 tick = CShingleTime::GetMs();

    for (size_t i = 0; i < PART_COUNT; i++)
        m_data[i].Cleanup();

    tick = CShingleTime::GetMs() - tick;

    if (m_Log != nullptr) {
        m_Log->WriteMessageAndDataStatus(KMESSAGE, "RSLVSTOR: Cleanup compleate to %u msec.", tick);
        m_Log->FFlush();
    }
}

void TResolvStorage::Midnight() {
    for (size_t i = 0; i < PART_COUNT; i++)
        m_data[i].Midnight();
}

TResolvStorageStat TResolvStorage::GetStat() {
    TResolvStorageStat res = TResolvStorageStat();
    TResolvStorageStat res_tmp;

    for (size_t i = 0; i < PART_COUNT; i++) {
        res_tmp = m_data[i].GetStat();

        res.m_today.m_count = IncMax32(res.m_today.m_count, res_tmp.m_today.m_count);
        res.m_today.m_add_all = IncMax32(res.m_today.m_add_all, res_tmp.m_today.m_add_all);
        res.m_today.m_add_update = IncMax32(res.m_today.m_add_update, res_tmp.m_today.m_add_update);
        res.m_today.m_add_empty = IncMax32(res.m_today.m_add_empty, res_tmp.m_today.m_add_empty);
        res.m_today.m_get_all = IncMax32(res.m_today.m_get_all, res_tmp.m_today.m_get_all);
        res.m_today.m_get_notfound = IncMax32(res.m_today.m_get_notfound, res_tmp.m_today.m_get_notfound);
        res.m_today.m_get_empty = IncMax32(res.m_today.m_get_empty, res_tmp.m_today.m_get_empty);

        res.m_yesterday.m_count = IncMax32(res.m_yesterday.m_count, res_tmp.m_yesterday.m_count);
        res.m_yesterday.m_add_all = IncMax32(res.m_yesterday.m_add_all, res_tmp.m_yesterday.m_add_all);
        res.m_yesterday.m_add_update = IncMax32(res.m_yesterday.m_add_update, res_tmp.m_yesterday.m_add_update);
        res.m_yesterday.m_add_empty = IncMax32(res.m_yesterday.m_add_empty, res_tmp.m_yesterday.m_add_empty);
        res.m_yesterday.m_get_all = IncMax32(res.m_yesterday.m_get_all, res_tmp.m_yesterday.m_get_all);
        res.m_yesterday.m_get_notfound = IncMax32(res.m_yesterday.m_get_notfound, res_tmp.m_yesterday.m_get_notfound);
        res.m_yesterday.m_get_empty = IncMax32(res.m_yesterday.m_get_empty, res_tmp.m_yesterday.m_get_empty);
    }

    return res;
}

bool TResolvStorage::AddHostFromDump(TKIPv6 ip, ui32 lasttime, const TString& host) //true - exists, no update
{
    bool res = false;
    size_t index = PartByIP(ip);

    if (index < PART_COUNT)
        res = m_data[index].AddHostFromDump(ip, lasttime, host);

    return res;
}

void TResolvStorage::AddHost(TKIPv6 ip, const TString& host) {
    size_t index = PartByIP(ip);

    if (index < PART_COUNT)
        m_data[index].AddHost(ip, host);
}

TResolvRecordStructExt TResolvStorage::GetHost(TKIPv6 ip) {
    TResolvRecordStructExt res = TResolvRecordStructExt();
    size_t index = PartByIP(ip);

    if (index < PART_COUNT)
        res = m_data[index].GetHost(ip);

    return res;
}

//***************************************************************************************************************************
//                                                 TResolvQueue
//***************************************************************************************************************************

TResolvQueue::TResolvQueue() {
    m_max_queue_size = DEFAULT_LAST_UPDATE_SEC;
}

TResolvQueue::~TResolvQueue() = default;

void TResolvQueue::Init(ui32 max_queue_size) {
    if (max_queue_size > 0)
        m_max_queue_size = max_queue_size;
}

void TResolvQueue::Midnight() {
    m_Mutex.Acquire();

    m_stat.Midnight();

    m_Mutex.Release();
}

TResolvQueueStat TResolvQueue::GetStat() {
    TResolvQueueStat res = TResolvQueueStat();

    m_Mutex.Acquire();

    res = m_stat;
    res.m_today.m_count = m_data.size();

    m_Mutex.Release();

    return res;
}

void TResolvQueue::AddIPToQueue(TKIPv6 ip) {
    ui64 ip_shingle = 0;

    ip_shingle = ip.AsShingle64();

    m_Mutex.Acquire();

    m_stat.m_today.m_input = IncMax32(m_stat.m_today.m_input, 1);

    if (m_data.size() < m_max_queue_size) {
        TResolvQueueDataIt it = m_data.find(ip_shingle);
        if (it != m_data.end()) {
            (*it).second = ip;
            m_stat.m_today.m_update = IncMax32(m_stat.m_today.m_update, 1);

        } else {
            m_data[ip_shingle] = ip;
        }

    } else {
        m_stat.m_today.m_lost = IncMax32(m_stat.m_today.m_lost, 1);
    }

    m_Mutex.Release();
}

TKIPv6 TResolvQueue::GetIPFromQueue() {
    TKIPv6 res = TKIPv6();

    m_Mutex.Acquire();

    TResolvQueueDataIt it = m_data.begin();
    if (it != m_data.end()) {
        res = (*it).second;
        m_data.erase(it);

        m_stat.m_today.m_output = IncMax32(m_stat.m_today.m_output, 1);
    }

    m_Mutex.Release();

    return res;
}

//***************************************************************************************************************************
//                                                 TResolvCache
//***************************************************************************************************************************

void RSLVCCHThreadProc(void* par) {
    auto* ss = (TResolvCache*)par;

    if (ss != nullptr) {
        int threadnumber = ss->IncrementNumberThread();

        //printf("sth=%d\n", threadnumber);
        while (!ss->QueueThreadShouldStop()) {
            ss->ActionFunction(threadnumber);
        }

        ss->DecrementNumberThread();
    }
}

TResolvCache::TResolvCache() {
    m_rblhostobj = nullptr;
    for (auto & i : m_QueueThread)
        i = nullptr;
    use_threadcount = 0;
    run_scan_thread = false;
    m_StopQueueThread = false;
    m_thread_count = 1;
    m_queue_size = DEFAULT_QUEUE_SIZE;
    m_last_update_sec = TResolvQueue::DEFAULT_LAST_UPDATE_SEC;
}

TResolvCache::~TResolvCache() {
    if (run_scan_thread) {
        StopQueueThread();
        while (!QueueThreadStopped())
            usleep(5000);
    }
}

bool TResolvCache::InitBase(TLogClass* Log, ui32 thread_count, ui32 queue_size, ui32 last_update_sec, const TString& dump_filename) {
    bool res = true;

    m_resolv_stat.Init();

    m_thread_count = thread_count;
    if (m_thread_count < 1)
        m_thread_count = 1;
    if (m_thread_count >= MAX_THREAD_WORK)
        m_thread_count = MAX_THREAD_WORK;

    if (queue_size > 0)
        m_queue_size = queue_size;
    m_queue.Init(m_queue_size);

    m_storage.Init(Log, dump_filename, last_update_sec);

    if (last_update_sec > 0)
        m_last_update_sec = last_update_sec;

    StartQueueThread();

    return res;
}

bool TResolvCache::Init(TRBLHostClass* rblhostobjA, TKConfig* configobjA, TLogClass* Log) {
    bool res = false;
    ui32 thread_count = DEFAULT_THREAD_WORK;
    ui32 queue_size = DEFAULT_QUEUE_SIZE;
    ui32 last_update_sec = TResolvQueue::DEFAULT_LAST_UPDATE_SEC;
    TString dump_filename = "";

    m_rblhostobj = rblhostobjA;

    if (configobjA != nullptr) {
        thread_count = configobjA->ReadInteger("resolv", "async_thread_count", DEFAULT_THREAD_WORK);
        queue_size = configobjA->ReadInteger("resolv", "async_queue_max_size", DEFAULT_QUEUE_SIZE);
        last_update_sec = configobjA->ReadInteger("resolv", "async_last_update_sec", TResolvQueue::DEFAULT_LAST_UPDATE_SEC);
        dump_filename = configobjA->ReadStroka("resolv", "async_dump_filename", "");
    }

    if (queue_size == 0)
        queue_size = 10000;
    if (queue_size > 5000000)
        queue_size = 5000000;

    res = InitBase(Log, thread_count, queue_size, last_update_sec, dump_filename);

    return res;
}

void TResolvCache::Midnight() {
    m_storage.Midnight();
    m_storage.Cleanup();
    m_queue.Midnight();
    m_resolv_stat.Midnight();
}

void TResolvCache::StartQueueThread() {
    m_QueueMutex.Acquire();

    m_StopQueueThread = false;

    for (size_t i = 0; i < m_thread_count; i++) {
        m_QueueThread[i] = MakeHolder<TThread>((TThread::TThreadProc)&RSLVCCHThreadProc, this);
        m_QueueThread[i]->Start();
    }
    run_scan_thread = true;

    m_QueueMutex.Release();
}

int TResolvCache::IncrementNumberThread() {
    int res = 0;

    m_QueueMutex.Acquire();

    use_threadcount++;
    res = use_threadcount;

    m_QueueMutex.Release();

    return res;
}

void TResolvCache::DecrementNumberThread() {
    m_QueueMutex.Acquire();

    if (use_threadcount > 0)
        use_threadcount--;

    m_QueueMutex.Release();
}

void TResolvCache::StopQueueThread() {
    m_QueueMutex.Acquire();
    m_StopQueueThread = true;
    m_QueueMutex.Release();
}

bool TResolvCache::QueueThreadStopped() {
    bool res = false;

    if (use_threadcount == 0)
        res = true;

    return res;
}

void TResolvCache::ActionFunction(int threadnumber) {
    bool needstop = false;

    RequestToResolvServer(threadnumber);

    if (needstop)
        StopQueueThread();
}

void TResolvCache::RequestToResolvServer(int threadnumber) {
    if ((m_rblhostobj != nullptr) && (m_rblhostobj->ResolvEnable())) {
        TKIPv6 ip = TKIPv6();
        ui32 ttt = 0;

        ttt = CShingleTime::GetMs();
        ip = m_queue.GetIPFromQueue();
        ttt = CShingleTime::GetMs() - ttt;
        if (!ip.Undefined()) {
            m_resolv_stat.AddTick("QUEUE_GET", ttt);

            TString host = "";

            ttt = CShingleTime::GetMs();
            if (!m_rblhostobj->GetResolvData("ASYNC_RESOLV_TH" + IntToStroka2(threadnumber), ip, host))
                host = "";
            ttt = CShingleTime::GetMs() - ttt;

            m_resolv_stat.AddTick("RSLV", ttt);
            if (!host.empty())
                m_resolv_stat.AddTick("RSLV_OK <b>[CNT]</b>", 0);
            else
                m_resolv_stat.AddTick("RSLV_FAILED <b>[CNT]</b>", 0);

            if (!host.empty()) {
                ttt = CShingleTime::GetMs();
                m_storage.AddHost(ip, host);
                ttt = CShingleTime::GetMs() - ttt;
                m_resolv_stat.AddTick("STORAGE_ADD", ttt);
            }

        } else {
            usleep(30000);
        }
    }
}

TResolvRecordStructExt TResolvCache::GetHost(const TString& /*NumbRequest*/, TKIPv6 ip) {
    TResolvRecordStructExt res = TResolvRecordStructExt();
    bool need_update = false;
    ui32 curr_time = time(nullptr);
    ui32 tick = 0;
    ui32 ttt = 0;

    tick = CShingleTime::GetMs();
    if (!ip.Undefined()) {
        ttt = CShingleTime::GetMs();
        res = m_storage.GetHost(ip);
        ttt = CShingleTime::GetMs() - ttt;
        m_resolv_stat.AddTick("STORAGE_FIND", ttt);
        if (res.m_notfound) {
            need_update = true;

        } else {
            ui32 diff_time = 0;

            if (curr_time > res.m_data.m_last_update)
                diff_time = curr_time - res.m_data.m_last_update;
            else
                need_update = true;

            if (diff_time >= m_last_update_sec)
                need_update = true;
        }

        if (need_update) {
            ttt = CShingleTime::GetMs();
            m_queue.AddIPToQueue(ip);
            ttt = CShingleTime::GetMs() - ttt;
            m_resolv_stat.AddTick("QUEUE_ADD", ttt);
        }
    }
    tick = CShingleTime::GetMs() - tick;
    m_resolv_stat.AddTick("*GET_FROM_CACHE", tick);

    return res;
}

TResolvRecordStructExt TResolvCache::GetHostOnlyFromStor(TKIPv6 ip) {
    TResolvRecordStructExt res = TResolvRecordStructExt();
    //ui32                   curr_time   = time(NULL);
    ui32 tick = 0;
    ui32 ttt = 0;

    tick = CShingleTime::GetMs();
    if (!ip.Undefined()) {
        ttt = CShingleTime::GetMs();
        res = m_storage.GetHost(ip);
        ttt = CShingleTime::GetMs() - ttt;
        m_resolv_stat.AddTick("STORAGE_FIND", ttt);
    }
    tick = CShingleTime::GetMs() - tick;
    m_resolv_stat.AddTick("*GET_FROM_CACHE", tick);

    return res;
}

TString TResolvCache::GetWebStat(const TString& ipfromcache_form) {
    TString res = "";
    TResolvQueueStat queue_stat;
    TResolvStorageStat stor_stat;

    res += "<i><b>Properties:</b></i><br>";
    res += "Thread count:    " + IntToStroka(m_thread_count) + ";<br>";
    res += "Queue size:      " + IntToStroka(m_queue_size) + ";<br>";
    res += "Last update sec: " + IntToStroka(m_last_update_sec) + ";<br>";
    res += "<br>";

    res += "<br>" + ipfromcache_form + "<br>";

    queue_stat = m_queue.GetStat();
    stor_stat = m_storage.GetStat();
    res += "<i><b>Queue & storage statistik:</b></i><br>";
    res += "<table border='1' width='100%' cellspacing='0' cellpadding='4' bgcolor=" + TString(color_table_default) + ">";
    res += "<tr align='center' bgcolor=" + TString(color_table_shap) + "><td width='20%'>&nbsp;</td><td width='30%' colspan='2'><b>QUEUE STAT</b></td><td width='20%'>&nbsp;</td><td width='30%' colspan='2'><b>STORAGE STAT</b></td></tr>";
    res += "<tr align='center' bgcolor=" + TString(color_table_shap) + "><td width='20%'>&nbsp;</td><td width='15%'><b>TODAY</b></td><td width='15%'><b>YESTERDAY</b></td><td width='20%'>&nbsp;</td><td width='15%'><b>TODAY</b></td><td width='15%'><b>YESTERDAY</b></td></tr>";
    res += "<tr align='right'><td align='left'>COUNT:</td><td>" + IntToStroka(queue_stat.m_today.m_count) + "</td><td>" + "-" + "</td><td align='left'>COUNT:</td><td>" + IntToStroka(stor_stat.m_today.m_count) + "</td><td>" + "-" + "</td></tr>";
    res += "<tr align='right'><td align='left'>INPUT:</td><td>" + IntToStroka(queue_stat.m_today.m_input) + "</td><td>" + IntToStroka(queue_stat.m_yesterday.m_input) + "</td><td align='left'>Add all:</td><td>" + IntToStroka(stor_stat.m_today.m_add_all) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_add_all) + "</td></tr>";
    res += "<tr align='right'><td align='left'>OUTPUT:</td><td>" + IntToStroka(queue_stat.m_today.m_output) + "</td><td>" + IntToStroka(queue_stat.m_yesterday.m_output) + "</td><td align='left'>Add update:</td><td>" + IntToStroka(stor_stat.m_today.m_add_update) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_add_update) + "</td></tr>";
    res += "<tr align='right'><td align='left'>LOST:</td><td>" + IntToStroka(queue_stat.m_today.m_lost) + "</td><td>" + IntToStroka(queue_stat.m_yesterday.m_lost) + "</td><td align='left'>Add empty:</td><td>" + IntToStroka(stor_stat.m_today.m_add_empty) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_add_empty) + "</td></tr>";
    res += "<tr align='right'><td align='left'>UPDATE:</td><td>" + IntToStroka(queue_stat.m_today.m_update) + "</td><td>" + IntToStroka(queue_stat.m_yesterday.m_update) + "</td><td align='left'>Get all:</td><td>" + IntToStroka(stor_stat.m_today.m_get_all) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_get_all) + "</td></tr>";
    res += "<tr align='right'><td align='left'>-</td><td>-</td><td>-</td><td align='left'>Get notfound:</td><td>" + IntToStroka(stor_stat.m_today.m_get_notfound) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_get_notfound) + "</td></tr>";
    res += "<tr align='right'><td align='left'><b>FIND&RETURN FROM CACHE (GET HOST BY IP - OK):</b></td><td>" + IntToStroka(stor_stat.m_today.FindAndReturnFromCache()) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.FindAndReturnFromCache()) + "</td><td align='left'>Get empty:</td><td>" + IntToStroka(stor_stat.m_today.m_get_empty) + "</td><td>" + IntToStroka(stor_stat.m_yesterday.m_get_empty) + "</td></tr>";
    res += "</table><br>";

    res += "<i><b>Resolv time statistik:</b></i><br>";
    res += m_resolv_stat.GetWebData();

    return res;
}

TString TResolvCache::GetRslvIPFromCache(TKIPv6 ip) {
    TString res = "";

    TResolvRecordStructExt ipdata = GetHostOnlyFromStor(ip);

    if (ipdata.m_notfound) {
        res += "IP " + ip.toStroka() + " in resolv cache notfound!";

    } else {
        res += "<table border='1' width='100%' cellspacing='0' cellpadding='4' bgcolor=" + TString(color_table_default) + ">";
        res += "<tr><td width='20%' align='left'>IP:</td><td width='20%' align='right'>" + ipdata.m_data.m_ip.toStroka() + "</td>";
        res += "<tr><td width='20%' align='left'>LASTUPDATE:</td><td width='20%' align='right'>" + IntToStroka(ipdata.m_data.m_last_update) + " (" + GetTimePeriod(time(nullptr) - ipdata.m_data.m_last_update) + ")</td>";
        res += "<tr><td width='20%' align='left'>HOST:</td><td width='20%' align='right'>'" + ipdata.m_data.m_host + "'</td>";
        res += "</table>";
    }

    return res;
}

//**************************************************************************************************************************
