#pragma once

#include <util/generic/string.h>
#include "util/system/mutex.h"
#include "util/system/thread.h"
#include "kfunc.h"
#include "tlogclass.h"
#include "tkconfig.h"
#include "trblhostclass.h"
#include "tkipv6.h"
#include <vector>
#include "ttraccert_stat.h"

//***************************************************************************************************************************
//                                                 TResolvStorageItem
//***************************************************************************************************************************

struct TResolvRecordStruct {
    ui32 m_last_update{};
    TKIPv6 m_ip;
    TString m_host;

    Y_SAVELOAD_DEFINE(m_last_update, m_ip, m_host)

    TResolvRecordStruct() {
        Clear();
    }

    TResolvRecordStruct(TKIPv6 ip, ui32 timeA, const TString& hostA) {
        m_last_update = timeA;
        m_ip = ip;
        m_host = hostA;
    }

    void Clear() {
        m_last_update = 0;
        m_ip = TKIPv6();
        m_host = "";
    }
};

typedef THashMap<ui64, TResolvRecordStruct> TResolvRecord;
typedef TResolvRecord::iterator TResolvRecordIt;

struct TResolvRecordStructExt {
    bool m_notfound{};
    TResolvRecordStruct m_data;

    TResolvRecordStructExt() {
        Clear();
    }

    void Clear() {
        m_notfound = true;
        m_data.Clear();
    }
};

struct TResolvStorageStatItem {
    ui32 m_count{};
    ui32 m_add_all{};
    ui32 m_add_update{};
    ui32 m_add_empty{};
    ui32 m_get_all{};
    ui32 m_get_notfound{};
    ui32 m_get_empty{};

    TResolvStorageStatItem() {
        Clear();
    }

    void Clear() {
        m_count = 0;
        m_add_all = 0;
        m_add_update = 0;
        m_add_empty = 0;
        m_get_all = 0;
        m_get_notfound = 0;
        m_get_empty = 0;
    }

    ui32 FindAndReturnFromCache() {
        return m_get_all - m_get_notfound - m_get_empty;
    }
};

struct TResolvStorageStat {
    TResolvStorageStatItem m_today;
    TResolvStorageStatItem m_yesterday;

    TResolvStorageStat() {
        Clear();
    }

    void Clear() {
        m_today.Clear();
        m_yesterday.Clear();
    }

    void Midnight() {
        m_yesterday = m_today;
        m_today.Clear();
    }
};

class TResolvStorageItem {
private:
    int m_index;
    TLogClass* m_Log;
    TString m_dump_filename;
    TResolvRecord m_datastor;
    TMutex m_Mutex;
    TResolvStorageStat m_stat;
    ui32 m_last_update_sec;

public:
    Y_SAVELOAD_DEFINE(m_datastor)
    TResolvStorageItem();

    void Init(int index, TLogClass* Log, const TString& dump_filename, ui32 last_update_sec);
    void Cleanup();
    void Midnight();
    TResolvStorageStat GetStat();

    bool AddHostFromDump(TKIPv6 ip, ui32 lasttime, const TString& host); //true - exists, no update
    void AddHost(TKIPv6 ip, const TString& host);
    TResolvRecordStructExt GetHost(TKIPv6 ip);
};

//***************************************************************************************************************************
//                                                 TResolvStorage
//***************************************************************************************************************************

class TResolvStorage {
    static const ui32 PART_COUNT = 16;

private:
    TVector<TResolvStorageItem> m_data = TVector<TResolvStorageItem>(PART_COUNT);
    TLogClass* m_Log{};

    static ui32 PartByIP(TKIPv6 ip);

public:
    Y_SAVELOAD_DEFINE(m_data)

    void Init(TLogClass* Log, const TString& dump_filename, ui32 last_update_sec);
    void Cleanup();
    void Midnight();
    TResolvStorageStat GetStat();

    bool AddHostFromDump(TKIPv6 ip, ui32 lasttime, const TString& host); //true - exists, no update
    void AddHost(TKIPv6 ip, const TString& host);
    TResolvRecordStructExt GetHost(TKIPv6 ip);
};

//***************************************************************************************************************************
//                                                 TResolvQueue
//***************************************************************************************************************************

struct TResolvQueueStatItem {
    ui32 m_input{};
    ui32 m_output{};
    ui32 m_count{};
    ui32 m_lost{};
    ui32 m_update{};

    TResolvQueueStatItem() {
        Clear();
    }

    void Clear() {
        m_input = 0;
        m_output = 0;
        m_count = 0;
        m_lost = 0;
        m_update = 0;
    }
};

struct TResolvQueueStat {
    TResolvQueueStatItem m_today;
    TResolvQueueStatItem m_yesterday;

    TResolvQueueStat() {
        Clear();
    }

    void Clear() {
        m_today.Clear();
        m_yesterday.Clear();
    }

    void Midnight() {
        m_yesterday = m_today;
        m_today.Clear();
    }
};

typedef THashMap<ui64, TKIPv6> TResolvQueueData;
typedef TResolvQueueData::iterator TResolvQueueDataIt;

class TResolvQueue {
public:
    static const ui32 DEFAULT_LAST_UPDATE_SEC = 7200;

private:
    TResolvQueueData m_data;
    TMutex m_Mutex;
    TResolvQueueStat m_stat;
    ui32 m_max_queue_size;

public:
    TResolvQueue();
    ~TResolvQueue();

    void Init(ui32 max_queue_size);
    void Midnight();
    TResolvQueueStat GetStat();

    void AddIPToQueue(TKIPv6 ip);
    TKIPv6 GetIPFromQueue();
};

//***************************************************************************************************************************
//                                                 TResolvCache
//***************************************************************************************************************************

class TResolvCache {
public:
    static const ui32 MAX_THREAD_WORK = 64;
    static const ui32 DEFAULT_THREAD_WORK = 32;
    static const ui32 DEFAULT_QUEUE_SIZE = 100000;

private:
    TRBLHostClass* m_rblhostobj;
    TResolvStorage m_storage;
    TResolvQueue m_queue;

    ui32 m_queue_size;
    ui32 m_last_update_sec;

    trcstat::TTraccertStat m_resolv_stat;

private:
    THolder<TThread> m_QueueThread[MAX_THREAD_WORK]{};
    TMutex m_QueueMutex;
    bool m_StopQueueThread;
    bool run_scan_thread;
    ui32 m_thread_count;

    int use_threadcount;
    void StartQueueThread();
    bool QueueThreadStopped();

public:
    Y_SAVELOAD_DEFINE(m_storage)
    ui32 GetThreadCount() {
        return m_thread_count;
    }
    void StopQueueThread();
    int IncrementNumberThread();
    void DecrementNumberThread();
    bool QueueThreadShouldStop() {
        return m_StopQueueThread;
    }
    void ActionFunction(int threadnumber);
    bool InitBase(TLogClass* Log, ui32 thread_count, ui32 queue_size, ui32 last_update_sec, const TString& dump_filename);

public:
    TResolvCache();
    ~TResolvCache();

    bool Init(TRBLHostClass* rblhostobjA, TKConfig* configobjA, TLogClass* Log);
    void Midnight();

    void RequestToResolvServer(int threadnumber);
    TResolvRecordStructExt GetHost(const TString& NumbRequest, TKIPv6 ip);
    TResolvRecordStructExt GetHostOnlyFromStor(TKIPv6 ip);

    TString GetWebStat(const TString& ipfromcache_form);
    TString GetRslvIPFromCache(TKIPv6 ip);
};

//***************************************************************************************************************************
