#include "tstoragerequeststat.h"

namespace NStorageStats {
    TSRSErrType ToTSRSErrTypeB(bool good) {
        TSRSErrType res = TSRSErrType();

        if (good)
            res = SRS_OK;
        else
            res = SRS_OTHER_ERROR;

        return res;
    }

    TStorageRequestStat::TChangeLabelStruct::TChangeLabelStruct()
        : m_acttype(SAT_UPDATE)
    {
    }

    TStorageRequestStat::TChangeLabelStruct::TChangeLabelStruct(TStorageActionType acttype, const TString& weblabel, const TString& loglabel) {
        m_acttype = acttype;
        m_weblabel = weblabel;
        m_loglabel = loglabel;
    }

    void TStorageRequestStat::TChangeLabelStruct::Clear() {
        m_acttype = SAT_UPDATE;
        m_weblabel.clear();
        m_loglabel.clear();
    }

    TStorageRequestStat::TWebLabels::TWebLabels() {
        Init();
    }

    void TStorageRequestStat::TWebLabels::Init() {
        m_web_labels_array[SAT_UPDATE]          = "update_count (UP)";
        m_web_labels_array[SAT_FINDONE]         = "findone_count (FO)";
        m_web_labels_array[SAT_ERASE]           = "erase_count (E)";
        m_web_labels_array[SAT_SIZE]            = "size_count (SZ)";
        m_web_labels_array[SAT_MULTYACTION]     = "multiaction_count (M)";
        m_web_labels_array[SAT_FIND]            = "find_count (F)";
        m_web_labels_array[SAT_CLEAR]           = "clear_count (CLR)";
        m_web_labels_array[SAT_FINDANDMODIFY]   = "findandmodify_count (FM)";
        m_web_labels_array[SAT_DROPTABLE]       = "droptable_count (DT)";
        m_web_labels_array[SAT_CREATETABLE]     = "createtable_count (CT)";
    }

    void TStorageRequestStat::TWebLabels::ChangeLabel(TChangeLabelStructList& change_labels_list) {
        TChangeLabelStructList::iterator it;

        it = change_labels_list.begin();
        while (it != change_labels_list.end()) {
            m_web_labels_array[(*it).m_acttype] = (*it).m_weblabel;

            ++it;
        }
    }

    TString TStorageRequestStat::TWebLabels::GetParamNameWeb(TStorageActionType acttype) {
        return m_web_labels_array[acttype];
    }

    TStorageRequestStat::TLogLabels::TLogLabels() {
        Init();
    }

    void TStorageRequestStat::TLogLabels::Init() {
        m_log_labels_array[SAT_UPDATE]          = "UP";
        m_log_labels_array[SAT_FINDONE]         = "FO";
        m_log_labels_array[SAT_ERASE]           = "E";
        m_log_labels_array[SAT_SIZE]            = "SZ";
        m_log_labels_array[SAT_MULTYACTION]     = "M";
        m_log_labels_array[SAT_FIND]            = "F";
        m_log_labels_array[SAT_CLEAR]           = "CLR";
        m_log_labels_array[SAT_FINDANDMODIFY]   = "FM";
        m_log_labels_array[SAT_DROPTABLE]       = "DT";
        m_log_labels_array[SAT_CREATETABLE]     = "CT";
    }

    void TStorageRequestStat::TLogLabels::ChangeLabel(TChangeLabelStructList& change_labels_list) {
        TChangeLabelStructList::iterator it;

        it = change_labels_list.begin();
        while (it != change_labels_list.end()) {
            m_log_labels_array[(*it).m_acttype] = (*it).m_loglabel;

            ++it;
        }
    }

    TString TStorageRequestStat::TLogLabels::GetParamNameLog(TStorageActionType acttype) {
        return m_log_labels_array[acttype];
    }

    bool TStorageRequestStat::TOperCountStatItemPart::Empty() {
        for (size_t i = 0; i < SRS_END; i++) {
            if (m_errors[i] != 0)
                return false;
        }

        return true;
    }

    ui64 TStorageRequestStat::TOperCountStatItemPart::TimeSumm() const {
        return m_0_10 + m_10_20 + m_20_50 + m_50_100 + m_100_150 + m_150_190 + m_190_500 + m_more500;
    }

    float TStorageRequestStat::TOperCountStatItemPart::CalcPercentItem(float value, float summ) const {
        if (summ > 0)
            return value / summ * 100;

        return 0.0f;
    }

    TString TStorageRequestStat::TOperCountStatItemPart::PrintWithPercent(bool print_percent, ui64 value, ui64 all) const {
        TString result = UI64ToStroka(value);

        if (print_percent) {
            float percent = CalcPercentItem(static_cast<float>(value), static_cast<float>(all));
            result += " (" + FloatToStr(percent) + "%)";
        }

        return result;
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_0_10_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_0_10, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_10_20_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_10_20, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_20_50_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_20_50, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_50_100_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_50_100, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_100_150_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_100_150, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_150_190_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_150_190, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_190_500_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_190_500, TimeSumm());
    }

    TString TStorageRequestStat::TOperCountStatItemPart::m_more500_s(bool print_percent) const {
        return PrintWithPercent(print_percent, m_more500, TimeSumm());
    }

    TStorageRequestStat::TOperCountStatItemPart::TOperCountStatItemPart() {
        Clear();
    }

    void TStorageRequestStat::TOperCountStatItemPart::Clear() {
        for (size_t i = 0; i < SRS_END; i++)
            m_errors[i] = 0;
        m_0_10 = 0;
        m_10_20 = 0;
        m_20_50 = 0;
        m_50_100 = 0;
        m_100_150 = 0;
        m_150_190 = 0;
        m_190_500 = 0;
        m_more500 = 0;
    }

    ui64 TStorageRequestStat::TOperCountStatItemPart::SummGoodBadCount() {
        ui64 res = 0;

        for (size_t i = 0; i < SRS_END; i++)
            res = IncMax64(res, m_errors[i]);

        return res;
    }

    ui64 TStorageRequestStat::TOperCountStatItemPart::SummQuantileCount() {
        ui64 res = 0;

        res = IncMax64(res, m_0_10);
        res = IncMax64(res, m_10_20);
        res = IncMax64(res, m_20_50);
        res = IncMax64(res, m_50_100);
        res = IncMax64(res, m_100_150);
        res = IncMax64(res, m_150_190);
        res = IncMax64(res, m_190_500);
        res = IncMax64(res, m_more500);

        return res;
    }

    void TStorageRequestStat::TOperCountStatItemPart::Increment(ui32 delay, int count) {
        if (delay < 10)
            m_0_10 = IncMax64(m_0_10, count);
        else if (delay < 20)
            m_10_20 = IncMax64(m_10_20, count);
        else if (delay < 50)
            m_20_50 = IncMax64(m_20_50, count);
        else if (delay < 100)
            m_50_100 = IncMax64(m_50_100, count);
        else if (delay < 150)
            m_100_150 = IncMax64(m_100_150, count);
        else if (delay < 190)
            m_150_190 = IncMax64(m_150_190, count);
        else if (delay < 500)
            m_190_500 = IncMax64(m_190_500, count);
        else
            m_more500 = IncMax64(m_more500, count);
    }

    TStorageRequestStat::TOperCountStatItemPart& TStorageRequestStat::TOperCountStatItemPart::operator+=(const TOperCountStatItemPart& value) {
        for (size_t i = 0; i < SRS_END; i++)
            m_errors[i] = IncMax64(m_errors[i], value.m_errors[i]);

        m_0_10 = IncMax64(m_0_10, value.m_0_10);
        m_10_20 = IncMax64(m_10_20, value.m_10_20);
        m_20_50 = IncMax64(m_20_50, value.m_20_50);
        m_50_100 = IncMax64(m_50_100, value.m_50_100);
        m_100_150 = IncMax64(m_100_150, value.m_100_150);
        m_150_190 = IncMax64(m_150_190, value.m_150_190);
        m_190_500 = IncMax64(m_190_500, value.m_190_500);
        m_more500 = IncMax64(m_more500, value.m_more500);

        return *this;
    }

    TString TStorageRequestStat::TOperCountStatItemPart::PrintToLog() {
        char tbuff[256];

        memset(tbuff, 0, sizeof(tbuff));
        snprintf(tbuff, sizeof(tbuff) - 1, "%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 ",%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "-%" PRIu64 "", m_errors[SRS_OK], m_errors[SRS_POOL_TIMEOUT], m_errors[SRS_RQST_TIMEOUT], m_errors[SRS_OTHER_ERROR], m_0_10, m_10_20, m_20_50, m_50_100, m_100_150, m_150_190, m_190_500, m_more500);

        return TString(tbuff);
    }

    TStorageRequestStat::TOperCountStatItem::TOperCountStatItem() {
        Clear();
    }

    void TStorageRequestStat::TOperCountStatItem::Clear() {
        update_count.Clear();
        findone_count.Clear();
        erase_count.Clear();
        size_count.Clear();
        find_count.Clear();
        multiaction_count.Clear();
        clear_count.Clear();
        findandmodify_count.Clear();
        droptable_count.Clear();
        createtable_count.Clear();
    }

    bool TStorageRequestStat::TOperCountStatItem::Empty() {
        bool res = true;
        bool b[10]{};

        b[0] = update_count.Empty();
        b[1] = findone_count.Empty();
        b[2] = erase_count.Empty();
        b[3] = size_count.Empty();
        b[4] = find_count.Empty();
        b[5] = multiaction_count.Empty();
        b[6] = clear_count.Empty();
        b[7] = findandmodify_count.Empty();
        b[8] = droptable_count.Empty();
        b[9] = createtable_count.Empty();

        for (size_t i = 0; i < 10; i++)
        {
            if (!b[i])
            {
                res = false;
                break;
            }
        }

        return res;
    }

    TStorageRequestStat::TOperCountStatItemPart TStorageRequestStat::TOperCountStatItem::GetAll() {
        TOperCountStatItemPart res = TOperCountStatItemPart();

        res += update_count;
        res += findone_count;
        res += erase_count;
        res += size_count;
        res += find_count;
        res += multiaction_count;
        res += clear_count;
        res += findandmodify_count;
        res += droptable_count;
        res += createtable_count;

        return res;
    }

    TStorageRequestStat::TOperCountStatItem& TStorageRequestStat::TOperCountStatItem::operator+=(const TOperCountStatItem& value) {
        update_count        += value.update_count;
        findone_count       += value.findone_count;
        erase_count         += value.erase_count;
        size_count          += value.size_count;
        find_count          += value.find_count;
        multiaction_count   += value.multiaction_count;
        clear_count         += value.clear_count;
        findandmodify_count += value.findandmodify_count;
        droptable_count     += value.droptable_count;
        createtable_count   += value.createtable_count;

        return *this;
    }

    ui64 TStorageRequestStat::TOperCountStatItem::SummaryErrsCount(TSRSErrType index) {
        ui64 res = 0;

        if ((index >= 0) && (index < SRS_END)) {
            res += update_count.m_errors[index];
            res += findone_count.m_errors[index];
            res += erase_count.m_errors[index];
            res += size_count.m_errors[index];
            res += find_count.m_errors[index];
            res += multiaction_count.m_errors[index];
            res += clear_count.m_errors[index];
            res += findandmodify_count.m_errors[index];
            res += droptable_count.m_errors[index];
            res += createtable_count.m_errors[index];
        }

        return res;
    }

    TString TStorageRequestStat::TOperCountStatItem::PrintToLog() {
        TString res = "";

        //[UP: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[FO: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[E: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[SZ: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[M: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[F: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[CLR: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[FM: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[CT: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]
        //[DT: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500]

        if (!update_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_UPDATE) + ": " + update_count.PrintToLog() + "]";
        if (!findone_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_FINDONE) + ": " + findone_count.PrintToLog() + "]";
        if (!erase_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_ERASE) + ": " + erase_count.PrintToLog() + "]";
        if (!size_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_SIZE) + ": " + size_count.PrintToLog() + "]";
        if (!multiaction_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_MULTYACTION) + ": " + multiaction_count.PrintToLog() + "]";
        if (!find_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_FIND) + ": " + find_count.PrintToLog() + "]";
        if (!clear_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_CLEAR) + ": " + clear_count.PrintToLog() + "]";
        if (!findandmodify_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_FINDANDMODIFY) + ": " + findandmodify_count.PrintToLog() + "]";
        if (!droptable_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_DROPTABLE) + ": " + droptable_count.PrintToLog() + "]";
        if (!createtable_count.Empty())
            res += "[" + m_loglabels.GetParamNameLog(SAT_CREATETABLE) + ": " + createtable_count.PrintToLog() + "]";

        return res;
    }

    void TStorageRequestStat::TOperCountStatItem::ChangeLabelLog(TChangeLabelStructList& change_labels_list) {
        m_loglabels.ChangeLabel(change_labels_list);
    }

    TStorageRequestStat::TOperCountStat::TOperCountStat() {
        Clear();
    }

    void TStorageRequestStat::TOperCountStat::Clear() {
        today.Clear();
        yesterday.Clear();
    }

    void TStorageRequestStat::TOperCountStat::AddCount(TStorageActionType acttype, int count, TSRSErrType good, ui32 tick) {
        switch (acttype) {
            case SAT_UPDATE:
                                    today.update_count.m_errors[good] = IncMax64(today.update_count.m_errors[good], count);
                                    today.update_count.Increment(tick, count);
                                    break;
            case SAT_FINDONE:
                                    today.findone_count.m_errors[good] = IncMax64(today.findone_count.m_errors[good], count);
                                    today.findone_count.Increment(tick, count);
                                    break;
            case SAT_ERASE:
                                    today.erase_count.m_errors[good] = IncMax64(today.erase_count.m_errors[good], count);
                                    today.erase_count.Increment(tick, count);
                                    break;
            case SAT_SIZE:
                                    today.size_count.m_errors[good] = IncMax64(today.size_count.m_errors[good], count);
                                    today.size_count.Increment(tick, count);
                                    break;
            case SAT_MULTYACTION:
                                    today.multiaction_count.m_errors[good] = IncMax64(today.multiaction_count.m_errors[good], count);
                                    today.multiaction_count.Increment(tick, count);
                                    break;
            case SAT_FIND:
                                    today.find_count.m_errors[good] = IncMax64(today.find_count.m_errors[good], count);
                                    today.find_count.Increment(tick, count);
                                    break;
            case SAT_CLEAR:
                                    today.clear_count.m_errors[good] = IncMax64(today.clear_count.m_errors[good], count);
                                    today.clear_count.Increment(tick, count);
                                    break;
            case SAT_FINDANDMODIFY:
                                    today.findandmodify_count.m_errors[good] = IncMax64(today.findandmodify_count.m_errors[good], count);
                                    today.findandmodify_count.Increment(tick, count);
                                    break;
            case SAT_DROPTABLE:
                                    today.droptable_count.m_errors[good] = IncMax64(today.droptable_count.m_errors[good], count);
                                    today.droptable_count.Increment(tick, count);
                                    break;
            case SAT_CREATETABLE:
                                    today.createtable_count.m_errors[good] = IncMax64(today.createtable_count.m_errors[good], count);
                                    today.createtable_count.Increment(tick, count);
                                    break;
            case SAT_END:
                                    break;
        };
    }

    TStorageRequestStat::TOperCountStat& TStorageRequestStat::TOperCountStat::operator+=(const TOperCountStat& value) {
        today += value.today;
        yesterday += value.yesterday;

        return *this;
    }

    void TStorageRequestStat::TOperCountStat::Midnight() {
        yesterday = today;
        today.Clear();
    }

    bool TStorageRequestStat::TOperCountStat::Empty() {
        return today.Empty() && yesterday.Empty();
    }

    TString TStorageRequestStat::TOperCountStat::PrintToLog() {
        TString res = "";

        res = today.PrintToLog();

        return res;
    }

    void TStorageRequestStat::TOperCountStat::ChangeLabelLog(TChangeLabelStructList& change_labels_list) {
        today.ChangeLabelLog(change_labels_list);
    }

    //***********************************************************************************************************************
    //                                              TStorageRequestStat
    //***********************************************************************************************************************

    TStorageStat::TStorageStat(const TString& name)
        : m_name(name)
    {
    }

    TStorageStat::TStorageStat(const TString& name, TStorageRequestStat::TStorageActionType acttype, int count, TSRSErrType good, ui32 tick) {
        m_name = name;
        m_stat.Clear();
        m_stat.AddCount(acttype, count, good, tick);
    }

    void TStorageStat::ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        m_stat.ChangeLabelWeb(change_labels_list);
    }

    void TStorageStat::ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        m_stat.ChangeLabelLog(change_labels_list);
    }

    void TStorageStat::Midnight() {
        m_stat.Midnight();
    }

    bool TStorageStat::Empty()
    {
        return m_stat.Empty();
    }

    bool TStorageStat::operator<(const TStorageStat& value) const {
        return (m_name < value.m_name);
    }

    bool TStorageStat::operator==(const TStorageStat& value) const {
        return (m_name == value.m_name);
    }

    TString TStorageStat::PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary) {
        TString res = "<i>" + m_name + " storage statistik.</i></b>&nbsp;";
        res += m_stat.PrintDataWEB(print_percent, disable_errors, disable_summary);

        return res;
    }

    TString TStorageStat::PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary) {
        TString res = "<i>" + m_name + " storage statistik.</i></b>&nbsp;";
        res += m_stat.PrintDataWEBShort(print_percent, disable_errors, disable_summary);

        return res;
    }

    TString TStorageStat::PrintDataLog() {
        //<DBNameTable: [UPdate: good-bad,0_10-10_20-20_50-50_100-100_150-150_190-190_500-more500][FindOne: ...][Erase: ...][SiZe: ...][Multiaction: ...][Find: ..][CLeaR: ...][FindandModify: ...] >
        return "<" + m_name + ": " + m_stat.PrintDataLog() + " >";
    }

    void TStorageStat::Clear() {
        m_stat.Clear();
    }

    TStorageStatVector& TStorageStatVector::operator+=(const TStorageStatVector& value) {
        TStorageStatVector::iterator it;
        TStorageStatVector::const_iterator it_val;

        it = begin();
        while (it != end()) {
            it_val = std::find(value.begin(), value.end(), TStorageStat((*it).m_name));
            if (it_val != value.end())
                (*it).m_stat += (*it_val).m_stat;

            ++it;
        }

        it_val = value.begin();
        while (it_val != value.end()) {
            it = std::find(begin(), end(), TStorageStat((*it_val).m_name));
            if (it == end())
                push_back(TStorageStat((*it_val)));

            ++it_val;
        }

        return *this;
    }

    void TStorageStatVector::Clear() {
        TStorageStatVector::iterator it;

        for (it = begin(); it != end(); ++it)
            (*it).Clear();
    }

    TString TStorageStatVector::PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";

        std::sort(begin(), end());
        it = begin();
        while (it != end()) {
            res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";

        std::sort(begin(), end());
        it = begin();
        while (it != end()) {
            res += (*it).PrintDataWEBShort(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataWEBInOrder(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";
        std::vector<TString>::const_iterator lit;
        bool exists = false;
        TString st1 = "", st2 = "";

        std::sort(begin(), end());

        lit = orderlist.begin();
        while (lit != orderlist.end()) {
            if (!(*lit).empty()) {
                it = begin();
                while (it != end()) {
                    st1 = (*lit);
                    st2 = (*it).m_name;
                    if ((*lit) == (*it).m_name) {
                        res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";
                        break;
                    }

                    ++it;
                }
            }

            ++lit;
        }

        it = begin();
        while (it != end()) {
            exists = false;
            lit = orderlist.begin();
            while (lit != orderlist.end()) {
                if ((*lit) == (*it).m_name) {
                    exists = true;
                    break;
                }

                ++lit;
            }
            if (!exists)
                res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataWEBInOrderReplace(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary, std::map<TString, TString>& replace_list) {
        std::map<TString, TString>::iterator it;
        size_t rplcpos = 0;
        size_t rplcit = 0;
        TString res = PrintDataWEBInOrder(orderlist, print_percent, disable_errors, disable_summary);

        it = replace_list.begin();
        while (it != replace_list.end()) {
            while ((rplcpos = res.find((*it).first, rplcit)) != TString::npos) {
                res.replace(rplcpos, (*it).first.length(), (*it).second);
                rplcit = rplcpos + 1;
            }

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataWEBInOrderShort(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";
        std::vector<TString>::const_iterator lit;
        bool exists = false;
        TString st1 = "", st2 = "";

        std::sort(begin(), end());

        lit = orderlist.begin();
        while (lit != orderlist.end()) {
            if (!(*lit).empty()) {
                it = begin();
                while (it != end()) {
                    st1 = (*lit);
                    st2 = (*it).m_name;
                    if ((*lit) == (*it).m_name) {
                        res += (*it).PrintDataWEBShort(print_percent, disable_errors, disable_summary) + "<br>";
                        break;
                    }

                    ++it;
                }
            }

            ++lit;
        }

        it = begin();
        while (it != end()) {
            exists = false;
            lit = orderlist.begin();
            while (lit != orderlist.end()) {
                if ((*lit) == (*it).m_name) {
                    exists = true;
                    break;
                }

                ++lit;
            }
            if (!exists)
                res += (*it).PrintDataWEBShort(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataWEBInOrderShortReplace(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary, std::map<TString, TString>& replace_list) {
        std::map<TString, TString>::iterator it;
        size_t rplcpos = 0;
        size_t rplcit = 0;
        TString res = PrintDataWEBInOrderShort(orderlist, print_percent, disable_errors, disable_summary);

        it = replace_list.begin();
        while (it != replace_list.end()) {
            while ((rplcpos = res.find((*it).first, rplcit)) != TString::npos) {
                res.replace(rplcpos, (*it).first.length(), (*it).second);
                rplcit = rplcpos + 1;
            }

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataLog() {
        TStorageStatVector::iterator it;
        TString res = "";

        std::sort(begin(), end());
        it = begin();
        while (it != end()) {
            res += (*it).PrintDataLog();

            ++it;
        }

        return res;
    }

    TString TStorageStatVector::PrintDataLogReplace(std::map<TString, TString>& replace_list) {
        std::map<TString, TString>::iterator it;
        size_t rplcpos = 0;
        size_t rplcit = 0;
        TString res = PrintDataLog();

        it = replace_list.begin();
        while (it != replace_list.end()) {
            while ((rplcpos = res.find((*it).first, rplcit)) != TString::npos) {
                res.replace(rplcpos, (*it).first.length(), (*it).second);
                rplcit = rplcpos + 1;
            }

            ++it;
        }

        return res;
    }

    void TStorageStatVector::ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        TStorageStatVector::iterator it;

        it = begin();
        while (it != end()) {
            (*it).ChangeLabelWeb(change_labels_list);

            ++it;
        }
    }

    void TStorageStatVector::ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        TStorageStatVector::iterator it;

        it = begin();
        while (it != end()) {
            (*it).ChangeLabelLog(change_labels_list);

            ++it;
        }
    }

    TStorageRequestStat::TStorageRequestStat() {
    }

    TStorageRequestStat::~TStorageRequestStat() {
    }

    TStorageRequestStat& TStorageRequestStat::operator+=(const TStorageRequestStat& value) {
        data += value.data;
        return *this;
    }

    void TStorageRequestStat::AddCount(TStorageActionType acttype, int count, TSRSErrType good, ui32 tick) {
        data.AddCount(acttype, count, good, tick);
    }

    void TStorageRequestStat::Midnight() {
        data.Midnight();
    }

    bool TStorageRequestStat::Empty() {
        return data.Empty();
    }

    TString TStorageRequestStat::PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary) {
        return PrintDataWEBImpl(print_percent, disable_errors, disable_summary, true);
    }

    TString TStorageRequestStat::PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary) {
        return PrintDataWEBImpl(print_percent, disable_errors, disable_summary, false);
    }

    TString TStorageRequestStat::GetErrorRow(const TString& label, TStorageRequestStat::TOperCountStatItemPart today_count, TStorageRequestStat::TOperCountStatItemPart yesterday_count, bool print_percent, bool print_always, ui32& count) {
        if (!print_always && (today_count.SummGoodBadCount() == 0) && (yesterday_count.SummGoodBadCount() == 0))
            return TString();

        ++count;
        TString today, yesterday;
        for (size_t i = 0; i < SRS_END; ++i) {
            today += "<td width='9%'>" + today_count.PrintWithPercent(print_percent, today_count.m_errors[i], today_count.SummGoodBadCount()) + "</td>";
            yesterday += "<td width='9%'>" + yesterday_count.PrintWithPercent(print_percent, yesterday_count.m_errors[i], yesterday_count.SummGoodBadCount()) + "</td>";
        }

        return "<tr align='right'><td width='28%' align='left' colspan='2'>" + label + "</td>" + today + yesterday + "<tr>";
    }

    static TString GetDelayRow(TStorageRequestStat::TOperCountStatItemPart count, bool print_percent) {
        return "<td>" + count.m_0_10_s(print_percent) + "</td><td>" + count.m_10_20_s(print_percent) + "</td><td>" + count.m_20_50_s(print_percent) + "</td><td>" + count.m_50_100_s(print_percent) + "</td>" + "<td>" + count.m_100_150_s(print_percent) + "</td><td>" + count.m_150_190_s(print_percent) + "</td><td>" + count.m_190_500_s(print_percent) + "</td><td>" + count.m_more500_s(print_percent) + "</td>";
    }

    TString TStorageRequestStat::GetDelayRow(const TString& label, TStorageRequestStat::TOperCountStatItemPart today_count, TStorageRequestStat::TOperCountStatItemPart yesterday_count, bool print_percent, bool print_always, ui32& count) {
        if (!print_always && (today_count.SummQuantileCount() == 0) && (yesterday_count.SummQuantileCount() == 0))
            return TString();

        ++count;
        return "<tr align='right'><td width='23%' align='left' rowspan='2'>" + label + "</td><td align='center' width='5%'>today</td>" + ::NStorageStats::GetDelayRow(today_count, print_percent) + "</tr>" + "<tr align='right' bgcolor='#F8FCBF'><td align='center' width='5%'>yesterday</td>" + ::NStorageStats::GetDelayRow(yesterday_count, print_percent) + "</tr>";
    }

    TString TStorageRequestStat::PrintDataWEBImpl(bool print_percent, bool disable_errors, bool disable_summary, bool print_all) {
        const char* shap_cell_color = "'#ffffaa'";

        TOperCountStatItemPart today_summary = data.today.GetAll();
        TOperCountStatItemPart yesterday_summary = data.yesterday.GetAll();

        TString text = "<table border='1' width='100%' cellspacing='0' cellpadding='4' bgcolor='#ffffcc'>\n";
        if (!disable_errors) {
            text += "<tr bgcolor=" + TString(shap_cell_color) + " align='center'><td width='28%' align='left' colspan='2'><b>ERRORS</b></td><td colspan='4' width='35%'><b>TODAY</b></td><td colspan='4' width='35%'><b>YESTERDAY</b></td><tr>";
            text += "<tr bgcolor=" + TString(shap_cell_color) + " align='center'><td width='28%' align='left' colspan='2'>&nbsp;</td><td width='9%'><b>OK</b></td><td width='9%'><b>POOL_TO</b></td><td width='9%'><b>RQST_TO</b></td><td width='9%'><b>OTH_ERR</b></td><td width='9%'><b>OK</b></td><td width='9%'><b>POOL_TO</b></td><td width='9%'><b>RQST_TO</b></td><td width='9%'><b>OTH_ERR</b></td><tr>";

            ui32 bcount = 0;
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_UPDATE), data.today.update_count, data.yesterday.update_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_FINDONE), data.today.findone_count, data.yesterday.findone_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_FIND), data.today.find_count, data.yesterday.find_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_ERASE), data.today.erase_count, data.yesterday.erase_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_SIZE), data.today.size_count, data.yesterday.size_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_MULTYACTION), data.today.multiaction_count, data.yesterday.multiaction_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_CLEAR), data.today.clear_count, data.yesterday.clear_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_FINDANDMODIFY), data.today.findandmodify_count, data.yesterday.findandmodify_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_DROPTABLE), data.today.droptable_count, data.yesterday.droptable_count, print_percent, print_all, bcount);
            text += GetErrorRow(m_weblabels.GetParamNameWeb(SAT_CREATETABLE), data.today.createtable_count, data.yesterday.createtable_count, print_percent, print_all, bcount);

            if ((bcount > 1) && (!disable_summary))
                text += GetErrorRow("SUMMARY (all type request)", today_summary, yesterday_summary, print_percent, true, bcount);
        }

        ui32 bcount = 0;
        text = text + "<tr bgcolor=" + TString(shap_cell_color) + " align='center'><td width='28%' align='left' colspan='2'><b>DELAYS (ms)</b></td><td><b>0-10</b></td><td><b>10-20</b></td><td><b>20-50</b></td><td><b>50-100</b></td><td><b>100-150</b></td><td><b>150-190</b></td><td><b>190-500</b></td><td><b>more500</b></td></tr>";

        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_UPDATE), data.today.update_count, data.yesterday.update_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_FINDONE), data.today.findone_count, data.yesterday.findone_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_FIND), data.today.find_count, data.yesterday.find_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_ERASE), data.today.erase_count, data.yesterday.erase_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_SIZE), data.today.size_count, data.yesterday.size_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_MULTYACTION), data.today.multiaction_count, data.yesterday.multiaction_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_CLEAR), data.today.clear_count, data.yesterday.clear_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_FINDANDMODIFY), data.today.findandmodify_count, data.yesterday.findandmodify_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_DROPTABLE), data.today.droptable_count, data.yesterday.droptable_count, print_percent, print_all, bcount);
        text += GetDelayRow(m_weblabels.GetParamNameWeb(SAT_CREATETABLE), data.today.createtable_count, data.yesterday.createtable_count, print_percent, print_all, bcount);

        if ((bcount > 1) && (!disable_summary))
            text += GetDelayRow("SUMMARY (all type request)", today_summary, yesterday_summary, print_percent, true, bcount);

        text = text + "</table>";

        return text;
    }

    void TStorageRequestStat::ChangeLabelLog(TChangeLabelStructList& change_labels_list) {
        data.ChangeLabelLog(change_labels_list);
    }

    TString TStorageRequestStat::PrintDataLog() {
        return data.PrintToLog();
    }

    //***********************************************************************************************************************
    //                                              TStorageRequestStatMain
    //***********************************************************************************************************************

    TStorageRequestStatMain::TStorageRequestStatMain() {
    }

    TStorageRequestStatMain::~TStorageRequestStatMain() {
    }

    void TStorageRequestStatMain::AddCount(const TString& db, const TString& collection, TStorageRequestStat::TStorageActionType acttype, int count, TSRSErrType good, ui32 tick) {
        m_Mutex.Acquire();

        TString longname = "";
        TStorageStatVectorIt it;

        longname = db + "::" + collection;
        it = std::find(dbcoll_stor.begin(), dbcoll_stor.end(), TStorageStat(longname));
        if (it != dbcoll_stor.end())
            (*it).m_stat.AddCount(acttype, count, good, tick);
        else {
            //TStorageStatVector::iterator it = dbcoll_stor.insert(dbcoll_stor.end(), TStorageStatVector::value_type(TStorageStat(longname, acttype, count, good, tick)));
            //if (it != dbcoll_stor.end())
            //   (*it).Init();

            dbcoll_stor.push_back(TStorageStat(longname, acttype, count, good, tick));
        }

        m_Mutex.Release();
    }

    void TStorageRequestStatMain::Midnight() {
        m_Mutex.Acquire();

        TStorageStatVectorIt it;

        it = dbcoll_stor.begin();
        while (it != dbcoll_stor.end()) {
            (*it).Midnight();

            ++it;
        }

        m_Mutex.Release();
    }

    void TStorageRequestStatMain::DeleteEmpty() {
        m_Mutex.Acquire();

        TStorageStatVectorIt it;

        it = dbcoll_stor.begin();
        while (it != dbcoll_stor.end()) {
            if ((*it).Empty())
            {
                dbcoll_stor.erase(it);
            }
            else
            {
                ++it;
            }
        }

        m_Mutex.Release();
    }

    size_t TStorageRequestStatMain::Count()
    {
        return dbcoll_stor.size();
    }

    TStorageStatVector TStorageRequestStatMain::GetWebStat() {
        return dbcoll_stor;
    }

    TStorageStatVector TStorageRequestStatMain::GetMonStat() {
        TStorageStatVector res;

        m_Mutex.Acquire();

        res = dbcoll_stor;
        dbcoll_stor.Clear();

        m_Mutex.Release();

        return res;
    }

    void TStorageRequestStatMain::ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        dbcoll_stor.ChangeLabelWeb(change_labels_list);
    }

    void TStorageRequestStatMain::ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list) {
        dbcoll_stor.ChangeLabelLog(change_labels_list);
    }

    //***********************************************************************************************************************
} // namespace NStorage
