#pragma once

#include <util/generic/string.h>
#include <util/system/mutex.h>
#include <util/generic/hash.h>

#include <vector>
#include <map>

#include "kfunc.h"

namespace NStorageStats {
    //***********************************************************************************************************************
    //                                              TStorageRequestStat
    //***********************************************************************************************************************

    enum TSRSErrType { SRS_OK,
                       SRS_POOL_TIMEOUT,
                       SRS_RQST_TIMEOUT,
                       SRS_OTHER_ERROR,
                       SRS_END };

    TSRSErrType ToTSRSErrTypeB(bool good);

    class TStorageRequestStat {
    public:
        enum TStorageActionType { SAT_UPDATE,
                                  SAT_FINDONE,
                                  SAT_ERASE,
                                  SAT_SIZE,
                                  SAT_MULTYACTION,
                                  SAT_FIND,
                                  SAT_CLEAR,
                                  SAT_FINDANDMODIFY,
                                  SAT_DROPTABLE,
                                  SAT_CREATETABLE,
                                  SAT_END
                                };

        struct TChangeLabelStruct {
            TStorageActionType m_acttype;
            TString m_weblabel;
            TString m_loglabel;

            TChangeLabelStruct();
            TChangeLabelStruct(TStorageActionType acttype, const TString& weblabel, const TString& loglabel);

            void Clear();
        };

        typedef std::vector<TChangeLabelStruct> TChangeLabelStructList;

        struct TWebLabels {
            TString m_web_labels_array[SAT_END];

            TWebLabels();
            void Init();

            void ChangeLabel(TChangeLabelStructList& change_labels_list);
            TString GetParamNameWeb(TStorageActionType acttype);
        };

        struct TLogLabels {
            TString m_log_labels_array[SAT_END];

            TLogLabels();
            void Init();

            void ChangeLabel(TChangeLabelStructList& change_labels_list);
            TString GetParamNameLog(TStorageActionType acttype);
        };

        struct TOperCountStatItemPart {
            //ui64 good;        //good request count
            //ui64 bad;         //bad request count
            ui64 m_errors[SRS_END];
            ui64 m_0_10;    //0 - 10 ms
            ui64 m_10_20;   //10 - 20 ms
            ui64 m_20_50;   //20 - 50 ms
            ui64 m_50_100;  //50 - 100 ms
            ui64 m_100_150; //100 - 150 ms
            ui64 m_150_190; //150 - 190 ms
            ui64 m_190_500; //190 - 500 ms
            ui64 m_more500; //more 500 ms

            TOperCountStatItemPart();

            bool Empty();
            ui64 TimeSumm() const;
            float CalcPercentItem(float value, float summ) const;

            TString PrintWithPercent(bool print_percent, ui64 value, ui64 all) const;

            TString m_0_10_s(bool print_percent) const;
            TString m_10_20_s(bool print_percent) const;
            TString m_20_50_s(bool print_percent) const;
            TString m_50_100_s(bool print_percent) const;
            TString m_100_150_s(bool print_percent) const;
            TString m_150_190_s(bool print_percent) const;
            TString m_190_500_s(bool print_percent) const;
            TString m_more500_s(bool print_percent) const;

            void Clear();
            ui64 SummGoodBadCount();
            ui64 SummQuantileCount();
            void Increment(ui32 delay, int count);
            TOperCountStatItemPart& operator+=(const TOperCountStatItemPart& value);
            TString PrintToLog();
        };

        struct TOperCountStatItem {
            TLogLabels m_loglabels;
            TOperCountStatItemPart update_count;
            TOperCountStatItemPart findone_count;
            TOperCountStatItemPart erase_count;
            TOperCountStatItemPart size_count;
            TOperCountStatItemPart multiaction_count;
            TOperCountStatItemPart find_count;
            TOperCountStatItemPart clear_count;
            TOperCountStatItemPart findandmodify_count;
            TOperCountStatItemPart droptable_count;
            TOperCountStatItemPart createtable_count;

            TOperCountStatItem();

            void Clear();
            bool Empty();
            TOperCountStatItemPart GetAll();
            TOperCountStatItem& operator+=(const TOperCountStatItem& value);
            //ui64 GoodCount()
            //ui64 BadCount()

            ui64 SummaryErrsCount(TSRSErrType index);

            TString PrintToLog();
            void ChangeLabelLog(TChangeLabelStructList& change_labels_list);
        };

        struct TOperCountStat {
            TOperCountStatItem today;
            TOperCountStatItem yesterday;

            TOperCountStat();

            void Clear();
            void AddCount(TStorageActionType acttype, int count, TSRSErrType good, ui32 tick);
            TOperCountStat& operator+=(const TOperCountStat& value);

            void Midnight();
            bool Empty();
            TString PrintToLog();
            void ChangeLabelLog(TChangeLabelStructList& change_labels_list);
        };

    public:
        TWebLabels m_weblabels;
        TOperCountStat data;

        TStorageRequestStat();
        ~TStorageRequestStat();

        void AddCount(TStorageRequestStat::TStorageActionType acttype, int count, TSRSErrType good, ui32 tick);
        void Midnight();
        bool Empty();
        TOperCountStat GetStat() {
            return data;
        }
        void Clear() {
            data.Clear();
        }

        TStorageRequestStat& operator+=(const TStorageRequestStat& value);

        void ChangeLabelWeb(TChangeLabelStructList& change_labels_list) {
            m_weblabels.ChangeLabel(change_labels_list);
        }
        TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary);

        TString PrintDataLog();
        void ChangeLabelLog(TChangeLabelStructList& change_labels_list);

    private:
        TString PrintDataWEBImpl(bool print_percent, bool disable_errors, bool disable_summary, bool print_all);
        TString GetErrorRow(const TString& label, TStorageRequestStat::TOperCountStatItemPart today_count, TStorageRequestStat::TOperCountStatItemPart yesterday_count, bool print_percent, bool print_always, ui32& count);
        TString GetDelayRow(const TString& label, TStorageRequestStat::TOperCountStatItemPart today_count, TStorageRequestStat::TOperCountStatItemPart yesterday_count, bool print_percent, bool print_always, ui32& count);
    };

    //***********************************************************************************************************************
    //                                              TStorageRequestStatMain
    //***********************************************************************************************************************

    struct TStorageStat {
        TString m_name;
        TStorageRequestStat m_stat;

        TStorageStat() = default;
        TStorageStat(const TString& name);
        TStorageStat(const TString& name, TStorageRequestStat::TStorageActionType acttype, int count, TSRSErrType good, ui32 tick);

        TStorageStat(const TStorageStat&) = default;
        TStorageStat& operator=(const TStorageStat&) = default;

        void ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
        void ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
        void Midnight();
        bool Empty();

        bool operator<(const TStorageStat& value) const;
        bool operator==(const TStorageStat& value) const;

        TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary);

        TString PrintDataLog();
        void Clear();
    };

    class TStorageStatVector: public std::vector<TStorageStat> {
    private:
    public:
        TStorageStatVector& operator+=(const TStorageStatVector& value);
        void Clear();

        TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBInOrder(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBInOrderReplace(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary, std::map<TString, TString>& replace_list);
        TString PrintDataWEBInOrderShort(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary);
        TString PrintDataWEBInOrderShortReplace(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary, std::map<TString, TString>& replace_list);

        TString PrintDataLog();
        TString PrintDataLogReplace(std::map<TString, TString>& replace_list);
        void ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
        void ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
    };

    typedef TStorageStatVector::iterator TStorageStatVectorIt;

    class TStorageRequestStatMain {
    private:
        TStorageStatVector dbcoll_stor;
        TMutex m_Mutex;

    public:
        TStorageRequestStatMain();
        ~TStorageRequestStatMain();

        void   AddCount(const TString& db, const TString& collection, TStorageRequestStat::TStorageActionType acttype, int count, TSRSErrType good, ui32 tick);
        void   Midnight();
        void   DeleteEmpty();
        size_t Count();
        TStorageStatVector GetWebStat();
        TStorageStatVector GetMonStat();

        void ChangeLabelWeb(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
        void ChangeLabelLog(TStorageRequestStat::TChangeLabelStructList& change_labels_list);
    };

    //***********************************************************************************************************************
} // namespace NStorage
