#pragma once

#include <util/generic/string.h>
#include "util/system/mutex.h"
#include "kfunc.h"
#include <vector>

namespace trcstat
{
//*******************************************************************************************************************************************
//                                                    TTraccertStatItem
//*******************************************************************************************************************************************

struct TDiapEl
{
   TString m_label;
   ui32   m_min;
   ui32   m_max;

   TDiapEl()
   {
      Clear();
   }

   TDiapEl(const TString &label, ui32 min, ui32 max)
   {
      m_label = label;
      m_min   = min;
      m_max   = max;
   }

   void Clear()
   {
      m_label = "";
      m_min   = 0;
      m_max   = 0;
   }

};

typedef std::vector<TDiapEl> TDiapElVector;

class TTraccertStatItem
{
public:
         static const ui8 DEFAUL_CNT_COUNT = 12;

         TString m_srvc_label;
         ui32   m_count[DEFAUL_CNT_COUNT];

public:
         TTraccertStatItem()
         {
            Clear();
         }

         TTraccertStatItem(const TString &srvc_label)
         {
            Clear();
            m_srvc_label = srvc_label;
         }

         TTraccertStatItem(const TString &srvc_label, ui32 tick, const TDiapElVector &diap_obj)
         {
            Clear();
            m_srvc_label = srvc_label;
            Increment(tick, diap_obj);
         }

         void Clear()
         {
            m_srvc_label = "";
            for (size_t i = 0 ; i < DEFAUL_CNT_COUNT; i++)
               m_count[i] = 0;
         }

         void Increment(ui32 tick, const TDiapElVector &diap_obj)
         {
            int index = -1;

            if (diap_obj.size() > 0)
            {
               for (size_t i = 0; i < diap_obj.size(); i++)
               {
                  if ( (tick >= diap_obj[i].m_min) && (tick <= diap_obj[i].m_max) )
                  {
                     index = i;
                     break;
                  }
               }

               if ( (index >= 0) && (index < DEFAUL_CNT_COUNT) )
                  m_count[index] = IncMax32(m_count[index], 1);

            }
         }

         bool operator<(const TTraccertStatItem &value) const
         {
            return m_srvc_label < value.m_srvc_label;

         }
};

typedef std::vector<TTraccertStatItem>    TTraccertStatItemVector;
typedef TTraccertStatItemVector::iterator TTraccertStatItemVectorIt;

typedef THashMap<TString, TTraccertStatItem>  TTraccertStatItemHash;
typedef TTraccertStatItemHash::iterator   TTraccertStatItemHashIt;

//*******************************************************************************************************************************************
//                                                       TTraccertStat
//*******************************************************************************************************************************************

class TTraccertStat
{
private:
         TDiapElVector         diap_vect;
         TTraccertStatItemHash srvc_hash_today;

         TMutex                m_Mutex;
public:
         TTraccertStat();

         void     Init();
         void     AddTick(const TString &ident, ui32 tick);
         void     Midnight();

         TString   GetWebData();


};

//*******************************************************************************************************************************************

}

