#include "so_log.h"
#include "tudnsclass.h"
#include "shtime.h"

dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const TIpAddr& addr) {
    if (addr.IsIpv6())
        return dns_resolve_a6ptr(ctx, addr.GetIn6Addr());
    else
        return dns_resolve_a4ptr(ctx, addr.GetInAddr());
}

dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const TIpAddr& addr, const char* dnsbl) {
    if (addr.IsIpv6())
        return dns_resolve_a6dnsbl(ctx, addr.GetIn6Addr(), dnsbl);
    else
        return dns_resolve_a4dnsbl(ctx, addr.GetInAddr(), dnsbl);
}

dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const TIpAddr& addr, const char* dnsbl) {
    if (addr.IsIpv6())
        return dns_resolve_a6dnsbl_txt(ctx, addr.GetIn6Addr(), dnsbl);
    else
        return dns_resolve_a4dnsbl_txt(ctx, addr.GetInAddr(), dnsbl);
}

dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const struct in_addr* addr) {
    return dns_resolve_a4ptr(ctx, addr);
}

dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const struct in6_addr* addr) {
    return dns_resolve_a6ptr(ctx, addr);
}

dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const struct in_addr* addr, const char* dnsbl) {
    return dns_resolve_a4dnsbl(ctx, addr, dnsbl);
}

dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const struct in6_addr* addr, const char* dnsbl) {
    return dns_resolve_a6dnsbl(ctx, addr, dnsbl);
}

dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const struct in_addr* addr, const char* dnsbl) {
    return dns_resolve_a4dnsbl_txt(ctx, addr, dnsbl);
}

dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const struct in6_addr* addr, const char* dnsbl) {
    return dns_resolve_a6dnsbl_txt(ctx, addr, dnsbl);
}

bool TUdnsClass::Open(const TString& sSrv) {
    return Open(53, sSrv);
}

bool TUdnsClass::Open(int port, const TString& sSrv) {
    if (context) {
        if (dns_init(context, 0) < 0) {
            if (is_spk) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KERROR, "udns dns_init() error: %s", dns_strerror(errno));
            }
        }

        sprintf(timeoutstr, "timeout:%d", dns_timeout);
        if (dns_set_opts(context, timeoutstr) != 0) {
            if (is_spk) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KWARNING, "udns error setting timeout %s", timeoutstr);
            }
        }

        if (dns_set_opts(context, "attempts:1") != 0) {
            if (is_spk) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KWARNING, "udns error setting attempts");
            }
        }

        if (sSrv.length() > 0) {
            dns_add_serv(context, nullptr);
            if (dns_add_serv(context, sSrv.c_str()) < 0) {
                if (is_spk) {
                    if (log != nullptr)
                        log->WriteMessageAndDataStatus(KERROR, "udns add_serv error %s for %s", dns_strerror(errno),  sSrv.c_str());
                }
            }
        }

        if (port > 0) {
            if (dns_set_opt(context, DNS_OPT_PORT, port) < 0) {
                if (is_spk) {
                    if (log != nullptr)
                        log->WriteMessageAndDataStatus(KWARNING, "udns error setting port %d", port);
                }
            }
        }

        if (dns_open(context) < 0) {
            if (is_spk) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KERROR, "udns open error: %s", dns_strerror(errno));
            }
            return false;
        }
        bReady = true;
        return true;
    }
    return false;
}


bool TUdnsClass::ResolveHost(const TString& NumbRequest, const char* sHost, TVector<TString>* pResults)
{
    struct dns_rr_a4* rr_a4 = nullptr;
    struct dns_rr_a6* rr_a6 = nullptr;
    bool result = false;
    char buf[DNS_MAXLABEL];
    ui32 tick = 0;

    udnsMutex.Acquire();
    if (!pResults)
        pResults = &vResults;
    pResults->clear();
    tick = CShingleTime::GetMs();

    rr_a6 = dns_resolve_a6(context, sHost, 0);
    int err6 = dns_status(context);
    rr_a4 = dns_resolve_a4(context, sHost, 0);
    int err4 = dns_status(context);

    tick = CShingleTime::GetMs() - tick;

    errcode = std::max(err4, err6);
    if (errcode < 0) {
        if (is_spk) {
            if (errcode == DNS_E_TEMPFAIL || errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KERROR, "% 3u %s udns_%s error '%s' for %s", tick,
                                                   NumbRequest.c_str(), sLabel.empty() ? "loc" : sLabel.c_str(),
                                                   dns_strerror(errcode), sHost);
            }
        }
    } else
        result = true;

    if (rr_a4) {
        for (int i = 0; i < rr_a4->dnsa4_nrr; i++) {
            memset(buf, '\0', sizeof(buf));
            inet_ntop(AF_INET, (void*)&rr_a4->dnsa4_addr[i], buf, sizeof(buf));
            pResults->push_back(buf);
        }
        free(rr_a4);
    }
    if (rr_a6) {
        for (int i = 0; i < rr_a6->dnsa6_nrr; i++) {
            memset(buf, '\0', sizeof(buf));
            inet_ntop(AF_INET6, (void*)&rr_a6->dnsa6_addr[i], buf, sizeof(buf));
            pResults->push_back(buf);
        }
        free(rr_a6);
    }

    if (is_spk) {
        if (!pResults->empty())
            for (auto it = pResults->begin(); it != pResults->end(); it++) {
                if (log != nullptr)
                    log->WriteMessageAndDataStatus(KMESSAGE, "% 3u %s udns_%s %s resolved to %s", tick,
                                                   NumbRequest.c_str(), sLabel.empty() ? "loc" : sLabel.c_str(), sHost,
                                                   it->c_str());
            }
    }

    udnsMutex.Release();

    return result;
}

bool TUdnsClass::ResolveDNSBL(const TString& NumbRequest, const TIpAddr& host, const char* zone, TString& resvalue) {
    bool res = false;
    struct dns_rr_txt* rr_txt = nullptr;
    ui32 tick = 0;

    resvalue = "";

    udnsMutex.Acquire();

    tick = CShingleTime::GetMs();
    rr_txt = dns_resolve_dnsbl_txt(context, host, zone);
    tick = CShingleTime::GetMs() - tick;
    errcode = dns_status(context);
    if (rr_txt && (rr_txt->dnstxt_txt[0].len > 0)) {
        sStr = (const char*)rr_txt->dnstxt_txt[0].txt;
    }
    // DNS_E_TEMPFAIL occurs when DNSBL list contains no record for host, must not log error
    if (/*errcode == DNS_E_TEMPFAIL ||*/ errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY) {
        if (log != nullptr)
            log->WriteMessageAndDataStatus(KERROR, "% 3u udns_%s error: %s for %s.%s", tick, NumbRequest.c_str(), sLabel.empty() ? "bl" : sLabel.c_str(), dns_strerror(errcode), host.ToDNSBLString().c_str(), zone);
    }

    if (rr_txt) {
        free(rr_txt);
        resvalue = TString(GetStr());
        res = true;
    }

    udnsMutex.Release();

    return res;
}

bool TUdnsClass::ResolveAddr(const TIpAddr& host) {
    struct dns_rr_ptr* rr = nullptr;
    ui32 tick = 0;

    tick = CShingleTime::GetMs();
    rr = dns_resolve_ptr(context, host);
    tick = CShingleTime::GetMs() - tick;
    errcode = dns_status(context);

    if (rr)
        sStr = *rr->dnsptr_ptr;

    if (rr) {
        free(rr);
        return true;
    }
    return false;
}

bool TUdnsClass::ResolveAddr(const TString& NumbRequest, const TIpAddr& host, TString& resvalue) {
    bool res = false;
    struct dns_rr_ptr* rr = nullptr;
    ui32 tick = 0;

    resvalue = "";

    udnsMutex.Acquire();

    tick = CShingleTime::GetMs();
    rr = dns_resolve_ptr(context, host);
    tick = CShingleTime::GetMs() - tick;
    errcode = dns_status(context);

    if (rr)
        sStr = *rr->dnsptr_ptr;

    if (errcode == DNS_E_TEMPFAIL || errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY) {
        if (log != nullptr)
            log->WriteMessageAndDataStatus(KERROR, "% 3u %s udns_%s error: %s for %s", tick, NumbRequest.c_str(), sLabel.empty() ? "ptr" : sLabel.c_str(), dns_strerror(errcode), host.ToString().c_str());
    }

    if (rr) {
        free(rr);
        resvalue = TString(GetStr());
        res = true;
    }

    udnsMutex.Release();

    return res;
}
