#include "unnamedsem.h"

#ifdef _win_
#include <malloc.h>
#elif defined(_sun)
#include <alloca.h>
#endif

#include <cerrno>
#include <cstring>

#ifdef _win_
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#else
#include <signal.h>
#include <unistd.h>
#include <semaphore.h>
#endif

class TUnnamedSemaphore::TImpl {
private:
#ifdef _win_
    typedef HANDLE SEMHANDLE;
#else
    typedef sem_t SEMHANDLE;
#endif

    bool inited;
    SEMHANDLE Handle;
    int Errno;

public:
    inline TImpl(ui32 max_free_count) {
        inited = false;
        Errno = 0;
#ifdef _win_
        const char* name = "win_sema_name";

        char* key = (char*)name;
        if (name) {
            size_t len = strlen(name);
            key = (char*)alloca(len + 1);
            strcpy(key, name);
            if (len > MAX_PATH)
                *(key + MAX_PATH) = 0;
            char* p = key;
            while (*p) {
                if (*p == '\\')
                    *p = '/';
                p++;
            }
        }
        // non-blocking on init
        Handle = ::CreateSemaphore(0, max_free_count, max_free_count, nullptr);
        inited = (Handle != nullptr);
#else
        inited = sem_init(&Handle, 0, max_free_count) == 0;
        if (!inited)
            Errno = errno;
#endif
    }

    inline ~TImpl() {
        if (!inited)
            return;

#ifdef _win_
        ::CloseHandle(Handle);
#else
        if (sem_destroy(&Handle) == -1)
            Errno = errno;
#endif
    }

    inline bool Release() {
        if (!inited)
            return false;
#ifdef _win_
        return ::ReleaseSemaphore(Handle, 1, 0) ? true : false;
#else

        int ret = sem_post(&Handle);
        if (ret != 0)
            Errno = errno;
        return ret == 0;
#endif
    }

    //The UNIX semaphore object does not support a timed "wait", and
    //hence to maintain consistancy, for win32 case we use INFINITE or 0 timeout.
    inline bool Acquire() {
        if (!inited)
            return false;
#ifdef _win_
        return ::WaitForSingleObject(Handle, INFINITE) == WAIT_OBJECT_0;
#else
        int ret = sem_wait(&Handle);
        if (ret != 0)
            Errno = errno;
        return ret == 0;
#endif
    }

    inline bool TimedAcquire(const TDuration& timeout) {
        if (!inited)
            return false;

#ifdef _win_
        return ::WaitForSingleObject(Handle, timeout.MilliSeconds()) == WAIT_OBJECT_0;
#else
        const TInstant& whenToWakeUp = Now() + timeout;

        return TimedAcquire(whenToWakeUp);
#endif
    }

    inline bool TimedAcquire(const TInstant& whenToWakeUp) {
        if (!inited)
            return false;

#ifdef _win_
        TDuration timeout = whenToWakeUp - Now();
        return ::WaitForSingleObject(Handle, timeout.MilliSeconds()) == WAIT_OBJECT_0;
#else

        ui64 secs = whenToWakeUp.NanoSeconds() / 1000000000;
        ui64 nsecs = whenToWakeUp.NanoSeconds() % 1000000000;

        const timespec posixTimeout = {time_t(secs), long(nsecs)};
        int ret = 0;
        while ((ret = sem_timedwait(&Handle, &posixTimeout)) == -1 && (errno == EINTR))
            continue;
        if (ret != 0)
            Errno = errno;
        return ret == 0;
#endif
    }

    inline bool TryAcquire() {
        if (!inited)
            return false;
#ifdef _win_
        // zero-second time-out interval
        // WAIT_OBJECT_0: current free count > 0
        // WAIT_TIMEOUT:  current free count == 0
        return ::WaitForSingleObject(Handle, 0) == WAIT_OBJECT_0;
#else
        int ret = sem_trywait(&Handle);
        if (ret != 0)
            Errno = errno;
        return ret == 0;
#endif
    }

    inline const char* get_strerror() {
#ifdef _win_
        return "";
#else
        return strerror(Errno);
#endif
    }
};

TUnnamedSemaphore &TUnnamedSemaphore::operator=(TUnnamedSemaphore&&) noexcept = default;
TUnnamedSemaphore::TUnnamedSemaphore(TUnnamedSemaphore&&) noexcept = default;

TUnnamedSemaphore::TUnnamedSemaphore(ui32 maxFreeCount)
    : Impl_(new TImpl(maxFreeCount))
{
}

TUnnamedSemaphore::~TUnnamedSemaphore() = default;

bool TUnnamedSemaphore::Release() {
    return Impl_->Release();
}

bool TUnnamedSemaphore::Acquire() {
    return Impl_->Acquire();
}

bool TUnnamedSemaphore::TimedAcquire(const TInstant& until) {
    return Impl_->TimedAcquire(until);
}

bool TUnnamedSemaphore::TimedAcquire(const TDuration& timeout) {
    return Impl_->TimedAcquire(timeout);
}

bool TUnnamedSemaphore::TryAcquire() {
    return Impl_->TryAcquire();
}
