#pragma once

#ifndef urlparser_H
#define urlparser_H

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/string.h>

#include <mail/so/libs/tld/is_tld.h>

#include <list>
#include <array>

class TZoneDetector {
public:
    enum TSymbEncodeType { TE_WIN1251,
        TE_KOI8R };

public:
    struct TKUrlItem {
        TString m_url;       //само значение урла
        TString m_host;      //только хост
        ui8    m_is_http;    //перед урлом есть http://, значение означает порядковый номер в http_ident_list, 0 - нет http
        int    m_starturl;   //позиция от начала буфера, где начинается выделенный урл (хост)
        ui8    m_count;      //количество

        TKUrlItem();
        TKUrlItem(const TString &url, const TString &host, ui8 is_http, int starturl, ui8 count);
        void Clear();

        ui8 GetCount() const;
        const TString& GetValue() const;
    };

    typedef std::list<TKUrlItem> TKUrlList;
    typedef TKUrlList::iterator TKUrlListIt;

    typedef THashMap<TString, TKUrlItem> TKUrlHash;
    typedef TKUrlHash::iterator TKUrlHashIt;
public:
    typedef THashSet<TString> TZoneDetectorSet;

    typedef std::list<TString> TStrokaList;
    typedef TStrokaList::iterator TStrokaListIt;

    typedef THashMap<TString, ui8> TStrokaHash;
    typedef TStrokaHash::iterator TStrokaHashIt;

    static const ui32 MAX_URLMAIL_COUNT = 100;

private:
    TSymbEncodeType encodeType;
    TZoneDetectorSet m_top_level_domains_excludes;
    std::list<TString> http_ident_list;
    std::array<ui8, 256> m_symbol_table;
    const NTld::TDetector TldDetector;

    void constructor_base(TSymbEncodeType setype);
    void InitTable(TSymbEncodeType setype);
    bool IsAllowSymbolHost(char symb) const;
    bool IsAllowSymbolMail(char symb) const;
    bool IsAllowSymbolUrl(char symb) const;
    bool IsAllowSymbolDigist(char symb) const;
    ui8 GetSymbolPriznak(char symb) const;
    TString NormalizeUrl(const TStringBuf& url) const;
    TString NormalizeMail(const TStringBuf& mail) const;
    int GetCommonUrl(const char* pbuf, int buflen, int cur, int* pnot_url, TString& url, bool full_url) const;
    int GetCommonUrl(const char* pbuf, int buflen, int cur, int* pnot_url, ui8& is_http, TString& url_s, TString& host_s, int& start_url) const;
    void ParseHostWithHttp(const char* BUFF, int BUFFSIZE, TKUrlList& urlslist, ui32& urlmailcount, size_t max_element);
    TKUrlItem ParseHostWithHttpItem(const char* BUFF, int BUFFSIZE, int* pnot_url);
    bool IsAllowZone(const TStringBuf& domen) const;
    bool CheckZone(const TStringBuf& domen, const TZoneDetectorSet& domains) const;
    bool CheckExcludes(const TStringBuf& url, const TStringBuf& host) const;

    TString Recode(TStringBuf src) const;
    bool InTld(TStringBuf src) const;

public:
    bool CheckHost(const TString& host) const;

public:
    TZoneDetector();
    TZoneDetector(TSymbEncodeType setype);

    //инициализация класса (установка типа кодировки символов, по-умолчанию win1251)
    void Init(TSymbEncodeType setype);

    //возвращает домен уровня на 1 больше common zone
    TStringBuf GetCommonZone(TStringBuf url) const;
    TStringBuf GetCommonZoneFromHost(const TStringBuf& host) const;
    TStringBuf GetCommonZoneFromAddr(const TStringBuf& sAddr) const;

    //проверяет url на соотвествие основной(common) зоне  (use alex@)
    bool CheckCommonZone(TStringBuf url) const;
    //Параметры:
    //url - указатель на url (строка с завершающим 0)
    //Возвращаемое значение:
    //true - url является common zone
    //false - url не является common zone

    //выделяет url(mail) из текста, заданного в буфере pbuf размером buflen (url - длинные, с параметрами)   (use alex@)
    int getcommonurl(const char* pbuf, int buflen, int cur, int* pnot_url, TString& targetUrl) const;
    //Параметры:
    //pbuf      - указатель на буфер с текстом
    //buflen    - длина буфера pbuf
    //cur       - текущая позиция в буфере pbuf
    //pnot_url  - указатель на смещение в буфере pbuf, начиная с которого можно искать следующий url(mail)
    //purl      - указатель на url(mail)
    //url_len   - указатель на длину урла
    //Возвращаемые значения:
    //0         - ничего не найдено
    //1         - найден url
    //2         - найден email

    //выделяет host(mail) из текста, заданного в буфере pbuf размером buflen (host - только хост из url, без параметров)
    int getcommonhost(const char *pbuf, int buflen, int cur, int *pnot_url, TString& targetUrl) const;
    //Параметры:
    //pbuf      - указатель на буфер с текстом
    //buflen    - длина буфера pbuf
    //cur       - текущая позиция в буфере pbuf
    //pnot_url  - указатель на смещение в буфере pbuf, начиная с которого можно искать следующий url(mail)
    //purl      - указатель на url(mail)
    //url_len   - указатель на длину урла
    //Возвращаемые значения:
    //0         - ничего не найдено
    //1         - найден url
    //2         - найден email

    //выделяет url(mail) из текста, заданного в буфере pbuf размером buflen
    int GetEmailsUrls(const char *pbuf, int buflen, TKUrlList &urlslist, TStrokaList &mailslist);
    //Параметры:
    //pbuf      - указатель на буфер с текстом
    //buflen    - длина буфера pbuf
    //urlslist  - список, в который помещаются все найденные url (в списке могут быть повторяющиеся url)
    //mailslist - список, в который помещаются все найденные email (в списке могут быть повторяющиеся email)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email

    //выделяет url(mail) из текста, заданного в буфере pbuf размером buflen, ограниченное число урлов и майл
    int GetEmailsUrlsLimited(const char *pbuf, int buflen, TKUrlList &urlslist, TStrokaList &mailslist, size_t max_element);
    //Параметры:
    //pbuf        - указатель на буфер с текстом
    //buflen      - длина буфера pbuf
    //urlslist    - список, в который помещаются все найденные url (в списке могут быть повторяющиеся url)
    //mailslist   - список, в который помещаются все найденные email (в списке могут быть повторяющиеся email)
    //max_element - ограничение на кол-во урлов и мыло (общее для них)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email

    //выделяет url(mail) из текста, заданного в буфере pbuf размером buflen. Возвращает только уникальные url(email)
    int GetEmailsUrlsUniq(const char *pbuf, int buflen, bool full_url, TStrokaHash &urlshash, TStrokaHash &mailshash);
    //Параметры:
    //pbuf      - указатель на буфер с текстом
    //buflen    - длина буфера pbuf
    //urlslist  - список, в который помещаются все найденные url (в списке только уникальные url)
    //mailslist - список, в который помещаются все найденные email (в списке только уникальные email)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email

    //выделяет url(mail) из текста (ограниченное кол-во = max_element), заданного в буфере pbuf размером buflen. Возвращает только уникальные url(email)
    int GetEmailsUrlsUniqLimited(const char *pbuf, int buflen, bool full_url, TStrokaHash &urlshash, TStrokaHash &mailshash, TKUrlHash &urlhash_t, int max_element);
    //Параметры:
    //pbuf        - указатель на буфер с текстом
    //buflen      - длина буфера pbuf
    //urlslist    - список, в который помещаются все найденные url (в списке только уникальные url)
    //mailslist   - список, в который помещаются все найденные email (в списке только уникальные email)
    //max_element - ограничение на кол-во урлов и мыло (общее для них)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email

    int GetEmailsUrlsUniqFull(const char *pbuf, int buflen, TKUrlHash &urlshash, TStrokaHash &mailshash);
    //Параметры:
    //pbuf      - указатель на буфер с текстом
    //buflen    - длина буфера pbuf
    //urlslist  - список, в который помещаются все найденные url (в списке только уникальные url)
    //mailslist - список, в который помещаются все найденные email (в списке только уникальные email)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email

    //выделяет url(mail) из текста (ограниченное кол-во = max_element), заданного в буфере pbuf размером buflen. Возвращает только уникальные url(email)
    int GetEmailsUrlsUniqLimitedFull(const char *pbuf, int buflen, TKUrlHash &urlshash, TStrokaHash &mailshash, int max_element);
    //Параметры:
    //pbuf        - указатель на буфер с текстом
    //buflen      - длина буфера pbuf
    //urlslist    - список, в который помещаются все найденные url (в списке только уникальные url)
    //mailslist   - список, в который помещаются все найденные email (в списке только уникальные email)
    //max_element - ограничение на кол-во урлов и мыло (общее для них)
    //Возвращаемое значение - возвращает сумму найденных (находящихся в urlslist и mailslist) url и email
};

#endif
