/*
 * AnyValue_v2_ut.cpp
 *
 *  Created on: 27 мар. 2017 г.
 *      Author: luckybug
 */

#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/buffer.h>
#include <util/generic/buffer.h>
#include <util/datetime/base.h>
#include <library/cpp/json/json_reader.h>

#include "AnyValue_v2.h"

Y_UNIT_TEST_SUITE(Oid) {
    Y_UNIT_TEST(Main) {
        {
            NAnyValue::TOid oid;
            UNIT_ASSERT_EQUAL_C(oid.get(), "0", oid.get());
        }
        {
            NAnyValue::TOid oid(42);
            UNIT_ASSERT_EQUAL_C(oid.get(), "0000000000000002A", oid.get());
        }
        {
            NAnyValue::TOid oid("some oid");
            UNIT_ASSERT_EQUAL_C(oid.get(), "some oid", oid.get());
        }
        {
            NAnyValue::TOid oid1(42);
            NAnyValue::TOid oid2("0000000000000002A");
            UNIT_ASSERT_EQUAL(oid1, oid2);
        }
    }
}

Y_UNIT_TEST_SUITE(Scalar) {
    Y_UNIT_TEST(Type) {

        NAnyValue::TScalar s1,s2;
        {
            NAnyValue::TScalar s;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Notype);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT(s.IsNull());

            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);

            UNIT_ASSERT_EQUAL_C(s.Hash(), 0, s.Hash());

            UNIT_ASSERT_EQUAL_C(s.Hash(), 0, s.Hash());

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 0);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), false);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{});
        }
        {
            NAnyValue::TScalar s = ui64(123);
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Ui64);

            s1 = ui64(123);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<ui64>(), 123);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<ui64>()(123));


            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{123});
        }
        {
            NAnyValue::TScalar s = i64(123);
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::I64);

            s1 = i64(123);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<i64>(), 123);
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<i64>()(123));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{123});
        }
        {
            NAnyValue::TScalar s = ui32(123);
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Ui32);

            s1 = ui32(123);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<ui32>(), 123);
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<ui32>()(123));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{123});
        }
        {
            NAnyValue::TScalar s = ui16(123);
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Ui16);

            s1 = ui16(123);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<ui16>(), 123);
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<ui16>()(123));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{123});
        }
        {
            NAnyValue::TScalar s = double(123);
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Double);

            s1 = double(123);
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<double>(), 123);
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<double>()(123));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{123});
        }
        {
            NAnyValue::TScalar s = TString("123");
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::String);

            s1 = TString("123");
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<TString>(), "123");
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<TString>()("123"));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{"123"});
        }
        {
            NAnyValue::TScalar s = false;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Bool);

            s1 = false;
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<bool>(), false);
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<NAnyValue::TOid>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<bool>()(false));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 0);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 0);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "0", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), false);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid{0}, s.GetRobust<NAnyValue::TOid>().get());
        }
        {
            NAnyValue::TScalar s = NAnyValue::TOid("123");
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TScalarType::Oid);

            s1 = NAnyValue::TOid("123");
            s2 = s;

            UNIT_ASSERT_EQUAL(s1, s2);
            UNIT_ASSERT_EQUAL(s1, s);

            UNIT_ASSERT_EQUAL(s.As<NAnyValue::TOid>(), NAnyValue::TOid("123"));
            UNIT_ASSERT_EXCEPTION(s.As<ui64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<i64>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui32>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<ui16>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<double>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<TString>(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.As<bool>(), std::exception);

            UNIT_ASSERT_EQUAL(s.Hash(), THash<NAnyValue::TOid>()(NAnyValue::TOid("123")));

            UNIT_ASSERT_EQUAL(s.GetRobust<ui64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<i64>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui32>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<ui16>(), 123);
            UNIT_ASSERT_EQUAL(s.GetRobust<double>(), 123);
            UNIT_ASSERT_EQUAL_C(s.GetRobust<TString>(), "123", s.GetRobust<TString>());
            UNIT_ASSERT_EQUAL(s.GetRobust<bool>(), true);
            UNIT_ASSERT_EQUAL(s.GetRobust<NAnyValue::TOid>(), NAnyValue::TOid("123"));
        }
    }
    Y_UNIT_TEST(Json) {
        {
            NAnyValue::TScalar s = ui64(123);
            TStringStream stream;

            NJson::TJsonWriter writer(&stream, false);


            s.Write(writer);
            writer.Flush();

            UNIT_ASSERT_EQUAL_C(stream.Str(), "123", stream.Str());
        }
        {
            NAnyValue::TScalar s = TString("abc");
            TStringStream stream;

            NJson::TJsonWriter writer(&stream, false);


            s.Write(writer);
            writer.Flush();

            for(char c : stream.Str())
                Cout << c << Endl;

            UNIT_ASSERT_EQUAL_C(stream.Str(), "\"abc\"", stream.Str());
        }
        {
            NAnyValue::TScalar s = NAnyValue::TOid(42);
            TStringStream stream;

            NJson::TJsonWriter writer(&stream, false);


            s.Write(writer);
            writer.Flush();

            UNIT_ASSERT_EQUAL_C(stream.Str(), TString("\"0000000000000002A\""), stream.Str());
        }
        {
            NAnyValue::TScalar s;
            TStringStream stream;

            NJson::TJsonWriter writer(&stream, false);


            s.Write(writer);
            writer.Flush();

            UNIT_ASSERT_EQUAL_C(stream.Str(), "null", stream.Str());
        }
        {
            NAnyValue::TScalar s = ui64(123);
            TStringStream stream;

            NJson::TJsonWriter writer(&stream, false);

            writer.OpenMap();

            s.Write(writer, "key");

            writer.CloseMap();
            writer.Flush();
            UNIT_ASSERT_EQUAL_C(stream.Str(), "{\"key\":123}", ":" << stream.Str());
        }
    }
}

Y_UNIT_TEST_SUITE(AnyValueV2) {

    Y_UNIT_TEST(Main) {
        {
            NAnyValue::TAnyValue s;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Notype);

            UNIT_ASSERT(s.IsNull());
            UNIT_ASSERT(!s.IsMap());
            UNIT_ASSERT(!s.IsArray());
            UNIT_ASSERT(!s.IsScalar());

            UNIT_ASSERT_EXCEPTION(s.AsArray(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.AsMap(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.AsScalar(), std::exception);
        }
        {
            const NAnyValue::TMap expected = {
                    {TString("key"), ui64(42)}
            };
            NAnyValue::TAnyValue s = expected;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Map);

            UNIT_ASSERT(!s.IsNull());
            UNIT_ASSERT(s.IsMap());
            UNIT_ASSERT(!s.IsArray());
            UNIT_ASSERT(!s.IsScalar());


            UNIT_ASSERT_EXCEPTION(s.AsArray(), std::exception);
            UNIT_ASSERT_EQUAL(s.AsMap(), expected);
            UNIT_ASSERT_EXCEPTION(s.AsScalar(), std::exception);
        }
        {
            const NAnyValue::TArray expected = {
                    {ui64(42), TString{"42"}}
            };
            NAnyValue::TAnyValue s = expected;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Array);

            UNIT_ASSERT(!s.IsNull());
            UNIT_ASSERT(!s.IsMap());
            UNIT_ASSERT(s.IsArray());
            UNIT_ASSERT(!s.IsScalar());


            UNIT_ASSERT_EQUAL(s.AsArray(), expected);
            UNIT_ASSERT_EXCEPTION(s.AsMap(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.AsScalar(), std::exception);
        }
        {
            const NAnyValue::TScalar expected = ui64(42);

            NAnyValue::TAnyValue s = expected;
            UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Scalar);

            UNIT_ASSERT(!s.IsNull());
            UNIT_ASSERT(!s.IsMap());
            UNIT_ASSERT(!s.IsArray());
            UNIT_ASSERT(s.IsScalar());


            UNIT_ASSERT_EXCEPTION(s.AsArray(), std::exception);
            UNIT_ASSERT_EXCEPTION(s.AsMap(), std::exception);
            UNIT_ASSERT_EQUAL(s.AsScalar(), expected);
        }
    }

    Y_UNIT_TEST(SetType) {
        NAnyValue::TAnyValue s;

        s.SetType(NAnyValue::TType::Notype);
        UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Notype);

        s.SetType(NAnyValue::TType::Map);
        UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Map);
        UNIT_ASSERT_EQUAL(s.AsMap(), NAnyValue::TMap{});

        s.SetType(NAnyValue::TType::Array);
        UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Array);
        UNIT_ASSERT_EQUAL(s.AsArray(), NAnyValue::TArray{});

        s.SetType(NAnyValue::TType::Scalar);
        UNIT_ASSERT_EQUAL(s.GetType(), NAnyValue::TType::Scalar);
        UNIT_ASSERT_EQUAL(s.AsScalar(), NAnyValue::TScalar{});
    }

    Y_UNIT_TEST(ScalarMap) {
        NAnyValue::TAnyValue s;
        UNIT_ASSERT_EQUAL(s.ToScalarMap(), NAnyValue::TScalarMap{});

        s = ui64(42);
        UNIT_ASSERT_EQUAL(s.ToScalarMap(), NAnyValue::TScalarMap{});

        s = NAnyValue::TArray{ui64(42)};
        UNIT_ASSERT_EQUAL(s.ToScalarMap(), NAnyValue::TScalarMap{});

        s = NAnyValue::TMap{
                {TString{"scalar"}, TString{"42"}},
                {TString{"array"},  NAnyValue::TArray{ui64(42)}},
                {TString{"map"},    NAnyValue::TMap{
                        {TString{"scalar"}, TString{"42"}}
                }}
        };

        const NAnyValue::TScalarMap expected = {
                {TString{"scalar"}, TString{"42"}},
        };
        UNIT_ASSERT_EQUAL(s.ToScalarMap(), expected);
    }

    Y_UNIT_TEST(Hash) {
        {
            const auto s = NAnyValue::TScalarMap{
                    {TString{"scalar"}, TString{"42"}},
            };

            size_t expected = CombineHashes(ui64{}, THash<TString>()("scalar"));
            expected = CombineHashes(expected, THash<TString>()("42"));
            UNIT_ASSERT_EQUAL(NAnyValue::Hash(s), expected);

        }
        {
            const auto s = NAnyValue::TScalarVector{
                    TString{"scalar"}, ui64{42}, 2.0
            };

            size_t expected = CombineHashes(ui64{}, THash<TString>()("scalar"));
            expected = CombineHashes(expected, THash<ui64>()(42));
            expected = CombineHashes(expected, THash<double>()(2.0));
            UNIT_ASSERT_EQUAL(NAnyValue::Hash(s), expected);
        }
        {
            NAnyValue::TScalarVectorMap s;

            s["vector"].emplace_back(ui64{42});

            size_t expected = CombineHashes(ui64{}, THash<TString>()("vector"));
            expected = CombineHashes(expected, THash<ui64>()(42));
            expected = CombineHashes(ui64{}, expected);
            UNIT_ASSERT_EQUAL(NAnyValue::Hash(s), expected);
        }
    }

//    bool OnNull() final;
//    bool OnBoolean(bool v) final;
//    bool OnInteger(long long v) final;
//    bool OnUInteger(unsigned long long v) final;
//    bool OnDouble(double v) final;
//    bool OnString(const TStringBuf &v) final;
//    bool OnStringNoCopy(const TStringBuf &v) final;
//    bool OnOpenMap() final;
//    bool OnMapKey(const TStringBuf &v) final;
//    bool OnCloseMap() final;
//    bool OnOpenArray() final;
//    bool OnCloseArray() final;
//    bool OnMapKeyNoCopy(const TStringBuf &v) final;

    Y_UNIT_TEST(WriteParse) {
        const TString mock = R"JSON({"uint":9223372036854776001,"map":{"array":[1,2,"text"]},"bool":true,"int":42,"null":null,"double":2.1,"string":"text"})JSON";

        NAnyValue::TAnyValue parsed;
        {
            TStringInput stream(mock);
            NAnyValue::TAnyValueParser parser(parsed);
            UNIT_ASSERT(NJson::ReadJson(&stream, &parser));
        }

        UNIT_ASSERT(parsed.IsMap());
        auto m = parsed.AsMap();

        UNIT_ASSERT_EQUAL_C(m.size(), 7, m.size());
        UNIT_ASSERT(m.contains("null"));
        UNIT_ASSERT(m["null"].AsScalar().IsNull());

        UNIT_ASSERT(m.contains("bool"));
        UNIT_ASSERT_EQUAL(m["bool"].AsScalar().As<bool>(), true);

        UNIT_ASSERT(m.contains("int"));
        UNIT_ASSERT_EQUAL(m["int"].AsScalar().As<i64>(), 42);

        UNIT_ASSERT(m.contains("uint"));
        UNIT_ASSERT_EQUAL(m["uint"].AsScalar().As<ui64>(), 9223372036854776001ull);

        UNIT_ASSERT(m.contains("double"));
        UNIT_ASSERT_EQUAL(m["double"].AsScalar().As<double>(), 2.1);

        UNIT_ASSERT(m.contains("string"));
        UNIT_ASSERT_EQUAL(m["string"].AsScalar().As<TString>(), "text");

        UNIT_ASSERT(m.contains("map"));

        UNIT_ASSERT(m["map"].IsMap());
        UNIT_ASSERT(m["map"].AsMap().contains("array"));
        UNIT_ASSERT_EQUAL(m["map"].AsMap().size(), 1);
        UNIT_ASSERT(m["map"].AsMap()["array"].IsArray());

        const auto & arr = m["map"].AsMap()["array"].AsArray();

        UNIT_ASSERT_EQUAL(arr.size(), 3);

        UNIT_ASSERT_EQUAL(arr[0].AsScalar().AsI64(), 1);
        UNIT_ASSERT_EQUAL(arr[1].AsScalar().AsI64(), 2);
        UNIT_ASSERT_EQUAL(arr[2].AsScalar().AsString(), "text");

        TString result;
        {
            TStringOutput stream(result);
            NAnyValue::TWriter writer(&stream, false, true);

            writer.Write(parsed);
        }

        NAnyValue::TAnyValue parsed2;
        {
            TStringInput stream(result);
            NAnyValue::TAnyValueParser parser(parsed2);
            UNIT_ASSERT(NJson::ReadJson(&stream, &parser));
        }

        UNIT_ASSERT_EQUAL(parsed, parsed2);
    }

}
