/*
 * ipv6_ut.cpp
 *
 *  Created on: 14 авг. 2017 г.
 *      Author: luckybug
 */

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/strbuf.h>
#include "ipv6.h"
#include "tkipv6.h"

Y_UNIT_TEST_SUITE(ip) {
    Y_UNIT_TEST(Valid) {
        UNIT_ASSERT_EQUAL(TIpAddr("0xC0.0x00.0x02.0xEB").IsValid(), true);

        UNIT_ASSERT_EQUAL(TIpAddr("3221226219").IsValid(), true);
        UNIT_ASSERT_EQUAL(TIpAddr("030000001353").IsValid(), true);
        UNIT_ASSERT_EQUAL(TIpAddr("0xC00002EB").IsValid(), true);
        UNIT_ASSERT_EQUAL(TIpAddr("0300.0000.0002.0353").IsValid(), true);
        UNIT_ASSERT_EQUAL(TIpAddr("192.168.1.1"), TIpAddr("192.168.1.1"));
        UNIT_ASSERT_EQUAL(TIpAddr("192.168.1.1").ToString(), "192.168.1.1");
        UNIT_ASSERT_EQUAL(TIpAddr("192.168.1.1").IsValid(), true);
        UNIT_ASSERT_EQUAL(!TIpAddr("892.168.1.1").IsValid(), true);
        UNIT_ASSERT_UNEQUAL(TIpAddr("192.168.1.0"), TIpAddr("192.168.1.1"));
        UNIT_ASSERT_EQUAL(TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233"), TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233"));
        UNIT_ASSERT_UNEQUAL(TIpAddr("0000:0011:2233:4455:6677:8899:0011:2233"), TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233"));
        UNIT_ASSERT_EQUAL(TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233").IsValid(), true);
        UNIT_ASSERT_EQUAL(!TIpAddr("fffff:0011:2233:4455:6677:8899:0011:2233").IsValid(), true);
        UNIT_ASSERT_EQUAL(!TIpAddr("ffff:z011:2233:4455:6677:8899:0011:2233").IsValid(), true);
        UNIT_ASSERT_UNEQUAL(TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233"), TIpAddr("123.22.33.45"));
        UNIT_ASSERT_UNEQUAL(TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233"), TIpAddr("523.22.33.45"));
        UNIT_ASSERT_EQUAL((TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233") & TIpAddr("::")), TIpAddr("::"));
        UNIT_ASSERT_EQUAL((TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233") & TIpAddr("ffff:0000:0000:ffff:ffff:0000:00ff:000f")), TIpAddr("ffff:0000:0000:4455:6677:0000:0011:0003"));
        UNIT_ASSERT_EQUAL((TIpAddr("ffff:0011:2233:4455:6677:8899:0011:2233") | TIpAddr("ffff:0000:0000:ffff:ffff:0000:00ff:000f")), TIpAddr("ffff:0011:2233:ffff:ffff:8899:00ff:223f"));
        UNIT_ASSERT_EQUAL(TIpAddr("0000:0000:0000:0000:0000:ffff:187.250.251.113").IsValid(), true);
        UNIT_ASSERT_EQUAL(TIpAddr("0123:4567:89ab:cdef:0000:ffff:0000:1122").ToDNSBLString(), "2.2.1.1.0.0.0.0.f.f.f.f.0.0.0.0.f.e.d.c.b.a.9.8.7.6.5.4.3.2.1.0");
        UNIT_ASSERT_EQUAL(TIpAddr("192.168.0.1").ToDNSBLString(), "1.0.168.192");
        UNIT_ASSERT_EQUAL(TIpAddr("292.168.0.1").ToDNSBLString(), "");
        UNIT_ASSERT_EQUAL(TIpAddr("__23:4567:89ab:cdef:0000:ffff:0000:1122").ToDNSBLString(), "");

        for (int i = 0; i < 10000; ++i) {
            TIpAddr("0000:0000:0000:0000:0000:ffff:187.250.251.113").ToString();
            TIpAddr ipa;
            TIpAddr ipa2;
            ipa2.FromString("ff10:0000:1111:0000:2233:8899:1112:1234");
            ipa.ToString();
            TIpAddr("0123:4567:89ab:cdef:0000:ffff:0000:1122").ToDNSBLString();
            ipa.FromString("192.168.1.1");
            ipa.ToString();
            ipa.FromString("292.168.1.1");
            ipa.ToString();
            ipa.FromString("ff11:0000:1111:0000:2233:8899:1112:1234");
            ipa.ToString();
            // (ipa == ipa2);
            (ipa & ipa2).ToString();
            ipa2.FromString("ffff:ffff:0000:ffff:0000:00ff:ff00:f00f");
            (ipa & ipa2).ToString();
            ipa.FromString("192.168.1.1");
            ipa2.FromString("255.255.255.0");
            (ipa & ipa2).ToString();
        }
    }

    Y_UNIT_TEST(MasksV6) {
        TIpAddr add;
        add.MakeMask(0, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "::");
        add.MakeMask(1, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "8000::");
        add.MakeMask(2, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "c000::");
        add.MakeMask(4, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "f000::");
        add.MakeMask(8, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ff00::");
        add.MakeMask(16, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff::");
        add.MakeMask(17, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:8000::");
        add.MakeMask(24, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ff00::");
        add.MakeMask(25, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ff80::");
        add.MakeMask(26, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffc0::");
        add.MakeMask(32, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff::");
        add.MakeMask(33, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:8000::");
        add.MakeMask(64, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff::");
        add.MakeMask(65, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff:8000::");
        add.MakeMask(72, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff:ff00::");
        add.MakeMask(73, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff:ff80::");
        add.MakeMask(88, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff:ffff:ff00::");
        add.MakeMask(128, true);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff");
    }

    Y_UNIT_TEST(MasksV4) {
        TIpAddr add;

        add.MakeMask(0, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "0.0.0.0");
        add.MakeMask(1, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "128.0.0.0");
        add.MakeMask(2, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "192.0.0.0");
        add.MakeMask(4, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "240.0.0.0");
        add.MakeMask(8, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "255.0.0.0");
        add.MakeMask(16, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "255.255.0.0");
        add.MakeMask(24, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "255.255.255.0");
        add.MakeMask(31, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "255.255.255.254");
        add.MakeMask(32, false);
        UNIT_ASSERT_STRINGS_EQUAL(add.ToString(), "255.255.255.255");
    }

    Y_UNIT_TEST(TKIPv6) {
        TIpAddr v4("127.0.0.3");
        TIpAddr v6("0123:4567:89ab:cdef:0000:ffff:0000:1122");

        TKIPv6 tv4("127.0.0.3");
        TKIPv6 tv6("0123:4567:89ab:cdef:0000:ffff:0000:1122");

        Y_DECLARE_UNUSED auto v = v4.ToTKIPv6();
        UNIT_ASSERT_EQUAL(v4.ToTKIPv6(), tv4);
        UNIT_ASSERT_EQUAL(v6.ToTKIPv6(), tv6);
    }
}
