#include "phone_parser.h"
#include <iostream>
#include <util/system/file.h>
#include <library/cpp/charset/recyr.hh>
#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(PhoneParser) {
    Y_UNIT_TEST(Main) {
        const TString text = Recode(CODES_UTF8, CODES_KOI8,
            "бла бла бла ((4:9*5)) 7*2-2^1~9:3~9 - +7+7+7 926 892 64 22\n" //4957221939 79268926422
            "--Ч_9 5-- 7 Ч ... I I -- 7 87\n" //4957411787
            "ла ля ля плюс 7 ((9~1*6)) 0*шесть^2^1~9:два~9 - + 8 + 926 + 791 + 64 + 22 +\n" //79160621929 89267916422
            "тыр тыр пыр +7 < 9 o З } 1 O 5 бб 17\n" //79031056617
            "Здравствуйте, . \nпродам \nРЭС 90 030 87г 35 шт \nРЭС 90 000 92г 200 шт\nпкн117-1зв 92г 10 шт\nпкн117-1бв 40 шт\n2рмгп22б4ш3е2 56 шт\nм\nс импорт IRIS 4015 200 шт\n\nудалить адрес nikksim@mail.ru - не работает , заменить на niksim@infos.ru\nв рассылку по радиокомпонентам прошу внести адрес niksim@infos.ru \n\nКому не по теме, приношу извинения -- DEL в теме, и я Вас больше не побеспокою .\n\nС уважением , Николай \nтел 8 -921- 9000-316\n" //89219000316
            "Сдается в аренду офис\nПлощадь:\t2O0 кв.м.\nРасположение:\tна первом этаже отдельон стоящего особняка.\nОтделка:\tХороший евроремонт.\nПарковка:\t3 машиноместа.\nПрямая аренда без посредников.\nСобственник:\t8 9--1--6 2--9--9--6--8--36\n" //89162996836
            "ВСТУПЛЕНИЕ В СРО\n\nДистанционное обучение\n(в т.ч. без контакта с образовательной структурой)\n\nПовышение квалификации продолжительностью\n72 часа для вступления в строительные СРО.\n\n\nОбучение проводится на выбор по более чем 30\nтемам, которые могут быть уточнены с заказчиком.\n\nСтоимость с выдачей удостоверения\n5 000 рублей за одного человека.\n\nтел. 8 (499) 187:9О.11\n\n" //84991879011
            "$650,500.00USD have been awarded to you by UN pin # UNO-361/EH6/2010\n" //none
            "СТУПЛЕНИЕ В СРО\n\nДистанционное обучение\n(в т.ч. без контакта с образовательной структурой)\n\nПовышение квалификации продолжительностью\n72 часа для вступления в строительные СРО.\n\nОбучение проводится на выбор по более чем 30\nтемам, которые могут быть уточнены с заказчиком.\n\nСтоимость с выдачей удостоверения\n5 000 рублей за одного человека.\n\nтел. 8 (499) 187~9О_11\n++\n" //84991879011
            "Адрес:\tг. Москва, Волоколамское ш., д. 5\nРасположение:\tна 1-м этаже отдельно стоящего офисного особняка.\nОтделка:\tХороший навый евроремонт.\nПарковка:\t3 меставо дфоре.\nАренда без посредников.\nСобственник:\t4^^9^^9 1^^5^^8^^02^^03\n\n" //4991580203
            "Сдается в аренду офис\n\nОбщая арендуемая площадь:\t20O кв.м.\nПарковка:\t3 машиноместаво дворе.\nПрямая аренда без пасредников.\nСобственник:\t4-99 1-5-8-02-03\n\n" //4991580203
            "Обучение проводится на выбор по более чем 30\nтемам, которые могут быть уточнены с заказчиком.\n\nСтоимость с выдачей удостоверения\n5 000 рублей за одного человека.\n\nтел. 8 (499) 187`9О_11\n$$\n" //84991879011
            "Sending почты четко и своевременно. Следование порядков почтовых paccылoк. Уникальное реекламное предложениее! Подробнее по телефону плюс7-9O4-6I3 54-6шесть\n" //79046135466
            "Здравствуйте!\n\nВозврат долгов, поиск кидал. Беремся даже за самые тяжелые случаи (например, кидалово при обналичке). Минимальные цены. Законные методы (обходимся без бит).\nЗвоните +7.-.9_2_1.-.З_З_4.-.6_1.-.9_З\n\nДо свидания! Александр.\n" //79213346193
            "Помощь в получении кредита юридическим лйцом\n(при заключенном договоре о бухгалтерском обслуживании)\n\nЗвонить: 5-O-5-69-7\n" //505697
            "\"Банкет-кафе 51\" на крыше приглашает провести свадьбу, день рождение, любой праздник.\n\nНеограниченное время работы, неограниченная громкость музыки, алкоголь полностью Ваш.\nВкусное индивидуальное меню, професиональный свет и звук, шикарный вид на Екатеринбург с высоты птичьего полета.\nКафе работает только для Вас и Вашей компании.\n\nВосточная 51. 9 этаж. тел: 268955O. \n" //2689550
            "Начался прием заявок от образовательных учреждений на участие в\nТретьем Открытом Чемпионате России по универсальному марафону.\n\nВпервые наравне с учащимися за собственные коллективы будут выступать взрослые - педагоги,\nвоспитатели, сотрудники администраций образовательных учреждений РФ и стран Содружества.\nЗачеты проводятся как по интеллектуальной, так и по физической подготовке.\n\nПоложение и регистрационная форма - http://files.mail.ru/4W5LVR.\n\n\nОргкомитет: (495) 979-96-19, 979-93-90, 8 (901) 524-03-50\n" //4959799619 9799390 89015240350
            "Мы продаем мега-фиговину CMPSB-12345-7 +7 (901) 524-03-50\n" //79015240350
            "Мы продаем мега-фиговину CMPSB-12345-7 +380 (901) 524-030 dop1 2 3 4\n" //380901524030
            "Мы продаем мега-фиговину CMPSB-12345  812 524-03-00 dop1 2 3 4\n" //8125240300
            "звоните нам из заграницы, мы крутые вааще застрелицца CMPSB-12345  007-812-524-03-00 dop1 2 3 4\n" //0078125240300
            "Бросил любимый человек? Страдаете от неразделенной любви?\nТеперь у вас есть возможность восстановить потерянные отношения, вернуть любимого человека и привязать к себе на долгое время \nс помощью авторского метода воздействия. \nРезультат сразу после проведения работы.Именно этот метод является самым эффективным и реально действующим на сегодняшний день.\nПодробная информация и запись по (495) 740 2820. 7398816426717.5341770272095326 1577484034948.2590403594602830\n" //4957402820
            "тел [ ч 9 5 ] 9 пять 6 семь О 3 о\nОперативно ответим на любой Ваш запрос!\n" //4959567030
            "phon: 495> два два семь _ шесть ноль _ девять пять\n" //4952276095
        );

        TPhoneParser parser;
        TPhoneParser::TParseResults results = parser.GetPhones(text);

        UNIT_ASSERT_EQUAL(results.size(), 28);
        UNIT_ASSERT_EQUAL(results[0].GetPhone(), "4957221939");
        UNIT_ASSERT_EQUAL(results[1].GetPhone(), "79268926422");
        UNIT_ASSERT_EQUAL(results[2].GetPhone(), "4957411787");
        UNIT_ASSERT_EQUAL(results[3].GetPhone(), "79160621929");
        UNIT_ASSERT_EQUAL(results[4].GetPhone(), "89267916422");
        UNIT_ASSERT_EQUAL(results[5].GetPhone(), "79031056617");
        UNIT_ASSERT_EQUAL(results[6].GetPhone(), "89219000316");
        UNIT_ASSERT_EQUAL(results[7].GetPhone(), "89162996836");
        UNIT_ASSERT_EQUAL(results[8].GetPhone(), "84991879011");
        UNIT_ASSERT_EQUAL(results[9].GetPhone(), "650500"); //wrong detect
        UNIT_ASSERT_EQUAL(results[10].GetPhone(), "84991879011");
        UNIT_ASSERT_EQUAL(results[11].GetPhone(), "4991580203");
        UNIT_ASSERT_EQUAL(results[12].GetPhone(), "4991580203");
        UNIT_ASSERT_EQUAL(results[13].GetPhone(), "84991879011");
        UNIT_ASSERT_EQUAL(results[14].GetPhone(), "79046135466");
        UNIT_ASSERT_EQUAL(results[15].GetPhone(), "79213346193");
        UNIT_ASSERT_EQUAL(results[16].GetPhone(), "505697");
        UNIT_ASSERT_EQUAL(results[17].GetPhone(), "2689550");
        UNIT_ASSERT_EQUAL(results[18].GetPhone(), "4959799619");
        UNIT_ASSERT_EQUAL(results[19].GetPhone(), "9799390");
        UNIT_ASSERT_EQUAL(results[20].GetPhone(), "89015240350");
        UNIT_ASSERT_EQUAL(results[21].GetPhone(), "79015240350");
        UNIT_ASSERT_EQUAL(results[22].GetPhone(), "380901524030");
        UNIT_ASSERT_EQUAL(results[23].GetPhone(), "8125240300");
        UNIT_ASSERT_EQUAL(results[24].GetPhone(), "0078125240300");
        UNIT_ASSERT_EQUAL(results[25].GetPhone(), "4957402820");
        UNIT_ASSERT_EQUAL(results[26].GetPhone(), "4959567030");
        UNIT_ASSERT_EQUAL(results[27].GetPhone(), "4952276095");
    }

    Y_UNIT_TEST(Null) {
        const TString text = Recode(CODES_UTF8, CODES_KOI8, "$650,500.00USD have been awarded to you by UN pin # UNO-361/EH6/2010"); //none

        TPhoneParser parser;
        TPhoneParser::TParseResults results = parser.GetPhones(text);

        UNIT_ASSERT_EQUAL(results.size(), 0);
    }

    Y_UNIT_TEST(WrongDetect) {
        const TString text = Recode(CODES_UTF8, CODES_KOI8, "тел. 8 (499) 187:9О.11\n\n$650,500.00USD have been awarded to you by UN pin # UNO-361/EH6/2010"); //none

        TPhoneParser parser;
        TPhoneParser::TParseResults results = parser.GetPhones(text);

        UNIT_ASSERT_EQUAL(results.size(), 2);
        UNIT_ASSERT_EQUAL(results[0].GetPhone(), "84991879011");
        UNIT_ASSERT_EQUAL(results[1].GetPhone(), "650500");
    }

    Y_UNIT_TEST(Crash) {
        TString text = "Some";
        text += '\0';
        text += "0123456789asasdssd";

        TPhoneParser parser;
        TPhoneParser::TParseResults results = parser.GetPhones(text);

        UNIT_ASSERT_EQUAL(results.empty(), true);
    }

    Y_UNIT_TEST(GarbageResult) {
        TString text = "-------- Перенаправленное сообщение -------- Тема: Платежное поручение обучение Дата: Wed, 22 Feb 2017 12:41:16 +0300 От: Финансовый менеджер ЦРО <finans_manager@center.lazurit.com> Отвечать: finans_manager@center.lazurit.com Кому: Секретарь ЦРО <office@lip.lazurit.com> -- С Яковлева Светлана Финансовый менеджер Центрального регионального отделения                        Адрес : г. Воронеж, 396313 ул. Индустриальная, 3.       тел: +7 (473) 204 53 77 доб 106 моб: +7 (906) 675 64 47 mail:  finans_manager@center.lazurit.com web: www.lazurit.com                                                                        -- ";
        text += '\0';
        text += "                                                            -- ";
        text += '\0';
        text += " (906) 675 64 47                                                                   mail:  finans_manager@center.lazurit.com web: www.lazurit.com                                                                        -- -------- Перенаправленное сообщение --------\n"
            "Тема: Платежное поручение обучение\n"
            "Дата: Wed, 22 Feb 2017 12:41:16 +0300\n"
            "От: Финансовый менеджер ЦРО <finans_manager@center.lazurit.com>\n"
            "Отвечать: finans_manager@center.lazurit.com\n"
            "Кому: Секретарь ЦРО <office@lip.lazurit.com>\n"
            "--\n"
            "Суважением.\n"
            "_______________________\n"
            "Лого с квадратом2.jpg\n"
            "*Яковлева Светлана *\n"
            "Финансовый менеджер\n"
            "Центрального регионального отделения\n"
            "Адрес : г. Воронеж, 396313 ул. Индустриальная, 3.\n"
            "тел: +7 (473) 204 53 77 доб 106\n"
            "моб: +7 (906) 675 64 47\n"
            "mail: finans_manager@center.lazurit.com\n"
            "web: www.lazurit.com <http://www.lazurit.com>\n"
            "--\n"
            "Суважением.\n"
            "_______________________\n"
            "Лого с квадратом2.jpg\n"
            "*Полякова Оксана Олеговна*\n"
            "Офис-менеджер\n"
            "Центрального регионального отделения\n"
            "Адрес : г. Воронеж, 396313 ул. Индустриальная, 3.\n"
            "тел: +7 (473) 204 53 77 доб. 101\n"
            "моб: +7 (906) 675 63 25\n"
            "mail: office@lip.lazurit.com\n"
            "web: www.lazurit.com <http://www.lazurit.com>\n";

        text = Recode(CODES_UTF8, CODES_KOI8, text);
        int positions[] = { 91, 92, 98, 99, 100, 101, 103, 104, 106, 107, 109, 110, 113, 114, 115, 116, 394, 395, 396, 397, 398, 399, 421, 436, 439, 440, 441, 444, 445, 446, 448, 449, 451, 452, 458, 459, 460, 468, 471, 472, 473, 476, 477, 478, 480, 481,
            483, 484, 690, 691, 692, 695, 696, 697, 699, 700, 702, 703, 998, 999, 1005, 1006, 1007, 1008, 1010, 1011, 1013, 1014, 1016, 1017, 1020, 1021, 1022, 1023, 1233, 1337, 1338, 1339, 1340, 1341, 1342, 1364, 1373, 1376, 1377, 1378, 1381,
            1382, 1383, 1385, 1386, 1388, 1389, 1395, 1396, 1397, 1405, 1408, 1409, 1410 };

        TPhoneParser parser;
        for (int position: positions) {
            TPhoneParser::TParseResult result;
            if (parser.GetPhone((ui8*)text.c_str(), (ui8*)(text.c_str() + position), (ui8*)(text.c_str() + text.size()), result))
                UNIT_ASSERT_UNEQUAL(result.OriginalPhoneStart(), NULL);
        }
    }
}
