#include "tag_sets.h"
#include <util/thread/pool.h>
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <library/cpp/testing/unittest/registar.h>
#include <atomic>

Y_UNIT_TEST_SUITE(TagSets) {
    THashSet<TString> ToSet(const TVector<TString>& v) {
        return THashSet<TString>(v.begin(), v.end());
    }

    Y_UNIT_TEST(Main) {
        using TTestType = NTagSets::TStorage<ui64, TString>;
        TTestType object;
        TTestType::TSet set;

        set.insert(7);
        object.SetData("one", set);
        set.insert(23);
        object.SetData("two", set);
        set.insert(42);
        object.SetData("three", set);

        auto data = ToSet(object.GetTags(42));
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.contains("three"), true);

        data = ToSet(object.GetTags(7));
        UNIT_ASSERT_EQUAL(data.size(), 3);
        UNIT_ASSERT_EQUAL(data.contains("three"), true);
        UNIT_ASSERT_EQUAL(data.contains("two"), true);
        UNIT_ASSERT_EQUAL(data.contains("one"), true);

        object.RemoveData("three");
        data = ToSet(object.GetTags(23));
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.contains("two"), true);

        set.erase(23);
        object.SetData("two", set);
        data = ToSet(object.GetTags(23));
        UNIT_ASSERT_EQUAL(data.size(), 0);
    }

    template<typename T, typename Source = NTagSets::TSource<T, TString>>
    class TStaticSource : public Source {
    public:
        template<typename ... Types>
        TStaticSource(const typename Source::TDestination& destination, const T& v, const TString& tag, const Types& ... args)
            : Source(destination, args ...)
            , tag(tag)
        {
            set.insert(v);
        }

        void Update() final {
            this->SetData(tag, set);
        }

    private:
        THashSet<T> set;
        TString tag;
    };

    Y_UNIT_TEST(Source) {
        NTagSets::TManager<ui64, TString> manager;

        auto data = manager.Get()->GetTags(42);
        UNIT_ASSERT_EQUAL(data.size(), 0);

        manager.CreateSource<TStaticSource<ui64>>(42ULL, "tag")->Update();

        data = manager.Get()->GetTags(42);
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.front(), "tag");
    }

    Y_UNIT_TEST(Schedule) {
        NTagSets::TManager<ui64, TString> manager;
        manager.CreateScheduleSource<TStaticSource<ui64>>(TDuration::MilliSeconds(100), 42ULL, "tag");
        manager.CreateSource<TStaticSource<ui64>>(42ULL, "none");

        TAdaptiveThreadPool threadPool;
        threadPool.Start(0, 0);

        {TSimpleScheduler scheduler(threadPool);
            manager.SetSchedules(scheduler);
            usleep(250*1000);
        }

        auto data = manager.Get()->GetTags(42);
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.front(), "tag");
    }

    Y_UNIT_TEST(Convert) {
        using TRealSource = TStaticSource<TString, NTagSets::NHelpers::THashSource<TString, TString> >;

        NTagSets::TManager<ui64, TString> manager;
        manager.CreateSource<TRealSource>("message", "tag")->Update();

        auto data = manager.Get()->GetTags(THash<TString>()("message"));
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.front(), "tag");
    }

    Y_UNIT_TEST(Regexp) {
        using TRealSource = TStaticSource<TString, NTagSets::NHelpers::TRegexpReplaceSource<TString>>;

        NTagSets::TManager<TString, TString> manager;
        manager.CreateSource<TRealSource>("http://www.tibra.ru/wstat/", "tag", "^\\w+:?//|^www\\.|/$", "")->Update();

        auto data = manager.Get()->GetTags("tibra.ru/wstat");
        UNIT_ASSERT_EQUAL(data.size(), 1);
        UNIT_ASSERT_EQUAL(data.front(), "tag");
    }
}
