#include "tkipv6.h"

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TestIPv6) {
    Y_UNIT_TEST(Valid) {
        const TString value = "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff";

        TKIPv6 ip(value.c_str());
        UNIT_ASSERT_EQUAL(ip.Undefined(), false);
        UNIT_ASSERT_EQUAL(ip.toStroka(), value);
    }

    Y_UNIT_TEST(ShortValid) {
        const TString value = "2a02:2f0c:3000::1"; //5 symbols f in middle

        TKIPv6 ip(value.c_str());
        UNIT_ASSERT_EQUAL(ip.Undefined(), false);
        UNIT_ASSERT_EQUAL(ip.toStroka(), "2a02:2f0c:3000:0000:0000:0000:0000:0001");
    }

    Y_UNIT_TEST(InvalidSymvol) {
        const TString value = "ffff:ffff:ffff:fzff:ffff:ffff:ffff:ffff"; // z symbol in middle

        TKIPv6 ip(value.c_str());
        UNIT_ASSERT_EQUAL(ip.Undefined(), true);
    }

    Y_UNIT_TEST(InvalidCount) {
        const TString value = "ffff:ffff:ffff:fffff:ffff:ffff:ffff:ffff"; //5 symbols f in middle

        TKIPv6 ip(value.c_str());
        UNIT_ASSERT_EQUAL(ip.Undefined(), true);
    }

    Y_UNIT_TEST(Construct) {
        TKIPv6 sample((12ULL << 32) + 34, (56ULL << 32) + 78);
        TKIPv6 value (12, 34, 56, 78);
        UNIT_ASSERT_EQUAL(sample, value);
    }

    Y_UNIT_TEST(ConstructV4) {
        TKIPv6 sample(0, 0, 0xFFFF, 13);
        TKIPv6 value (13);
        UNIT_ASSERT_EQUAL(sample, value);
    }

    Y_UNIT_TEST(ToStringV4) {
        TKIPv6 value1(0x0A0B0C0D), value2(0xDEADDEAD);
        UNIT_ASSERT_EQUAL(value1.toStroka2(), "10.11.12.13");
        UNIT_ASSERT_EQUAL(value2.toStroka2(), "222.173.222.173");
    }

    Y_UNIT_TEST(GetNextIPV4) {
        TKIPv6 value(0xFFFFFFFE);
        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv4(), true);
        UNIT_ASSERT_EQUAL(value.LowPart(), 0xFFFFFFFFFFFFULL);

        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv4(), false);
        UNIT_ASSERT_EQUAL(value.LowPart(), 0x1000000000000ULL);
    }

    Y_UNIT_TEST(GetNextIP4to6) {
        TKIPv6 value(0,0, 0xFFFE, 0xFFFFFFFF);
        UNIT_ASSERT_EQUAL(value.IsIPv6(), true);

        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv4(), true);
    }

    Y_UNIT_TEST(GetNextIP6to4) {
        TKIPv6 value(0, 0, 0xFFFF, 0xFFFFFFFF), next(0, 0, 0x10000, 0), ipv4(0, 0, 0xFFFF, 0);
        UNIT_ASSERT_EQUAL(value.IsIPv4(), true);

        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv6(), true);
        UNIT_ASSERT_EQUAL_C(value < ipv4, false, "detect cycle");
    }

    Y_UNIT_TEST(GetNextIPV6) {
        TKIPv6 value(1, 1, 0, 0xFFFFFFFE);
        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv6(), true);
        UNIT_ASSERT_EQUAL(value.LowPart() , 0xFFFFFFFF);
        UNIT_ASSERT_EQUAL(value.HighPart(), 0x100000001ULL);

        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.IsIPv6(), true);
        UNIT_ASSERT_EQUAL(value.LowPart(), 0x100000000ULL);
        UNIT_ASSERT_EQUAL(value.HighPart(), 0x100000001ULL);
    }

    Y_UNIT_TEST(GetNextIPV6_2) {
        TKIPv6 value(0, 2, 0xFFFFFFFF, 0xFFFFFFFF);
        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.LowPart() , 0);
        UNIT_ASSERT_EQUAL(value.HighPart(), 3);
    }

    Y_UNIT_TEST(GetNextIPUndefined) {
        TKIPv6 value(0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF);
        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.Undefined(), true);

        value = value.GetNextIP();
        UNIT_ASSERT_EQUAL(value.HighPart(), 0);
        UNIT_ASSERT_EQUAL(value.LowPart() , 1);
    }

    Y_UNIT_TEST(IsLocal) {
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0.1").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("128.0.0.1").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6("192.168.54.2").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("192.178.54.2").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6("172.16.21.12").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("173.16.21.12").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6("10.11.12.13").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("11.11.12.13").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6("169.254.78.87").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("169.255.78.87").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6("0.0.0.1").IsLocalIP(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("1.0.0.0").IsLocalIP(), false);

        UNIT_ASSERT_EQUAL(TKIPv6(0, 0).IsLocalIP(), false);
    }

    Y_UNIT_TEST(Operators) {
        TKIPv6 one(1, 1), two(2, 2), high(2, 1), low(1, 2);

        UNIT_ASSERT_EQUAL(one < two , true);
        UNIT_ASSERT_EQUAL(one < high, true);
        UNIT_ASSERT_EQUAL(one < low , true);
        UNIT_ASSERT_EQUAL(one < one, false);

        UNIT_ASSERT_EQUAL(one <= two, true);
        UNIT_ASSERT_EQUAL(one <= high, true);
        UNIT_ASSERT_EQUAL(one <= low, true);
        UNIT_ASSERT_EQUAL(one <= one, true);

        UNIT_ASSERT_EQUAL(two  >= one, true);
        UNIT_ASSERT_EQUAL(high >= one, true);
        UNIT_ASSERT_EQUAL(low  >= one, true);
        UNIT_ASSERT_EQUAL(one  >= one, true);

        UNIT_ASSERT_EQUAL(two  > one, true);
        UNIT_ASSERT_EQUAL(high > one, true);
        UNIT_ASSERT_EQUAL(low  > one, true);
        UNIT_ASSERT_EQUAL(one  > one, false);

        UNIT_ASSERT_EQUAL(one == one, true);
        UNIT_ASSERT_EQUAL(one == two, false);

        UNIT_ASSERT_EQUAL(one != one, false);
        UNIT_ASSERT_EQUAL(one != two, true);
    }

    Y_UNIT_TEST(AddressMaskIPv4) {
        TKIPv6 value(std::numeric_limits<ui32>::max());
        UNIT_ASSERT_EQUAL(value.GetAddressByMask(0).Undefined(), true);

        for (ui8 i = 1; i < 33; ++i)
            UNIT_ASSERT_EQUAL_C(value.GetAddressByMask(i).IPv4Address(), std::numeric_limits<ui32>::max() << (32 - i), static_cast<ui32>(i));

        for (ui32 i = 33; i < 256; ++i)
            UNIT_ASSERT_EQUAL_C(value.GetAddressByMask(static_cast<ui8>(i)).Undefined(), true, i);
    }

    Y_UNIT_TEST(AddressMaskIPv6) {
        TKIPv6 value(std::numeric_limits<ui64>::max(), std::numeric_limits<ui64>::max());
        UNIT_ASSERT_EQUAL(value.GetAddressByMask(0).Undefined(), true);

        for (ui8 i = 1; i < 65; ++i)
        {
            auto masked = value.GetAddressByMask(i);
            UNIT_ASSERT_EQUAL_C(masked.HighPart(), std::numeric_limits<ui64>::max() << (64 - i), static_cast<ui32>(i));
            UNIT_ASSERT_EQUAL_C(masked.LowPart(), 0, static_cast<ui32>(i));
        }

        for (ui8 i = 65; i < 129; ++i)
        {
            auto masked = value.GetAddressByMask(i);
            UNIT_ASSERT_EQUAL_C(masked.HighPart(), std::numeric_limits<ui64>::max(), static_cast<ui32>(i));
            UNIT_ASSERT_EQUAL_C(masked.LowPart(), std::numeric_limits<ui64>::max() << (128 - i), static_cast<ui32>(i));
        }

        for (ui32 i = 129; i < 256; ++i)
            UNIT_ASSERT_EQUAL_C(value.GetAddressByMask(static_cast<ui8>(i)).Undefined(), true, i);
    }

    Y_UNIT_TEST(ConstructIPv4) {
        UNIT_ASSERT_EQUAL(TKIPv6("127...").IPv4Address(), 0x7F000000);
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0.1").IPv4Address(), 0x7F000001);
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0.12 ").IPv4Address(), 0x7F00000C);
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0.  12 ").IPv4Address(), 0x7F00000C);

        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.0.1.").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6(".127.0.0.1").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("127.0.257.1").Undefined(), true);
    }

    Y_UNIT_TEST(ConstructIPv4_2) {
        UNIT_ASSERT_EQUAL(TKIPv6("::ffff:127...").IPv4Address(), 0x7F000000);
        UNIT_ASSERT_EQUAL(TKIPv6("::ffff:127.0.0.1").IPv4Address(), 0x7F000001);
        UNIT_ASSERT_EQUAL(TKIPv6("::f f  f   f : 127.  0.  0.1").IPv4Address(), 0x7F000001);
        UNIT_ASSERT_EQUAL(TKIPv6("0:0:0:0:0:FFFF:127.0.0.1").IPv4Address(), 0x7F000001);
        UNIT_ASSERT_EQUAL(TKIPv6("0:0:0:0:0:ffff:127.0.0.1").IPv4Address(), 0x7F000001);

        UNIT_ASSERT_EQUAL(TKIPv6("::ffff:127.0.0").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("::FFFF:127.0.0").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("::ffff:.127.0.0.1").Undefined(), true);

        UNIT_ASSERT_EQUAL(TKIPv6("::ff:127.0.0.1").IsIPv4(), false);
        UNIT_ASSERT_EQUAL(TKIPv6("1:0:0:0:0:FFFF:127.0.0.1").IsIPv4(), false);
    }

    Y_UNIT_TEST(ConstructIPv6) {
        UNIT_ASSERT_EQUAL(TKIPv6("1:2:3:4:5:6:7:8"), TKIPv6(0x1000200030004, 0x5000600070008));
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:DEAD:DEAD:0000:0000:0000:0001"), TKIPv6(0xDEADDEAD, 1));
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:DEAD:DEAD::0001:0001"), TKIPv6(0xDEADDEAD, 0x10001));
        UNIT_ASSERT_EQUAL(TKIPv6("::DEAD:DEAD:0000:0000:0000:0001"), TKIPv6(0xDEADDEAD, 1));
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:dead:dead:0:0:0:1"), TKIPv6(0xDEADDEAD, 1));
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:DEAD:DEAD::0001"), TKIPv6(0xDEADDEAD, 1));
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:DEAD:DEAD::"), TKIPv6(0xDEADDEAD, 0));

        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEAD:0000:0000:0000:0000:0000:0001:0004").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEAD:0000:0000:0000:0000:0000:0001::").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("0000:0000:DEAD:DEAD:0000:0000:0000::0001").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEADDAD:0000:0000:0000:0000:0000").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEAD:0000:0000:0000:0000:0000").Undefined(), true);
    }

    Y_UNIT_TEST(ConstructIPv4Embedded) {
        UNIT_ASSERT_EQUAL(TKIPv6("0:0:0:DEAD:0:0:127.0.0.1"), TKIPv6(0xDEAD, 0x7F000001));
        UNIT_ASSERT_EQUAL(TKIPv6("0:0:0:DEAD::0:127.0.0.1"), TKIPv6(0xDEAD, 0x7F000001));
        UNIT_ASSERT_EQUAL(TKIPv6("0:0:0:DEAD::127.0.0.1"), TKIPv6(0xDEAD, 0x7F000001));

        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEAD:0000:0000:0000:0000:0000:0001:127.0.0.1").Undefined(), true);
        UNIT_ASSERT_EQUAL(TKIPv6("DEAD:DEAD:0000:0000:0000:0000:0001:127.0.0.1").Undefined(), true);
    }
}
