#include "tdumpreadclass.h"
#include "tstatipobject.h"

//**************************************************************************************************************************
//                                           TDumpReadClass
//**************************************************************************************************************************

TDumpReadClass::TDumpReadClass()
{
   statipobj               = NULL;
   m_stortype              = MST_UNKNOWN;
   fulltick = CShingleTime::GetMs();
   m_dumpfilename          = "";
   m_handle                = NULL;
   m_read_from_file        = 0;
   m_log                   = NULL;
   m_buffersize            = DEFAULT_QUEUE_SIZE;
   m_full_read_file        = false;
   m_parser_error          = 0;
   m_send_ok               = 0;
   m_negativ_value         = 0;
   m_big_value             = 0;
   m_send_error            = 0;
   m_id_notfound           = 0;
   m_id_empty              = 0;
   m_last_read_from_file   = 0;
   m_last_parser_error     = 0;
   m_last_send_ok          = 0;
   m_last_negativ_value    = 0;
   m_last_big_value        = 0;
   m_last_send_error       = 0;
   m_last_id_notfound      = 0;
   m_last_id_empty         = 0;
   m_writelog_time         = time(NULL);
   m_start                 = false;
}

TDumpReadClass::~TDumpReadClass()
{
   fulltick = CShingleTime::GetMs() - fulltick;
   if ( (m_start) && (m_log != NULL) )
   {
      m_log->WriteMessageAndDataStatus(KMESSAGE, "READDUMP %s: LOADCOMPLETE to %u msec, send %lld record (ok=%llu, parserr=%llu, senderr=%llu, negativvalue=%llu, bigvalue=%llu, id_notfound=%llu, id_empty=%llu)", m_ident.c_str(), fulltick, m_read_from_file, m_send_ok, m_parser_error, m_send_error, m_negativ_value, m_big_value, m_id_notfound, m_id_empty);
      m_log->FFlush();
   }

   if (m_handle != NULL)
   {
      fclose(m_handle);
      m_handle = NULL;
   }
}

bool TDumpReadClass::Init(void *statipobjA, const TString ident, ui32 thread_count, ui32 buffersize, const TString &dumpfilenameA, TLogClass *log, TStorageType stortype)
{
   bool    res = false;
   TString  tstr = "";
   char    tbuff[16];
   ui32    crc  = 0;

   statipobj = statipobjA;

   tstr = ident + "_" + IntToStroka(time(NULL));
   if (!tstr.empty())
   {
      crc = Crc32((unsigned char *)tstr.c_str(), tstr.length());
      memset(tbuff, 0, sizeof(tbuff));

      snprintf(tbuff, sizeof(tbuff) - 1, "%08x", crc);
      tstr = TString(tbuff);
   }

   if (thread_count < 1)
      thread_count = 1;
   if (thread_count >= TDumpReadClassMain::MAX_THREAD_WORK)
      thread_count = TDumpReadClassMain::MAX_THREAD_WORK;

   m_ident        = ident + "-" + tstr;
   m_ident        = to_upper(m_ident);
   m_dumpfilename = dumpfilenameA;
   m_log          = log;
   m_stortype     = stortype;
   if (buffersize == 0)
      m_buffersize = DEFAULT_QUEUE_SIZE;
   else if (buffersize > (50 * DEFAULT_QUEUE_SIZE))
      m_buffersize = 50 * DEFAULT_QUEUE_SIZE;
   else
      m_buffersize = buffersize;

   if (FileExists(dumpfilenameA))
   {
      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KMESSAGE, "READDUMP %s: START read dump '%s' (thread_count=%u, buffer_size=%u)", m_ident.c_str(), m_dumpfilename.c_str(), thread_count, m_buffersize);
         m_log->FFlush();
      }

      m_start = true;
      res     = true;

   } else
   {
      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KMESSAGE, "READDUMP %s: START read dump '%s' FAILED, file not found", m_ident.c_str(), m_dumpfilename.c_str());
         m_log->FFlush();
      }

      m_start = false;
   }

   return res;
}

void TDumpReadClass::UserLock()
{
   m_Mutex.Acquire();
}

void TDumpReadClass::UserUnLock()
{
   m_Mutex.Release();
}

bool TDumpReadClass::FileExists(const TString &filename)
{
   bool res = false;

   if (!filename.empty())
   {
      FILE *handle = fopen(filename.c_str(), "rb");
      if (handle != NULL)
      {
         res = true;

         fclose(handle);
         handle = NULL;
      }
   }

   return res;
}

void TDumpReadClass::ActionFunction(int threadnumber, bool &needstop)
{
   TDumpReadQueueIt  it;
   ui32              needdiff     = 0;
   bool              needloaddata = false;
   bool              rcv_endfile  = false;
   nosql::HashMap    hash;
   int               a            = 0;
   TQueueItem        qi;
   ui32              currenttime  = time(NULL);
   ui32              difftime     = 0;
   bool              need_print_stat = false;

   needstop = false;

   UserLock();

   it = queue.begin();
   if (it != queue.end())
   {
      qi = (*it);
      queue.erase(it);
   }
   if ( (!rcv_endfile) && (threadnumber == 1) )
      needloaddata = NeedLoadData(needdiff);

   UserUnLock();

   if (qi.is_end)
   {
      rcv_endfile = true;
      needstop = true;

   } else
   {
      if (!qi.data.empty())
      {
         need_print_stat = false;

         if (ParseDumpRecord(qi, hash))
            SendToStorage(qi, hash);
         else
            m_parser_error = IncMax64(m_parser_error, 1);

         if (currenttime > m_writelog_time)
            difftime = currenttime - m_writelog_time;
         else
            difftime = 0;

         if ( ((difftime >= 60) || ((qi.str_number % PRINT_PERIOD_COUNT) == 0)) && (m_log != NULL) )
         {
            ui64  read_from_file       = 0;
            ui64  parser_error         = 0;
            ui64  send_ok              = 0;
            ui64  negativ_value        = 0;
            ui64  big_value            = 0;
            ui64  send_error           = 0;
            ui64  id_notfound          = 0;
            ui64  id_empty             = 0;
            float buffer_full_percent  = 0;

            UserLock();

            if (currenttime > m_writelog_time)
               difftime = currenttime - m_writelog_time;
            else
               difftime = 0;

            if (difftime >= 60)
            {
               read_from_file = m_read_from_file - m_last_read_from_file;
               parser_error   = m_parser_error   - m_last_parser_error;
               send_ok        = m_send_ok        - m_last_send_ok;
               negativ_value  = m_negativ_value  - m_last_negativ_value;
               big_value      = m_big_value      - m_last_big_value;
               send_error     = m_send_error     - m_last_send_error;
               id_notfound    = m_id_notfound    - m_last_id_notfound;
               id_empty       = m_id_empty       - m_last_id_empty;
               buffer_full_percent = 0;

               if (m_buffersize > 0)
                  buffer_full_percent = (float)queue.size() * (float)100 / (float)m_buffersize;
               else
                  buffer_full_percent = 0;
               m_last_read_from_file = m_read_from_file;
               m_last_parser_error   = m_parser_error;
               m_last_send_ok        = m_send_ok;
               m_last_negativ_value  = m_negativ_value;
               m_last_big_value      = m_big_value;
               m_last_send_error     = m_send_error;
               m_last_id_notfound    = m_id_notfound;
               m_last_id_empty       = m_id_empty;
               m_writelog_time       = currenttime;

               need_print_stat = true;
            }

            UserUnLock();

            if (need_print_stat)
            {
               m_log->WriteMessageAndDataStatus(KMESSAGE, "READDUMP %s: send %lld record (buff=%0.2f%%, ok=%llu, parserr=%llu, senderr=%llu, negativvalue=%llu, bigvale=%llu, id_notfound=%llu, id_empty=%llu)", m_ident.c_str(), qi.str_number, buffer_full_percent, send_ok, parser_error, send_error, negativ_value, big_value, id_notfound, id_empty);
               m_log->FFlush();
            }
         }
      }
   }

   if ( (!rcv_endfile) && (threadnumber == 1) && (needloaddata) )
      ReadToQueue();

}

bool TDumpReadClass::NeedLoadData(ui32 &needdiff)
{
   bool res = false;

   needdiff = m_buffersize - queue.size();
   if (needdiff >= (0.9 * m_buffersize))
      res = true;

   return res;
}

ui32 TDumpReadClass::Crc32(unsigned char *buf, ui32 len)
{
    ui32 crc_table[256];
    ui32 crc = 0;

    for (int i = 0; i < 256; i++)
    {
        crc = i;
        for (int j = 0; j < 8; j++)
            crc = crc & 1 ? (crc >> 1) ^ 0xEDB88320UL : crc >> 1;

        crc_table[i] = crc;
    };

    crc = 0xFFFFFFFFUL;

    while (len--)
        crc = crc_table[(crc ^ *buf++) & 0xFF] ^ (crc >> 8);

    return crc ^ 0xFFFFFFFFUL;
}


void TDumpReadClass::ReadToQueue()
{
   if (!m_full_read_file)
   {
      if (m_handle == NULL)
      {
         if (!m_dumpfilename.empty())
         {
            m_handle = fopen(m_dumpfilename.c_str(), "rb");
            current_numb_str = 0;
         }
      }
      if (m_handle != NULL)
      {
         ui32 need_diff      = 0;
         bool need_load_data = false;
         int  sz             = 0;

         UserLock();
         need_load_data = NeedLoadData(need_diff);
         UserUnLock();

         if (need_load_data)
         {
            char     tbuff[4096];
            ui32     readcount = need_diff;
            TString   tstr = "";

            while (readcount > 0)
            {
               if (m_handle != NULL)
               {
                  memset(tbuff, 0, sizeof(tbuff));
                  if (fgets(tbuff, sizeof(tbuff) - 1, m_handle) != NULL)
                  {
                     readcount--;

                     sz = strlen(tbuff);
                     if ( (sz > 0) && (sz < sizeof(tbuff)) )
                     {
                        for (int i = (sz - 1); i >= 0; i--)
                        {
                           if (tbuff[i] < ' ')
                              tbuff[i] = ' ';
                           else if (tbuff[i] == '&')
                              break;
                        }
                     }

                     current_numb_str++;

                     tstr = TString(tbuff);
                     if (!tstr.empty())
                     {
                        UserLock();

                        queue.push_back(TQueueItem(current_numb_str, tstr));
                        m_read_from_file = IncMax64(m_read_from_file, 1);

                        UserUnLock();
                     }

                  } else
                  {
                     if (feof(m_handle))
                     {
                        UserLock();

                        queue.push_back(TQueueItem(true));

                        UserUnLock();

                        if (m_log != NULL)
                        {
                           m_log->WriteMessageAndDataStatus(KMESSAGE, "READDUMP %s: READCOMPLETE, read %llu record.", m_ident.c_str(), m_read_from_file);
                           m_log->FFlush();
                        }

                     } else
                     {
                        int err = ferror(m_handle);

                        UserLock();

                        queue.push_back(TQueueItem(true));

                        UserUnLock();

                        if (m_log != NULL)
                        {
                           m_log->WriteMessageAndDataStatus(KERROR, "READDUMP %s: READFAILED (ERR=%d), read %llu record.", m_ident.c_str(), err, m_read_from_file);
                           m_log->FFlush();
                        }

                     }
                     m_full_read_file = true;
                     break;
                  }

               } else
               {
                  break;
               }
            }

         }
      }
   }
}

bool TDumpReadClass::ParseDumpRecord(TQueueItem &rec, nosql::HashMap &hash)
{
   bool       res             = false;
   const char *pb             = NULL;
   const char *pe             = NULL;
   const char *p1             = NULL;
   const char *p2             = NULL;
   const char *p3             = NULL;
   TString     idn2            = "=#";
   TString     fieldname       = "";
   TString     fieldtype       = "";
   TString     fieldvalue      = "";
   int        fieldtype_i     = 0;
   int        count           = 0;
   nosql::AnyValue::TYPE fieldtype_av;
   i64        value_i64_av    = 0;
   int        value_int_av    = 0;
   double     value_double_av = 0;
   TString     err_str         = "";
   ui32       numb_str        = rec.str_number;

   hash.clear();
   if (!rec.data.empty())
   {
      pb = rec.data.c_str();
      pe = (pb != NULL) ? strchr(pb, '&') : NULL;
      while (pe != NULL)
      {
         fieldname   = "";
         fieldtype   = "";
         fieldvalue  = "";
         fieldtype_i = 0;
         fieldtype_av = nosql::AnyValue::NOTYPE;

         p1 = strstr(pb, idn2.c_str());
         if ( (p1 != NULL) && (p1 < pe) )
         {
            count = p1 - pb;
            if (count > 0)
               fieldname = TString(pb, count);
            else
               err_str += "can't find fieldname2, ";

            p2 = strchr(p1 + idn2.length(), '\'');
            if ( (p2 != NULL) && (p2 < pe) )
            {
               count = p2 - p1 - idn2.length();
               if (count > 0)
               {
                  fieldtype = TString(p1 + idn2.length(), count);
                  if (!fieldtype.empty())
                  {
                     fieldtype_i = atoi(fieldtype.c_str());
                     fieldtype_av = static_cast<nosql::AnyValue::TYPE>(fieldtype_i);
                  } else
                  {
                     err_str += "can't find fieldtype2, ";
                  }

                  p3 = strchr(p2 + 1, '\'');
                  if ( (p3 != NULL) && (p3 < pe) )
                  {
                     count = p3 - p2 - 1;
                     fieldvalue = TString(p2 + 1, count);
                     if (!fieldvalue.empty())
                        CGIUnescape(fieldvalue);

                     switch (fieldtype_av)
                     {
                     case nosql::AnyValue::INTEGER:
                                                      value_int_av = 0;
                                                      if ( (!fieldvalue.empty()) && (sscanf(fieldvalue.c_str(), "%d", &value_int_av) == 1) )
                                                         hash[fieldname] = static_cast<nosql::AnyValue>(value_int_av);
                                                      break;
                     case nosql::AnyValue::INTEGER64:
                                                      value_i64_av = 0;
                                                      if ( (!fieldvalue.empty()) && (sscanf(fieldvalue.c_str(), "%lld", &value_i64_av) == 1) )
                                                         hash[fieldname] = static_cast<nosql::AnyValue>(value_i64_av);
                                                      break;
                     case nosql::AnyValue::DOUBLE:
                                                      value_double_av = 0;
                                                      if (!fieldvalue.empty())
                                                      {
                                                         value_double_av = atof(fieldvalue.c_str());
                                                         hash[fieldname] = static_cast<nosql::AnyValue>(value_double_av);
                                                      }
                                                      //if ( (!fieldvalue.empty()) && (sscanf(fieldvalue.c_str(), "%a", &value_double_av) == 1) )
                                                      //   hash[fieldname] = static_cast<nosql::AnyValue>(value_double_av);
                                                      break;
                     case nosql::AnyValue::VECTOR:
                                                      err_str += fieldname + ": no support VECTOR type, ";
                                                      break;
                     case nosql::AnyValue::STRING:
                                                      hash[fieldname] = static_cast<nosql::AnyValue>(fieldvalue);
                                                      break;
                     case nosql::AnyValue::LIST:
                                                      err_str += fieldname + ": no support LIST type, ";
                                                      break;
                     case nosql::AnyValue::HASH:
                                                      err_str += fieldname + ": no support HASH type, ";
                                                      break;
                     default:
                                                      err_str += fieldname + ": no define type, ";
                     };

                  } else
                  {
                     err_str += "can't find fieldvalue1, ";
                  }

               } else
               {
                  err_str += "can't find fieldtype2, ";
               }

            } else
            {
               err_str += "can't find fieldtype1, ";
            }

         } else
         {
            err_str += "can't find fieldname1, ";
         }


         pb = pe + 1;
         pe = (pb != NULL) ? strchr(pb, '&') : NULL;
      }

   } else
   {
      err_str += "empty record data";
   }
   if (!err_str.empty())
   {
      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KERROR, "READDUMP %s: PARSEERR [Ln=%u] ERR='%s'", m_ident.c_str(), numb_str, err_str.c_str());
         m_log->FFlush();
      }

   } else
      res = true;

   return res;
}

void TDumpReadClass::SendToStorage(TQueueItem &qi, nosql::HashMap &sets)
{
   TString         id          = "";
   TString         ip          = "";
   nosql::HashMap incrs;
   bool           err         = false;
   //int            flags       = MONGO_UPDATE_UPSERT;
   TString         ip_ident    = "ip";
   TString         id_ident    = "_id";
   bool           id_find     = false;
   bool           ip_find     = false;
   nosql::HashMap::iterator it;
   TNegativValueStruct negstat;
   bool           skeep       = false;
   TString         action_str  = "";
   int            try_count   = 0;

   //_id
   it = sets.find(id_ident);
   if (it != sets.end())
   {
      if ((*it).second.Type() == nosql::AnyValue::STRING)
      {
         id = (*it).second.String();
         id_find = true;
         sets.erase(it);
      }
   }
   //ip
   it = sets.find(ip_ident);
   if (it != sets.end())
   {
      if ((*it).second.Type() == nosql::AnyValue::STRING)
      {
         ip = (*it).second.String();
         ip_find = true;
      }
   }

   if ( (!skeep) && (FindNegativValue(sets, negstat)) )
   {
      m_negativ_value++;

      //skeep = true;
      if (skeep)
      {
         action_str = "skeep";

      } else
      {
         action_str = "correct";
         CorrectBigNegValue(sets);
      }

      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KWARNING, "READDUMP %s: NEGATIVVALUE, %s send [Ln=%lld, id='%s', ip='%s', err='%s']", m_ident.c_str(), action_str.c_str(), qi.str_number, id.c_str(), negstat.m_ip.c_str(), negstat.m_errs.c_str());
         m_log->FFlush();
      }
   }

   if ( (!skeep) && (FindBigValue(sets, negstat)) )
   {
      m_big_value++;

      //skeep = true;
      if (skeep)
      {
         action_str = "skeep";

      } else
      {
         action_str = "correct";
         CorrectBigNegValue(sets);
      }

      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KWARNING, "READDUMP %s: BIGVALUE, %s send [Ln=%lld, id='%s', ip='%s', err='%s']", m_ident.c_str(), action_str.c_str(), qi.str_number, id.c_str(), negstat.m_ip.c_str(), negstat.m_errs.c_str());
         m_log->FFlush();
      }
   }

   if (!skeep)
   {
      if ( (id_find) && (!id.empty()) )
      {
         try_count = 0;
         for (int i = 0; i < SEND_TRY_COUNT_DEFAULT; i++)
         {
            try_count++;
            err = false;

            if (statipobj != NULL)
            {
               if (((TStatIPObject *)statipobj)->GetStorage() != NULL)
               {
                  ((TStatIPObject *)statipobj)->GetStorage()->mongo_update_id_data(m_stortype, id, err, sets, incrs);
                  //nosql::Mongo_CachedClient::Instance().GetRawClient().Update( m_db, m_collection, id, incrs, sets, err, flags );
                  if (err)
                  {
                     if (i < (SEND_TRY_COUNT_DEFAULT - 1))
                     {
                        usleep(50000);  //50 msec

                     } else
                     {
                        m_send_error++;
                        if (m_log != NULL)
                        {
                           m_log->WriteMessageAndDataStatus(KERROR, "READDUMP %s: SETERR [try=%d, Ln=%lld, id='%s', ip='%s'] src=<%s>", m_ident.c_str(), try_count, qi.str_number, id.c_str(), ip.c_str(), qi.data.c_str());
                           m_log->FFlush();
                        }
                     }

                  } else
                  {
                     m_send_ok++;
                     if ( (try_count > 1) && (m_log != NULL) )
                     {
                        m_log->WriteMessageAndDataStatus(KWARNING, "READDUMP %s: TRYCNT [try=%d, ip='%s']", m_ident.c_str(), try_count, ip.c_str());
                        m_log->FFlush();
                     }

                     break;
                  }
               }
            }

         }

      } else if (!id_find)
      {

         m_id_notfound++;

         if (m_log != NULL)
         {
            m_log->WriteMessageAndDataStatus(KERROR, "READDUMP %s: IDNOTFOUND [Ln=%lld, ip='%s']", m_ident.c_str(), qi.str_number, ip.c_str());
            m_log->FFlush();
         }

      }  else if (id.empty())
      {

         m_id_empty++;

         if (m_log != NULL)
         {
            m_log->WriteMessageAndDataStatus(KERROR, "READDUMP %s: IDEMPTY [Ln=%lld, ip='%s']", m_ident.c_str(), qi.str_number, ip.c_str());
            m_log->FFlush();
         }
      }
   }
}

bool TDumpReadClass::FindNegativValue(nosql::HashMap &hash, TNegativValueStruct &stat)
{
    bool              res = false;
    nosql::HashMap::iterator hit;
    i64               history_spam = 0;
    i64               history_ham  = 0;
    i64               history_bath = 0;
    i64               today_spam   = 0;
    i64               today_ham    = 0;
    i64               today_bath   = 0;
    TString            ip           = "";
    TString            id           = "";

    stat.Clear();

    hit = hash.find("ip");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::STRING)
       {
          ip = (*hit).second.String();
          stat.m_ip = ip;
       }
    }

    hit = hash.find("_id");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::STRING)
       {
          id = (*hit).second.String();
          stat.m_id = id;
       }
    }

    hit = hash.find("ctSpam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          today_spam = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          today_spam = (*hit).second.Long();

    }

    hit = hash.find("ctHam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          today_ham = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          today_ham = (*hit).second.Long();

    }

    if ( (today_spam > 0) && (today_ham > 0) )
      today_bath = IncMax64(today_spam, today_ham);

    hit = hash.find("hctSpam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          history_spam = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          history_spam = (*hit).second.Long();

    }

    hit = hash.find("hctHam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          history_ham = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          history_ham = (*hit).second.Long();

    }

    if (today_spam < 0)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", ctSpam=" + I64ToStroka(today_spam);
       else
         stat.m_errs += "ctSpam=" + I64ToStroka(today_spam);

       res = true;
    }

    if (today_ham < 0)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", ctHam=" + I64ToStroka(today_ham);
       else
         stat.m_errs += "ctHam=" + I64ToStroka(today_ham);

       res = true;
    }

    if (history_spam < 0)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", hctSpam=" + I64ToStroka(history_spam);
       else
         stat.m_errs += "hctSpam=" + I64ToStroka(history_spam);

       res = true;
    }

    if (history_ham < 0)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", hctHam=" + I64ToStroka(history_ham);
       else
         stat.m_errs += "hctHam=" + I64ToStroka(history_ham);

       res = true;
    }

    return res;
}

bool TDumpReadClass::FindBigValue(nosql::HashMap &hash, TNegativValueStruct &stat)
{
    bool              res = false;
    nosql::HashMap::iterator hit;
    i64               history_spam = 0;
    i64               history_ham  = 0;
    i64               history_bath = 0;
    i64               today_spam   = 0;
    i64               today_ham    = 0;
    i64               today_bath   = 0;
    TString            ip           = "";
    TString            id           = "";

    stat.Clear();

    hit = hash.find("ip");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::STRING)
       {
          ip = (*hit).second.String();
          stat.m_ip = ip;
       }
    }

    hit = hash.find("_id");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::STRING)
       {
          id = (*hit).second.String();
          stat.m_id = id;
       }
    }

    hit = hash.find("ctSpam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          today_spam = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          today_spam = (*hit).second.Long();

    }

    hit = hash.find("ctHam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          today_ham = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          today_ham = (*hit).second.Long();

    }

    if ( (today_spam > 0) && (today_ham > 0) )
      today_bath = IncMax64(today_spam, today_ham);

    hit = hash.find("hctSpam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          history_spam = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          history_spam = (*hit).second.Long();

    }

    hit = hash.find("hctHam");
    if (hit != hash.end())
    {
       if ((*hit).second.Type() == nosql::AnyValue::INTEGER)
          history_ham = (*hit).second.Integer();
       else if ((*hit).second.Type() == nosql::AnyValue::INTEGER64)
          history_ham = (*hit).second.Long();

    }

    if (today_spam > MAX32BITVALUE)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", ctSpam=" + I64ToStroka(today_spam);
       else
         stat.m_errs += "ctSpam=" + I64ToStroka(today_spam);

       res = true;
    }

    if (today_ham > MAX32BITVALUE)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", ctHam=" + I64ToStroka(today_ham);
       else
         stat.m_errs += "ctHam=" + I64ToStroka(today_ham);

       res = true;
    }

    if (history_spam > MAX32BITVALUE)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", hctSpam=" + I64ToStroka(history_spam);
       else
         stat.m_errs += "hctSpam=" + I64ToStroka(history_spam);

       res = true;
    }

    if (history_ham > MAX32BITVALUE)
    {
       if (!stat.m_errs.empty())
         stat.m_errs += ", hctHam=" + I64ToStroka(history_ham);
       else
         stat.m_errs += "hctHam=" + I64ToStroka(history_ham);

       res = true;
    }

    return res;
}

void CorrectValue(nosql::HashMap &hash, const TString &field)
{
    nosql::HashMap::iterator hit;
    i64                      corr_value = 0;

    hit = hash.find(field);
    if (hit != hash.end())
       (*hit).second = corr_value;

}

void TDumpReadClass::CorrectBigNegValue(nosql::HashMap &hash)
{
    //today
    CorrectValue(hash, "malicspam_count");
    CorrectValue(hash, "ctMalicSpam");
    CorrectValue(hash, "ctMalicSpamMail");
    CorrectValue(hash, "ctSpam");
    CorrectValue(hash, "ctSpamMail");
    CorrectValue(hash, "ctHam");
    CorrectValue(hash, "ctHamMail");
    CorrectValue(hash, "cDsn");
    CorrectValue(hash, "cPop3");

    //yesterday
    CorrectValue(hash, "yctMalicSpam");
    CorrectValue(hash, "yctSpam");
    CorrectValue(hash, "yctHam");
    CorrectValue(hash, "ycDsn");
    CorrectValue(hash, "ycPop3");

    //history
    CorrectValue(hash, "hctMalicSpam");
    CorrectValue(hash, "hctSpam");
    CorrectValue(hash, "hctHam");
    CorrectValue(hash, "hcDsn");
    CorrectValue(hash, "hcPop3_Auth");
}

//***************************************************************************************************************************
//                                                 TDumpReadClassMain
//***************************************************************************************************************************

void DRCMThreadProc(void* par)
{
    TDumpReadClassMain *ss = (TDumpReadClassMain *)par;

    if (ss != NULL)
    {
      int threadnumber = ss->IncrementNumberThread();

      //printf("sth=%d\n", threadnumber);
      while( !ss->QueueThreadShouldStop() )
      {
         ss->ActionFunction(threadnumber);

      }

      ss->DecrementNumberThread();
    }
}

TDumpReadClassMain::TDumpReadClassMain()
{
   dr_obj = new TDumpReadClass();

   for (int i = 0; i < MAX_THREAD_WORK; i++)
      m_QueueThread[i] = NULL;
   use_threadcount         = 0;
   run_scan_thread         = false;
   m_StopQueueThread       = false;
   m_thread_count          = 1;
}

TDumpReadClassMain::~TDumpReadClassMain()
{
   if (run_scan_thread)
   {
      StopQueueThread();
      while(!QueueThreadStopped())
         usleep(5000);

      for (int i = 0; i < MAX_THREAD_WORK; i++)
      {
         if (m_QueueThread[i] != NULL)
         {
            delete m_QueueThread[i];
            m_QueueThread[i] = NULL;
         }
      }
   }

   if (dr_obj != NULL)
   {
      delete dr_obj;
      dr_obj = NULL;
   }
}

bool TDumpReadClassMain::InitBase(ui32 thread_count)
{
   bool res = true;

   m_thread_count = thread_count;
   if (m_thread_count < 1)
      m_thread_count = 1;
   if (m_thread_count >= MAX_THREAD_WORK)
      m_thread_count = MAX_THREAD_WORK;

   StartQueueThread();

   return res;
}

bool TDumpReadClassMain::Init(void *statipobjA, const TString &ident, ui32 thread_count, ui32 buffersize, const TString &dumpfilenameA, TLogClass *log, TStorageType stortype)
{
   bool res = false;

   if (dr_obj != NULL)
   {
      if (dr_obj->Init(statipobjA, ident, thread_count, buffersize, dumpfilenameA, log, stortype))
         res = InitBase(thread_count);
   }

   return res;
}

void TDumpReadClassMain::StartQueueThread()
{
    m_QueueMutex.Acquire();

    m_StopQueueThread = false;

    for (int i = 0; i < m_thread_count; i++)
    {
       m_QueueThread[i] = new TThread((TThread::TThreadProc)&DRCMThreadProc, this);
       m_QueueThread[i]->Start();
    }
    run_scan_thread = true;

    m_QueueMutex.Release();

}

int TDumpReadClassMain::IncrementNumberThread()
{
   int res = 0;

   m_QueueMutex.Acquire();

   use_threadcount++;
   res = use_threadcount;

   m_QueueMutex.Release();

   return res;
}

void TDumpReadClassMain::DecrementNumberThread()
{
   m_QueueMutex.Acquire();

   if (use_threadcount > 0)
      use_threadcount--;

   m_QueueMutex.Release();
}

void TDumpReadClassMain::StopQueueThread()
{
    m_QueueMutex.Acquire();
    m_StopQueueThread = true;
    m_QueueMutex.Release();

}

bool TDumpReadClassMain::QueueThreadStopped()
{
    bool res = false;

    if (use_threadcount == 0)
       res = true;

    return res;
}

void TDumpReadClassMain::Wait()
{
   if (run_scan_thread)
   {
      sleep(1);
      while(!QueueThreadStopped())
        sleep(1);
   }
}

void TDumpReadClassMain::ActionFunction(int threadnumber)
{
   bool needstop = false;

   if (dr_obj != NULL)
      dr_obj->ActionFunction(threadnumber, needstop);
   if (needstop)
      StopQueueThread();
}

//***************************************************************************************************************************
//                                                 TDiskDumpConverter
//***************************************************************************************************************************

TDiskDumpConverter::TDiskDumpConverter()
{
   m_IPDiskIndexFilename   = "";
   m_IPDiskBasaFilename    = "";
   m_IPDiskTargetFilename  = "";
   m_BanTargetFilename     = "";
   m_BanIndexFilename      = "";
   m_Ban7TargetFilename    = "";
   m_Ban7IndexFilename     = "";
   idxhandle               = NULL;
   bashandle               = NULL;
   m_open_indexfile        = false;
   m_open_basafile         = false;
   m_log                   = NULL;
}

TDiskDumpConverter::~TDiskDumpConverter()
{
   if (idxhandle != NULL)
   {
    delete idxhandle;
    idxhandle = NULL;
   }
   if (bashandle != NULL)
   {
    delete bashandle;
    bashandle = NULL;
   }
   if (m_log != NULL)
   {
      delete m_log;
      m_log = NULL;
   }
}

bool TDiskDumpConverter::Init(const TString &config_filename)
{
   bool res = false;

   m_open_indexfile      = false;
   m_open_basafile       = false;
   m_log                 = NULL;
   if (m_config.Load(config_filename))
   {
      TString logfilename = "";

      logfilename           = m_config.ReadStroka("convdmp", "logfilename", "");

      if (!logfilename.empty())
         m_log = new TLogClass(logfilename, true);

      m_IPDiskIndexFilename   = m_config.ReadStroka("convdmp", "ipdisk_index_filename", "");
      m_IPDiskBasaFilename    = m_config.ReadStroka("convdmp", "ipdisk_basa_filename", "");
      m_IPDiskTargetFilename  = m_config.ReadStroka("convdmp", "ipdisk_target_filename", "");

      m_BanTargetFilename     = m_config.ReadStroka("convdmp", "ban_target_filename", "");
      m_BanIndexFilename      = m_config.ReadStroka("convdmp", "ban_index_filename", "");

      m_Ban7TargetFilename    = m_config.ReadStroka("convdmp", "ban7_target_filename", "");
      m_Ban7IndexFilename     = m_config.ReadStroka("convdmp", "ban7_index_filename", "");

      if ( (!m_IPDiskIndexFilename.empty()) && (!m_IPDiskBasaFilename.empty()) && (!m_IPDiskTargetFilename.empty()) && (!m_BanTargetFilename.empty()) && (!m_BanIndexFilename.empty()) && (!m_Ban7TargetFilename.empty()) && (!m_Ban7IndexFilename.empty()) )
      {
         res = true;

      } else
      {
         if ( (m_IPDiskIndexFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ipdisk_index_filename not found.");
         else if ( (m_IPDiskBasaFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ipdisk_basa_filename not found.");
         else if ( (m_IPDiskTargetFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ipdisk_target_filename not found.");
         else if ( (m_BanTargetFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ban_target_filename not found.");
         else if ( (m_BanIndexFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ban_index_filename not found.");
         else if ( (m_Ban7TargetFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ban7_target_filename not found.");
         else if ( (m_Ban7IndexFilename.empty()) && (m_log != NULL) )
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: ban7_index_filename not found.");
         m_log->FFlush();

      }

   } else
   {
      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP: config not found.");
         m_log->FFlush();
      }
   }

   return res;
}

void TDiskDumpConverter::Lock()
{
   m_Mutex.Acquire();
}

void TDiskDumpConverter::UnLock()
{
   m_Mutex.Release();
}

bool TDiskDumpConverter::FileExists(const TString &filename)
{
   bool res = false;

   if (!filename.empty())
   {
      FILE *handle = fopen(filename.c_str(), "rb");
      if (handle != NULL)
      {
         res = true;

         fclose(handle);
         handle = NULL;
      }
   }

   return res;
}

bool TDiskDumpConverter::ReadBlock(ui32 ip, TIpDisk &value, int &err, bool &exists)
{
   bool     res         = false;
   TString   errtext     = "";

   err    = 0;
   exists = false;

   Lock();

   memset(&value, 0, sizeof(value));
   if ( (!m_IPDiskIndexFilename.empty()) && (!m_IPDiskBasaFilename.empty()) )
   {
      if (!m_open_indexfile)
      {
    idxhandle = new TFile(m_IPDiskIndexFilename, OpenExisting);
    if (idxhandle != NULL)
             m_open_indexfile = true;
      }
      if (!m_open_basafile)
      {
    bashandle = new TFile(m_IPDiskBasaFilename, OpenExisting);
          if (bashandle != NULL)
             m_open_basafile = true;
      }
      if (m_open_indexfile && m_open_basafile)
      {
         i64     pos = 0, pos2 = 0, pos3 = 0, pos4 = 0;
         i32     count = 0;
         ui32    rec_pos = 0;
         ui32    n = 0;

         pos = (i64)sizeof(TFileBaseInfo) + (i64)ip * (i64)sizeof(ui32);
         pos2 = idxhandle->Seek(pos, sSet);
         if ( (pos2 > -1) && (pos2 == pos) )
         {
            count = idxhandle->Read(&rec_pos, sizeof(rec_pos));
            if (count == sizeof(rec_pos))
            {
    if (rec_pos > 0)  //exists record
    {
     i64 fsize = 0;
     i64 fdiff = 0;

     pos2 = sizeof(TFileBaseInfo) + rec_pos * sizeof(TIpDisk);
     fsize = bashandle->Seek(0, sEnd);
     if (fsize > -1)
     {
      fdiff = fsize - sizeof(TIpDisk);
      if (pos2 > fdiff)
      {
       err = 9;

      }
      else
      {
       pos3 = bashandle->Seek(pos2, sSet);
       if ( (pos3 > -1) && (pos3 == pos2) )
       {
        count = bashandle->Read(&value, sizeof(TIpDisk));
        if (count == sizeof(TIpDisk))
        {
         if (value.GetCRC32(0) != value.crc32)
         {
          err = 8;

         }
         else
         {
          exists = true;
         }

        }
        else
         err = 7;
       }
       else
        err = 6;
      }
     }
               } else if (rec_pos == 0)
               {
                  err = 0; //   ip  

               } else
                  err = 5;

            } else
               err = 4;
         } else
            err = 3;
      } else
         err = 2;
   } else
      err = 1;

   UnLock();

   if (err == 0)
      res = true;

   return res;
}

bool TDiskDumpConverter::ConvertDataIPDisk(const TString &ident)
{
   bool              res            = false;

   if (!m_IPDiskIndexFilename.empty() && !m_IPDiskBasaFilename.empty())
   {
      if ( FileExists(m_IPDiskIndexFilename) && FileExists(m_IPDiskBasaFilename) )
      {
         const int         err_array_size = 10;
         TIpDisk           value;
         int               err            = 0;
         ui64              err_array[err_array_size];
         ui64              err_array_last[err_array_size];
         ui64              err_array_curr[err_array_size];
         nosql::HashMap    hash;
         TIpDisk           data_null      = TIpDisk();
         TFindToFileReport counters;
         TFindToFileReport counters_last;
         TFindToFileReport counters_curr;
         TString            ids            = "";
         TString            ips            = "";
         TString            ips2           = "";
         FILE              *handle        = NULL;
         TString            tstr           = "";
         int               cnt            = 0;
         ui32              fulltick       = 0;
         ui32              steptick       = 0;
         ui64              fullcount      = 0;
         TKIPv6            ipk;
         bool              exists         = false;
         ui64              exists_count      = 0;
         ui64              exists_count_last = 0;
         ui64              exists_count_curr = 0;

         memset(err_array, 0, sizeof(err_array));
         memset(err_array_last, 0, sizeof(err_array_last));
         memset(err_array_curr, 0, sizeof(err_array_curr));
         counters.Clear();
         counters_last.Clear();
         counters_curr.Clear();
         fulltick = CShingleTime::GetMs();
         if (!m_IPDiskTargetFilename.empty())
         {
            handle = fopen(m_IPDiskTargetFilename.c_str(), "wb");
            if (handle != NULL)
            {
               steptick = CShingleTime::GetMs();
               memset(err_array, 0, sizeof(err_array));
               for (ui32 ip = 0; ip < 0xFFFFFFFF; ip++)
               {
                  fullcount++;

                  if (ip > 0)
                  {
                     ipk = TKIPv6(IntToIp(ip).c_str());
                     ips2 = ipk.toStroka();

                  } else
                  {
                     ips2 = "0.0.0.0";
                  }

                  if (ReadBlock(ip, value, err, exists))
                  {
                     if ( (exists) && (value.tFirstDeal > 0) && (value.tLastMes > 0) )
                     {
                        exists_count = IncMax64(exists_count, 1);

                        hash.clear();
                        hash["ip"] = ips2;
                        hash["_id"] = TMemRecordSerialization::ConvertToId(ipk);

                        TIpDiskSerialization::SetSerializeWOCheck(value, hash);
                        tstr = TStorageNoSql::GetDataToLog(hash, counters, ids, ips);
                        if (!tstr.empty())
                        {
                           cnt = fprintf(handle, "%s\n", tstr.c_str());
                           if (cnt < 0)
                           {
                              if (m_log != NULL)
                              {
                                 m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): Error write data by ip=%s", ident.c_str(), ips2.c_str());
                                 m_log->FFlush();
                              }
                              printf("CONVDMP(%s): Error write data by ip=%s\n", ident.c_str(), ips2.c_str());
                           }
                        }
                     }

                  } else
                  {

                  }
                  if ( (err >= 0) && (err < err_array_size) )
                     err_array[err] = IncMax64(err_array[err], 1);
                  if ((ip % PRINT_STAT_STEP) == 0)
                  {
                     for (int i = 0; i < err_array_size; i++)
                     {
                        err_array_curr[i] = err_array[i] - err_array_last[i];
                        err_array_last[i] = err_array[i];
                     }

                     counters_curr = counters - counters_last;
                     counters_last = counters;

                     exists_count_curr = exists_count - exists_count_last;
                     exists_count_last = exists_count;

                     steptick = CShingleTime::GetMs() - steptick;
                     if (m_log != NULL)
                     {
                        m_log->WriteMessageAndDataStatus(KMESSAGE, "CONVDMP(%s): read %u ip to %u msec, exists=%llu, err=%llu,%llu-%llu-%llu-%llu-%llu-%llu-%llu,%llu-%llu, convtype=%s", ident.c_str(), ip + 1, steptick, exists_count_curr,
                                                                                         err_array_curr[0], err_array_curr[1], err_array_curr[2], err_array_curr[3], err_array_curr[4], err_array_curr[5], err_array_curr[6], err_array_curr[7], err_array_curr[8], err_array_curr[9],
                                                                                         counters_curr.GetCountersByType().c_str());
                        m_log->FFlush();
                     }
                     printf("CONVDMP(%s): read %u ip to %u msec, exists=%llu, err=%llu,%llu-%llu-%llu-%llu-%llu-%llu-%llu,%llu-%llu, convtype=%s\n", ident.c_str(), ip + 1, steptick, exists_count_curr,
                                                                                         err_array_curr[0], err_array_curr[1], err_array_curr[2], err_array_curr[3], err_array_curr[4], err_array_curr[5], err_array_curr[6], err_array_curr[7], err_array_curr[8], err_array_curr[9],
                                                                                         counters_curr.GetCountersByType().c_str());
                     steptick = CShingleTime::GetMs();
                  }

               }

               fclose(handle);
               handle = NULL;

               fulltick = CShingleTime::GetMs() - fulltick;
               if (m_log != NULL)
               {
                  m_log->WriteMessageAndDataStatus(KMESSAGE, "CONVDMP(%s): all complete, read %llu ip to %u msec, exists=%llu, err=%llu,%llu-%llu-%llu-%llu-%llu-%llu-%llu,%llu-%llu, convtype=%s", ident.c_str(), fullcount, fulltick, exists_count,
                                                                                            err_array[0], err_array[1], err_array[2], err_array[3], err_array[4], err_array[5], err_array[6], err_array[7], err_array[8], err_array[9],
                                                                                            counters.GetCountersByType().c_str());
                  m_log->FFlush();
               }
               printf("CONVDMP(%s): all complete, read %llu ip to %u msec, exists=%llu, err=%llu,%llu-%llu-%llu-%llu-%llu-%llu-%llu,%llu-%llu, convtype=%s\n", ident.c_str(), fullcount, fulltick, exists_count,
                                                                                            err_array[0], err_array[1], err_array[2], err_array[3], err_array[4], err_array[5], err_array[6], err_array[7], err_array[8], err_array[9],
                                                                                            counters.GetCountersByType().c_str());
               res = true;

            } else
            {
               if (m_log != NULL)
               {
                  m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): target file not opened (%s).", ident.c_str(), m_IPDiskTargetFilename.c_str());
                  m_log->FFlush();
               }
               printf("CONVDMP(%s): target file not opened (%s).\n", ident.c_str(), m_IPDiskTargetFilename.c_str());

            }

         } else
         {
            if (m_log != NULL)
            {
               m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): target file not defined.", ident.c_str());
               m_log->FFlush();
            }
            printf("CONVDMP(%s): target file not defined.\n",  ident.c_str());

         }
      } else
      {
         if (m_log != NULL)
         {
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): basa ('%s') or index ('%s') file not exists.", ident.c_str(), m_IPDiskBasaFilename.c_str(), m_IPDiskIndexFilename.c_str());
            m_log->FFlush();
         }
         printf("CONVDMP(%s): basa ('%s') or index ('%s') file not exists.", ident.c_str(), m_IPDiskBasaFilename.c_str(), m_IPDiskIndexFilename.c_str());
      }

   } else
   {
      if (m_log != NULL)
      {
         m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): basa ('%s') or index ('%s') file not defined.", ident.c_str(), m_IPDiskBasaFilename.c_str(), m_IPDiskIndexFilename.c_str());
         m_log->FFlush();
      }
      printf("CONVDMP(%s): basa ('%s') or index ('%s') file not defined.", ident.c_str(), m_IPDiskBasaFilename.c_str(), m_IPDiskIndexFilename.c_str());
   }

   return res;
}

bool TDiskDumpConverter::ConvertBanData(const TString &ident, const TString &index_filename, const TString &target_filename)
{
   bool              res            = false;
   FILE              *source        = NULL;
   FILE              *target        = NULL;
   size_t            count          = 0;
   TBanIpBasicOld    banip;
   int               err            = 0;
   ui32              recordcount    = 0;
   ui32              badcrccount    = 0;
   ui32              undefipcount   = 0;
   ui32              ok_count       = 0;
   char              sip[32];
   ui32              cs             = 0;
   TString            text           = "";
   TBanIpBasic       newdata;
   TFindToFileReport counters;
   TString            id_s           = "";
   TString            ip_s           = "";
   nosql::HashMap    incrs;
   nosql::HashMap    sets;
   TString            data_text      = "";
   ui32              tick           = 0;
   ui32              old_struct_size = sizeof(TBanIpBasicOld);
   TString            rec_id         = "";

   tick = CShingleTime::GetMs();
   if (!index_filename.empty())
   {
      source = fopen(index_filename.c_str(), "rb");
      if (source != NULL)
      {
         target = fopen(target_filename.c_str(), "wb");
         if (target != NULL)
         {
            while (true)
            {
               count = fread(&banip, sizeof(banip), 1, source);
               if (count == 1)
               {
                  recordcount = IncMax32(recordcount, 1);

                  cs = banip.GetCRC32(0);
                  if (cs != banip.cs)
            {
                     badcrccount = IncMax32(badcrccount, 1);
                     continue;
            }

                  memset(sip, 0, sizeof(sip));
                  IntToIp(banip.ip, sip);

                  newdata.Clear();
                  newdata.ip = TKIPv6(sip);
                  if (!newdata.ip.Undefined())
                  {
                     newdata.tAction     = banip.tAction;
                     newdata.chours      = banip.chours;
                     newdata.cMessage    = banip.cMessage;
                     newdata.cIntBanIp   = banip.cIntBanIp;
                   memcpy(newdata.cUser, banip.cUser, sizeof(newdata.cUser));
                     newdata.sendtorbl   = 1;
                     memset(newdata.cReserv, 0, sizeof(newdata.cReserv));
                     newdata.cs          = banip.cs;

                     incrs.clear();
                     sets.clear();

                     TBanIpBasicSerialization::SerializeWOCheck(newdata, /*incrs, */sets);
                     rec_id = TBanIpBasicSerialization::ConvertToId(newdata.ip);
                     sets["_id"] = rec_id;

                     data_text = TStorageNoSql::GetDataToLog(sets, counters, id_s, ip_s);
                     if (!data_text.empty())
                     {
                        fprintf(target, "%s\n", data_text.c_str());
                        ok_count = IncMax32(ok_count, 1);
                        res = true;
                     }

                  } else
                  {
                     undefipcount = IncMax32(undefipcount, 1);
                  }


               } else
               {
                  if (!feof(source))
                     err = 1;
                  break;
               }
            }

            fclose(target);
            target = NULL;

            tick = CShingleTime::GetMs() - tick;
            if (m_log != NULL)
            {
               m_log->WriteMessageAndDataStatus(KMESSAGE, "CONVDMP(%s): convert complete to %u msec (read=%u, ok=%u, undefined_ip=%u, bad_crc=%u), old_struct_size=%u", ident.c_str(), tick, recordcount, ok_count, undefipcount, badcrccount, old_struct_size);
               m_log->FFlush();
            }
            printf("CONVDMP(%s): convert complete to %u msec (read=%u, ok=%u, undefined_ip=%u, bad_crc=%u), old_struct_size=%u", ident.c_str(), tick, recordcount, ok_count, undefipcount, badcrccount, old_struct_size);

         } else
         {
            if (m_log != NULL)
            {
               m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): can't open file %s.", ident.c_str(),  target_filename.c_str());
               m_log->FFlush();
            }
            printf("CONVDMP(%s): can't open file %s.\n", ident.c_str(), target_filename.c_str());

         }

         fclose(source);
         source = NULL;

      } else
      {
         if (m_log != NULL)
         {
            m_log->WriteMessageAndDataStatus(KERROR, "CONVDMP(%s): can't open file %s.", ident.c_str(), index_filename.c_str());
            m_log->FFlush();
         }
         printf("CONVDMP(%s): can't open file %s.\n", ident.c_str(), index_filename.c_str());

      }

   }


   return res;
}

bool TDiskDumpConverter::ConvertData()
{
   bool res = true;

   if (!ConvertBanData("BAN", m_BanIndexFilename, m_BanTargetFilename))
      res = false;

   if (!ConvertBanData("BAN7", m_Ban7IndexFilename, m_Ban7TargetFilename))
      res = false;

   if (!ConvertDataIPDisk("IPDISK"))
      res = false;

   return res;
}

//**************************************************************************************************************************
