#include "tnetlist_ipv6.h"

//*********************************************************************************************************************
//                                                 THashIP
//*********************************************************************************************************************

TListIPv6Base::TListIPv6Base() {
    m_filename = "";
    m_log = nullptr;
    m_ident = "";
    m_clear_before_load = true;
}

TListIPv6Base::TListIPv6Base(const TListIPv6Base& value) {
    m_filename = value.m_filename;
    m_log = value.m_log;
    m_ident = value.m_ident;
    m_clear_before_load = value.m_clear_before_load;
    m_statHashIp = value.m_statHashIp;
}

TListIPv6Base::~TListIPv6Base() {
}

void TListIPv6Base::Lock() {
    m_Mutex.Acquire();
}

void TListIPv6Base::UnLock() {
    m_Mutex.Release();
}

void TListIPv6Base::LockLoad() {
    m_MutexLoad.Acquire();
}

void TListIPv6Base::UnLockLoad() {
    m_MutexLoad.Release();
}

void TListIPv6Base::InitBase(const TString& ident, const TString& filename, TBaseLogClass* logA, bool clear_before_load) {
    char buff[25];

    memset(buff, 0, sizeof(buff));
    snprintf(buff, sizeof(buff), "% 15s", ident.c_str());
    m_ident = TString(buff);

    m_filename = filename;
    m_log = logA;
    m_clear_before_load = clear_before_load;
}

bool TListIPv6Base::FileStateWasChangedK(const char* fn, TFileState* poldbufstat) {
    struct stat bufstat;

    if (fn == 0 || *fn == 0) {
        return false;
    }

    if (stat(fn, &bufstat)) {
        return false;
    }

    if (poldbufstat->mtime == bufstat.st_mtime && poldbufstat->size == bufstat.st_size)
        return false;

    poldbufstat->mtime = bufstat.st_mtime;
    poldbufstat->size = bufstat.st_size;

    return true;
}

void TListIPv6Base::ReloadFileList() {
    if (!m_filename.empty()) {
        if (ExistsFile()) {
            if (!FileStateWasChangedK(m_filename.c_str(), &m_statHashIp)) {
                if (m_log != nullptr)
                    m_log->WriteMessageAndDataStatus(KMESSAGE, "%s:    no changed.", m_ident.c_str());
                return;
            }

            FILE* handle = nullptr;
            int file_size = 0;
            char* buff = nullptr;
            int buff_size = 0;
            int read_count = 0;
            char tbuff[1024];
            int len = 0;
            TString restxt = "";
            THashStatNL stat;

            handle = fopen(m_filename.c_str(), "rb");
            if (handle != nullptr) {
                LockLoad();

                ui32 timepoint = CShingleTime::GetMs();

                PrefixFunction(); //�������������� ��������� �����

                while (fgets(tbuff, sizeof(tbuff), handle)) {
                    len = strlen(tbuff);
                    if (len > 0)
                        ReloadMemListA(tbuff, len, stat);
                }

                PostfixFunction(); //��������� ��������� ����� � ����������
                TriggerFunction();

                UnLockLoad();

                fclose(handle);

                timepoint = CShingleTime::GetMs() - timepoint;
                restxt = GetResultPrint(stat, timepoint);
                if (m_log != nullptr)
                    m_log->WriteMessageAndDataStatus(KMESSAGE, "%s", restxt.c_str());

            } else {
                if (m_log != nullptr)
                    m_log->WriteMessageAndDataStatus(KMESSAGE, "%s:   error open file to read: '%s'.", m_ident.c_str(), m_filename.c_str());
            }
        } else {
            if (m_log != nullptr)
                m_log->WriteMessageAndDataStatus(KMESSAGE, "%s:   file not found: '%s'.", m_ident.c_str(), m_filename.c_str());
        }
    } else {
        if (m_log != nullptr)
            m_log->WriteMessageAndDataStatus(KMESSAGE, "%s:   no defined filename.", m_ident.c_str());
    }
}

ui32 TListIPv6Base::GetStr(const char* source, ui32 source_size, char* destination, ui32 destination_size) {
    long pos = -1;
    ui32 res = 0;
    ui8 dob = 0;

    for (ui32 i = 0; i < source_size; i++) {
        if ((source[i] == 0x0D) && ((i + 1) < source_size) && (source[i + 1] == 0x0A)) {
            pos = i;
            dob = 1;
            break;
        }
        if (source[i] == 0x0A) {
            pos = i;
            dob = 0;
            break;
        }
    }
    res = pos + 1 + dob;
    if (res == 0)
        res = source_size;
    if (res > 0) {
        memset(destination, 0, destination_size);
        if (res <= (destination_size - 1)) {
            memcpy(destination, source, res);
        } else {
            memcpy(destination, source, destination_size - 1);
        }
    }
    return res;
}

void TListIPv6Base::ReloadMemListA(const char* buff, int buff_size, THashStatNL& stat) {
    ui32 m_buff_pos = 0;
    char tbuff[65500];
    const char* sourcebuff = nullptr;
    int sourcebuffsize = 0;
    ui32 strlength = 0;
    TString restxt = "";
    bool valid_str = false;

    sourcebuff = buff;
    sourcebuffsize = buff_size;
    m_buff_pos = 0;
    while (m_buff_pos < sourcebuffsize) {
        strlength = GetStr(sourcebuff + m_buff_pos, sourcebuffsize - m_buff_pos, tbuff, sizeof(tbuff));
        if (strlength > 0) {
            m_buff_pos += strlength;

            valid_str = false;
            for (int i = 0; i < strlength; i++) {
                if ((tbuff[i] < 0) || (tbuff[i] > ' ')) {
                    valid_str = true;
                    break;
                }
            }

            if (valid_str)
                ParseItem(tbuff, stat);

        } else
            break;
    }
}

void TListIPv6Base::ReloadMemList(const TString& data) {
    THashStatNL stat;
    TString restxt = "";

    LockLoad();

    ui32 timepoint = CShingleTime::GetMs();
    PrefixFunction(); //�������������� ��������� �����

    ReloadMemListA(data.c_str(), data.length(), stat);

    PostfixFunction(); //��������� ��������� ����� � ����������
    TriggerFunction();

    UnLockLoad();

    timepoint = CShingleTime::GetMs() - timepoint;
    restxt = GetResultPrint(stat, timepoint);
    if (m_log != nullptr)
        m_log->WriteMessageAndDataStatus(KMESSAGE, "%s", restxt.c_str());
}

bool TListIPv6Base::ExistsFile() {
    bool res = false;
    FILE* handle = nullptr;

    if (!m_filename.empty()) {
        handle = fopen(m_filename.c_str(), "rb");
        if (handle != nullptr) {
            res = true;
            fclose(handle);
        }
    }

    return res;
}

ui64 TListIPv6Base::CalcShingle(const TString& str_data) {
    ui64 res = 0;
    TString str_data_t = "";

    if (!str_data.empty()) {
        str_data_t = KTrim(str_data);
        if (!str_data_t.empty()) {
            char sshingle[32];
            TString stemp = "";

            stemp = "sh_" + str_data_t;
            ToLower(stemp.begin(), stemp.size(), *CodePageByCharset(CODES_WIN));
            memset(sshingle, 0, sizeof(sshingle));
            calc_strcrc64(stemp.c_str(), stemp.size(), sshingle);
            sscanf(sshingle, "%" PRIx64, &res);
        }
    }

    return res;
}

TString TListIPv6Base::IntToStroka(int value) {
    TString res = "";
    char BUFF[25];

    memset(BUFF, 0, sizeof(BUFF));
    snprintf(BUFF, sizeof(BUFF) - 1, "%u", value);
    res.assign(BUFF);

    return res;
}

TString TListIPv6Base::KTrim(const TString& s) {
    TString res = "", stemp = s;
    int nb = 0, ne = 0, nbc = 0, nec = 0;
    TString::iterator it;
    int count = 0;
    unsigned char uc = 0;

    it = stemp.begin();
    while (it != stemp.end()) {
        nb++;
        memcpy(&uc, it, sizeof(uc));
        if (uc > ' ')
            break;
        ++it;
    }

    stemp.reverse();
    it = stemp.begin();
    while (it != stemp.end()) {
        ne++;
        memcpy(&uc, it, sizeof(uc));
        if (uc > ' ')
            break;
        ++it;
    }

    nbc = nb > 0 ? nb - 1 : 0;
    nec = ne > 0 ? ne - 1 : 0;
    count = (int)s.size() - nbc - nec;
    if (count > 0)
        res = s.substr(nbc, count);

    return res;
}

ui32 TListIPv6Base::IncMax32(const ui32 value, const ui32 step) {
    ui32 res = 0;
    ui32 count = 0;
    ui32 stepn = 0;

    switch (step) {
        case 0:
            res = value;
            break;
        case 1:
            res = (value < 0xFFFFFFFF) ? (value + 1) : 0xFFFFFFFF;
            break;
        default:
            count = 0xFFFFFFFF - value;
            if (count > 0) {
                stepn = (step > count) ? count : step;
                res = value + stepn;
            } else
                res = value;
    };

    return res;
}

TString TListIPv6Base::IntToIpV4(ui32 ip) {
    TString res = "";
    char buff[64];
    ui32 oct[4];

    memset(buff, 0, sizeof(buff));
    oct[0] = (ip >> 24) & 0xFF;
    oct[1] = (ip >> 16) & 0xFF;
    oct[2] = (ip >> 8) & 0xFF;
    oct[3] = ip & 0xFF;
    snprintf(buff, sizeof(buff) - 1, "%u.%u.%u.%u", oct[0], oct[1], oct[2], oct[3]);
    res = TString(buff);

    return res;
}

char* TListIPv6Base::STRNCPY(char* strDest, const char* strSource, size_t DestSize, size_t count) {
    if (count > DestSize - 1)
        count = DestSize - 1;
    strncpy(strDest, strSource, count);
    strDest[count] = 0;
    return strDest;
}

bool TListIPv6Base::IpToIntV4(const char* pstr, int len, ui32* pnet, ui32* pip) {
    const char* pdot;
    int i;
    int c = pip ? 3 : 2;
    ui32 t[4];
    const int buf_sz = 64;
    char buf[buf_sz];

    for (i = 0; i < len; i++) {
        if (!isdigit(pstr[i]) && pstr[i] != '.')
            return false;
    }

    for (i = 0; i < c; i++) {
        pdot = strchr(pstr, '.');
        if (!pdot)
            return false;
        STRNCPY(buf, pstr, buf_sz, pdot - pstr);
        t[i] = atoi(buf);
        pstr = pdot + 1;
        if (t[i] > 255)
            return false;
    }

    if (!isdigit(*pstr))
        return false;
    t[c] = atoi(pstr);
    if (t[c] > 255)
        return false;

    *pnet = t[0] * 256 * 256 * 256;
    *pnet += t[1] * 256 * 256;
    *pnet += t[2] * 256;

    if (pip) {
        *pip = *pnet;
        *pip += t[3];
    }

    return true;
}

//*********************************************************************************************************************
//                                                 TWhiteNetv6
//*********************************************************************************************************************

TNetListIPv6::TNetListIPv6() {
    idents = "???";
    host_data = nullptr;
    host_data_temp = nullptr;
    range_data = nullptr;
    range_data_temp = nullptr;
}

TNetListIPv6::TNetListIPv6(const TString& ident) {
    idents = ident;
    host_data = nullptr;
    host_data_temp = nullptr;
    range_data = nullptr;
    range_data_temp = nullptr;
}

TNetListIPv6::~TNetListIPv6() {
    ClearData();
}

void TNetListIPv6::Init(const TString& filename, TBaseLogClass* logA) {
    InitTable();
    InitBase(idents, filename, logA, true);
}

void TNetListIPv6::ParseItem(const char* str, THashStatNL& stat) {
    if (str != nullptr) {
        TString strdata = NormalizeStr(str, strlen(str));
        TRecordType rectype = TNetListIPv6::TUNDEF;
        TWIPv6 wip;
        ui64 shingle = 0;
        TStrokaHashIt it;

        if ((!strdata.empty()) && (host_data_temp != nullptr) && (range_data_temp != nullptr)) {
            if (strdata[0] == '#') {
            } else {
                rectype = GetRecordType(strdata.c_str(), strdata.length());
                switch (rectype) {
                    case TNetListIPv6::THOST:
                        shingle = CalcShingle(strdata);
                        if (shingle != 0) {
                            it = host_data_temp->find(shingle);
                            if (it != host_data_temp->end()) {
                                stat.dublicat_count = IncMax32(stat.dublicat_count, 1);
#ifndef K_PRINT_ONLY_STAT
                                if ((LogsGroup != nullptr) && (LogsGroup->ActionLog() != nullptr))
                                    LogsGroup->ActionLog()->WriteMessageAndDataStatus(KWARNING, "%s:   doubling '%s'", m_ident.c_str(), strdata.c_str());
#endif
                            } else {
                                (*host_data_temp)[shingle] = strdata;
                                stat.trace_count = IncMax32(stat.trace_count, 1);
                            }
                        } else {
                            stat.bad_count = IncMax32(stat.bad_count, 1);
#ifndef K_PRINT_ONLY_STAT
                            if ((LogsGroup != nullptr) && (LogsGroup->ActionLog() != nullptr))
                                LogsGroup->ActionLog()->WriteMessageAndDataStatus(KERROR, "%s:   bad item '%s'", m_ident.c_str(), strdata.c_str());
#endif
                        }
                        break;
                    case TNetListIPv6::TIPNET: //range 1 (127.0.0.1/32)
                        if (ParseRangeType1(strdata, wip)) {
                            range_data_temp->push_back(wip);
                            stat.no_trace_count = IncMax32(stat.no_trace_count, 1);
                        } else {
                            stat.bad_count = IncMax32(stat.bad_count, 1);
                        }
                        break;
                    case TNetListIPv6::TIPRANGE: //range 2 (127.0.0.1 127.0.0.2)
                        if (ParseRangeType2(strdata, wip)) {
                            range_data_temp->push_back(wip);
                            stat.no_trace_count = IncMax32(stat.no_trace_count, 1);
                        } else {
                            stat.bad_count = IncMax32(stat.bad_count, 1);
                        }
                        break;
                };
            }
        }
    }
}

void TNetListIPv6::ClearData() {
    if (host_data != nullptr) {
        delete host_data;
        host_data = nullptr;
    }
    if (host_data_temp != nullptr) {
        delete host_data_temp;
        host_data_temp = nullptr;
    }
    if (range_data != nullptr) {
        delete range_data;
        range_data = nullptr;
    }
    if (range_data_temp != nullptr) {
        delete range_data_temp;
        range_data_temp = nullptr;
    }
}

TString TNetListIPv6::GetResultPrint(THashStatNL& stat, ui32 timedelay) {
    TString res = "";
    ui32 all = 0;

    all = IncMax32(all, stat.no_trace_count);
    all = IncMax32(all, stat.trace_count);
    all = IncMax32(all, stat.bad_count);
    all = IncMax32(all, stat.dublicat_count);
    res = m_ident + ":   Read list (" + IntToStroka(timedelay) + " ms), records: " + IntToStroka(all) + " (url=" + IntToStroka(stat.trace_count) + ", dublicat url=" + IntToStroka(stat.dublicat_count) + ", range=" + IntToStroka(stat.no_trace_count) + ", bad=" + IntToStroka(stat.bad_count) + ")";

    return res;
}

void TNetListIPv6::TriggerFunction() {
}

void TNetListIPv6::PrefixFunction() {
    host_data = new TStrokaHash();
    host_data_temp = new TStrokaHash();
    range_data = new TWIPv6List();
    range_data_temp = new TWIPv6List();
}

void TNetListIPv6::PostfixFunction() {
    TStrokaHash* host_data_t = nullptr;
    TWIPv6List* range_data_t = nullptr;

    //��������� ������� ����� �� ���������, ���� ��������� ������
    Lock();

    host_data_t = host_data;
    host_data = host_data_temp;
    host_data_temp = nullptr;

    range_data_t = range_data;
    range_data = range_data_temp;
    range_data_temp = nullptr;

    UnLock();

    //������� ������ ������
    if (host_data_t != nullptr) {
        delete host_data_t;
        host_data_t = nullptr;
    }
    if (range_data_t != nullptr) {
        delete range_data_t;
        range_data_t = nullptr;
    }
}

TString TNetListIPv6::NormalizeStr(const char* str, int strlength) {
    TString res = "";
    char symb = 0;
    char symb2 = 0;
    TString stemp = "";
    TString stemp2 = "";
    bool first_symb = true;
    bool next_slash = false;
    char last_symb = 0x00;
    const char* p = nullptr;
    int count = 0;

    stemp = TString(str, strlength);
    if (!stemp.empty()) {
        p = strchr(stemp.c_str(), '#');
        if (p != nullptr) {
            count = p - stemp.c_str();
            if (count > 0)
                stemp = TString(stemp.c_str(), count);
        }
        for (int i = 0; i < stemp.length(); i++) {
            symb = *(stemp.c_str() + i);
            if (symb <= ' ') {
                if (first_symb) {
                    continue;
                } else {
                    int j = 0;
                    next_slash = false;
                    for (j = (i + 1); j < stemp.length(); j++) {
                        symb2 = *(stemp.c_str() + j);
                        if (symb2 <= ' ') {
                            continue;
                        } else if ((symb2 == '\\') || (symb2 == '/')) {
                            next_slash = true;
                            break;
                        } else
                            break;
                    }
                    if ((!next_slash) && (last_symb != '/') && (last_symb != '\\') && (j < stemp.length()))
                        stemp2 = stemp2 + ' ';
                    i = j - 1;
                }
            } else {
                if (first_symb)
                    first_symb = false;
                stemp2 = stemp2 + symb;
                last_symb = symb;
            }
        }
        res = stemp2;
    }

    return res;
}

TNetListIPv6::TRecordType TNetListIPv6::GetRecordType(const char* str, int strlength) {
    TRecordType res = TUNDEF;
    char symb = 0;
    bool not_range1 = false;
    bool not_range2 = false;
    const char* pgood = nullptr;
    const char* pbad = nullptr;

    if ((str != nullptr) && (strlength > 0)) {
        for (int i = 0; i < strlength; i++) {
            symb = *(str + i);
            if (!IsAllowSymbolRange1(symb))
                not_range1 = true;
            if (!IsAllowSymbolRange2(symb))
                not_range2 = true;
            if (not_range1 && not_range2) {
                res = THOST;
                break;
            }
        }
    }
    if (!not_range1) //range 1 (127.0.0.1/32)
    {
        pgood = strchr(str, '/');
        if (pgood == nullptr)
            pgood = strchr(str, '\\');
        pbad = strchr(str, ' ');
        if ((pgood != nullptr) && (pbad == nullptr))
            res = TIPNET;
    }
    if ((res != TIPNET) && (!not_range2)) //range 2 (127.0.0.1 127.0.0.2)
    {
        pgood = strchr(str, ' ');
        pbad = strchr(str, '/');
        if (pbad == nullptr)
            pbad = strchr(str, '\\');
        if ((pgood != nullptr) && (pbad == nullptr))
            res = TIPRANGE;
    }
    if (res == TUNDEF) {
        TKIPv6 addr = TKIPv6(str);
        if (!addr.Undefined())
            res = TIPADDRESS;
    }

    return res;
}

void TNetListIPv6::InitTable() {
    memset(SymbTable, 0, sizeof(SymbTable));

    //range 1 (127.0.0.1/32)
    SymbTable['.'] = (SymbTable['.'] & 0xFE) + 0x01;
    SymbTable[':'] = (SymbTable[':'] & 0xFE) + 0x01;
    SymbTable['/'] = (SymbTable['/'] & 0xFE) + 0x01;
    SymbTable['\\'] = (SymbTable['\\'] & 0xFE) + 0x01;
    for (int i = 'a'; i <= 'f'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFE) + 0x01;
    for (int i = 'A'; i <= 'F'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFE) + 0x01;
    for (int i = '0'; i <= '9'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFE) + 0x01;

    //range 2
    SymbTable['.'] = (SymbTable['.'] & 0xFD) + 0x02;
    SymbTable[':'] = (SymbTable[':'] & 0xFD) + 0x02;
    SymbTable[' '] = (SymbTable[' '] & 0xFD) + 0x02;
    for (int i = 'a'; i <= 'f'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFD) + 0x02;
    for (int i = 'A'; i <= 'F'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFD) + 0x02;
    for (int i = '0'; i <= '9'; i++)
        SymbTable[i] = (SymbTable[i] & 0xFD) + 0x02;
}

bool TNetListIPv6::IsAllowSymbolRange1(char symb) {
    bool res = false;
    unsigned char uc = 0;
    ui8 value = 0;

    memcpy(&uc, &symb, sizeof(symb));
    value = SymbTable[uc];
    if (IS_SYMB_RANGE1(value) > 0)
        res = true;

    return res;
}

bool TNetListIPv6::IsAllowSymbolRange2(char symb) {
    bool res = false;
    unsigned char uc = 0;
    ui8 value = 0;

    memcpy(&uc, &symb, sizeof(symb));
    value = SymbTable[uc];
    if (IS_SYMB_RANGE2(value) > 0)
        res = true;

    return res;
}

bool TNetListIPv6::ParseRangeType1(const TString& text, TWIPv6& value) {
    bool res = false;
    const char* p1 = nullptr;
    const char* p2 = nullptr;
    int ipsize = 0;
    ui32 ip = 0;
    ui32 net = 0;
    ui32 ipmaskai2 = 0;
    ui32 firstip = 0;
    ui32 lastip = 0;
    ui32 ipcount = 0;
    int ipmaskai = 0;
    TString ipaddress = "";
    TString ipmaska = "";
    TString firstip_s = "";
    TString lastip_s = "";

    if (!text.empty()) {
        p1 = strstr(text.c_str(), "\\");
        p2 = strstr(text.c_str(), "/");
        if (p1 != nullptr) {
            ipsize = p1 - text.c_str();
            if (ipsize > 0) {
                ipaddress = KTrim(TString(text.c_str(), ipsize));
                ipmaska = KTrim(TString(p1 + 1));
                ipmaskai = atoi(ipmaska.c_str());
            }
        } else if (p2 != nullptr) {
            ipsize = p2 - text.c_str();
            if (ipsize > 0) {
                ipaddress = TString(text.c_str(), ipsize);
                ipmaska = TString(p2 + 1);
                ipmaskai = atoi(ipmaska.c_str());
            }
        }
        if ((!ipaddress.empty()) && (!ipmaska.empty() && (ipmaskai > 0))) {
            if (!IpToIntV4(ipaddress.c_str(), ipaddress.size(), &net, &ip))
                ip = 0;

            if (ip > 0) {
                ipmaskai2 = 0xFFFFFFFF << (32 - ipmaskai);
                ipcount = 1 << (32 - ipmaskai);
                firstip = ip & ipmaskai2;
                lastip = firstip + ipcount - 1;

                firstip_s = IntToIpV4(firstip);
                lastip_s = IntToIpV4(lastip);

                value.ip1 = TKIPv6(firstip_s.c_str());
                value.ip2 = TKIPv6(lastip_s.c_str());
                if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                    res = true;
            }
        }
    }

    return res;
}

bool TNetListIPv6::ParseRangeType2(const TString& text, TWIPv6& value) {
    bool res = false;
    char ipbuff1[255];
    char ipbuff2[255];

    if (!text.empty()) {
        memset(ipbuff1, 0, sizeof(ipbuff1));
        memset(ipbuff2, 0, sizeof(ipbuff2));
        if (sscanf(text.c_str(), " %s %s ", ipbuff1, ipbuff2) == 2) {
            value.ip1 = TKIPv6(ipbuff1);
            value.ip2 = TKIPv6(ipbuff2);
            if ((!value.ip1.Undefined()) && (!value.ip2.Undefined()))
                res = true;
        }
    }

    return res;
}

bool TNetListIPv6::IsNet(const char* phost) {
    bool res = false;
    TStrokaHashIt it;
    ui64 shingle = 0;
    TString st = "";

    if ((phost != nullptr) && (*phost != '-') && (host_data != nullptr)) {
        st = TString(phost);
        shingle = CalcShingle(st);
        if (shingle != 0) {
            Lock();

            it = host_data->find(shingle);
            if (it != host_data->end()) {
                if ((*it).second == st)
                    res = true;
            }

            UnLock();
        }
    }

    return res;
}

bool TNetListIPv6::IsNet(TKIPv6 ip) {
    TKIPv6 min_ip_from_diapason = TKIPv6();

    return IsNet(ip, min_ip_from_diapason);
}

bool TNetListIPv6::IsNet(TKIPv6 ip, TKIPv6& min_ip_from_diapason) {
    bool res = false;
    TWIPv6ListIt it;
    TWIPv6 wip;
    TString s1 = "", s2 = "", s3 = "";

    if ((!ip.Undefined()) && (range_data != nullptr)) {
        Lock();

        it = range_data->begin();
        while (it != range_data->end()) {
            wip = *it;

            s1 = ip.toStroka();
            s2 = wip.ip1.toStroka();
            s3 = wip.ip2.toStroka();

            if ((ip >= wip.ip1) && (ip <= wip.ip2)) {
                min_ip_from_diapason = wip.ip1;
                res = true;
                break;
            }

            ++it;
        }

        UnLock();
    }

    return res;
}

TNetListIPv6::TRecordType TNetListIPv6::GetAddresses(const TString& addr_text, TWIPv6& value) {
    TRecordType res = TUNDEF;
    TWIPv6 wip = TWIPv6();
    TString stemp = NormalizeStr(addr_text.c_str(), addr_text.length());
    TKIPv6 ip = TKIPv6();

    value = TWIPv6();
    res = GetRecordType(stemp.c_str(), stemp.length());
    switch (res) {
        case TNetListIPv6::THOST:
            break;
        case TNetListIPv6::TIPNET: //range 1 (127.0.0.1/32)
            if (ParseRangeType1(stemp, wip))
                value = wip;
            else
                res = TUNDEF;
            break;
        case TNetListIPv6::TIPRANGE: //range 2 (127.0.0.1 127.0.0.2)
            if (ParseRangeType2(stemp, wip))
                value = wip;
            else
                res = TUNDEF;
            break;
        case TNetListIPv6::TIPADDRESS:
            ip = TKIPv6(stemp.c_str());
            if (!ip.Undefined()) {
                value.ip1 = ip;
                value.ip2 = ip;
            } else
                res = TUNDEF;
            break;
    };

    return res;
}

//********************************************************************************************************************
