#pragma once

#include "util/system/mutex.h"
#include "tlogsgroup.h"
#include <mail/so/spamstop/tools/so-common/tlogclass.h>
#include <list>
#include <mail/so/spamstop/tools/so-common/thashipv6.h>
#include "util/system/sem.h"
#include <mail/so/spamstop/tools/so-common/unnamedsem.h>
#include <mail/so/spamstop/sp/rengine.h>

#define KSEMA

//************************************************************************************
//                            TRengineRuleActionWeights
//************************************************************************************

struct TRengineRuleActionWeights
{
   static constexpr float m_IPv4BanRuleWeight = IPv4BanRuleWeight;
   static constexpr float m_IPv4RPTRuleWeight = IPv4RPTRuleWeight;
   static constexpr float m_IPv4FRWDRuleWeight = IPv4FRWDRuleWeight;
   static constexpr float m_IPv4PROBBANRuleWeight = IPv4PROBBANRuleWeight;
   static constexpr float m_IPv4BAN2RuleWeight = IPv4BAN2RuleWeight;
   static constexpr float m_IPv4BAN7RuleWeight = IPv4BAN7RuleWeight;
   static constexpr float m_IPv4BAN8RuleWeight = IPv4BAN8RuleWeight;
   static constexpr float m_IPv6BanRuleWeight = IPv6BanRuleWeight;
   static constexpr float m_IPv6RPTRuleWeight = IPv6RPTRuleWeight;
   static constexpr float m_IPv6FRWDRuleWeight = IPv6FRWDRuleWeight;
   static constexpr float m_IPv6PROBBANRuleWeight = IPv6PROBBANRuleWeight;
   static constexpr float m_IPv6BAN2RuleWeight = IPv6BAN2RuleWeight;
   static constexpr float m_IPv6BAN7RuleWeight = IPv6BAN7RuleWeight;
   static constexpr float m_IPv6BAN8RuleWeight = IPv6BAN8RuleWeight;

   static TString PrintIPv4Rules()
   {
      char buff[256];

      memset(buff, 0, sizeof(buff));
      snprintf(buff, sizeof(buff) - 1, "%0.2f / %0.2f / %0.2f / %0.2f / %0.2f / %0.2f / %0.2f", m_IPv4BanRuleWeight, m_IPv4RPTRuleWeight, m_IPv4FRWDRuleWeight, m_IPv4PROBBANRuleWeight, m_IPv4BAN2RuleWeight, m_IPv4BAN7RuleWeight, m_IPv4BAN8RuleWeight);

      return TString(buff);
   }

   static TString PrintIPv6Rules()
   {
      char buff[256];

      memset(buff, 0, sizeof(buff));
      snprintf(buff, sizeof(buff) - 1, "%0.2f / %0.2f / %0.2f / %0.2f / %0.2f / %0.2f / %0.2f", m_IPv6BanRuleWeight, m_IPv6RPTRuleWeight, m_IPv6FRWDRuleWeight, m_IPv6PROBBANRuleWeight, m_IPv6BAN2RuleWeight, m_IPv6BAN7RuleWeight, m_IPv6BAN8RuleWeight);

      return TString(buff);
   }

};

//************************************************************************************
//                            TRenginePoolStat
//************************************************************************************

struct TRenginePoolStat
{
   ui32                       filter_count;
   ui32                       good_filter_count;
   ui32                       bad_filter_count;
   TString                     rules;
   float                      score;
   TString                     rps_str;
   TString                     get_filter_res;
   TString                     rulescs_str;

   TRenginePoolStat()
   {
      filter_count      = 0;
      good_filter_count = 0;
      bad_filter_count  = 0;
      rules             = "";
      score             = 0;
      rps_str           = "";
      get_filter_res    = "";
      rulescs_str       = "";
   }
};

//************************************************************************************
//                             TRengineElement
//************************************************************************************

class TRengineElement
{
private:
         TSoConfig            m_pSpTop;
         TSpLoggers        SpLoggers;

         THolder<TRengine>         m_hSp;          //rengine object
         int               m_index;
         TMutex            m_Mutex;
         TLogsGroup        *LogsGroup;
         kipv6::TStringsLists     *RulePrintList;
         TString            m_dnRules;
         float             m_Score;
         ui32              m_count;
         time_t            m_lastcalctime;
         float             m_cps;
         TString            m_rules_cs;
         time_t            m_loadrules_time;

         void              CalcCPS();
         void              OnlyCalcCPS();
public:
         TRengineElement();

         bool        Init(int index, TLogsGroup *LogsGroupA, const TString &dnRulesA, float ScoreA, kipv6::TStringsLists *RulePrintListA, TTrueAtomicSharedPtr<TRulesHolder> pRulesHolder);
         ui32        PrintActionLog(TLogStatus status, const char *msg, ...);
         TString     Reset(bool &ok, TTrueAtomicSharedPtr<TRulesHolder> pRulesHolder);
         TRengine&   GetFilterHandle(){ return *m_hSp; }

         void        Lock();
         void        UnLock();
         int         GetIndex(){ return m_index; }
         float       GetCPS();
         TString      GetRulesCS(){ return m_rules_cs; }
         time_t      GetLoadRulesTime(){ return m_loadrules_time; }
         TString      GetLoadRulesElapsed()
         {
            TString res = "???";

            if (m_loadrules_time != 0)
            {
               time_t currtime = time(NULL);
               ui32   diff     = 0;

               if (currtime >= m_loadrules_time)
               {
                  diff = currtime - m_loadrules_time;
                  res = IntToHourMinSec2(diff);
               }
            }

            return res;
         }

         bool        GoodFilter(){ return bool(m_hSp); }
};

//************************************************************************************
//                               TRenginePool
//************************************************************************************

typedef std::list<TRengineElement*>    TRengineElementList;
typedef TRengineElementList::iterator  TRengineElementListIt;

class TRenginePool
{
private:
         TString              m_rules_path;
         TSpLoggers           SpLoggersMain;
    TTrueAtomicSharedPtr<TRulesHolder>         pRulesHolder;  //rules object
         TLogsGroup           *LogsGroup;
         kipv6::TStringsLists *RulePrintList;
         TRengineElementList  m_AllElementList;
         TRengineElementList  m_AccessibleList;
         TMutex               m_Mutex;
         TMutex               m_MutexExit;
         TMutex               m_MutexCalcUsedRengine;
         int                  elementcount;
         bool                 m_exit;
         int                  m_half_elementcount;
         int                  m_quarter_elementcount;
         int                  m_threefourth_elementcount;
         time_t               m_last_calc_usedrengine;
         int                  m_max_usedrengine;
         TString               m_dnRules;
         float                m_Score;
         ui32                 m_good_filter_count;
         ui32                 m_bad_filter_count;
#ifdef KSEMA
         TUnnamedSemaphore    *m_Sema;
#endif
         ui32                 getfilter_count;
         ui64                 getfilter_summa_tick;
         ui32                 getfilter_max_tick;
         TMutex               m_MutexCalcStat;
         TString               get_filter_res;

         void        Lock();
         void        UnLock();
public:
         TRenginePool();
         ~TRenginePool();

         bool              Init(TLogsGroup *LogsGroupA, int elementcountA, const TString &dnRulesA, float ScoreA, kipv6::TStringsLists *RulePrintListA);
         TString            Reset(bool &ok);
         void              ReturnFilter(TRengineElement *elem);
         TRengineElement   *GetFilter();
         void              SetExit();
         ui32              GetAccessibleFilterCount();
         int               CalcUsedRengine();
         int               GetUsedRengine();
         TString            GetCPSList();
         TString            GetRulesCSList();
         TRenginePoolStat  GetStat();
         TString            GetStatForLog();
};

//************************************************************************************
